# Copyright 2023 solo-learn development team.

# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:

# The above copyright notice and this permission notice shall be included in all copies
# or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
# INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
# FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
# OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.

from typing import Any, Dict, List, Sequence

import omegaconf
import torch
import torch.nn as nn
from solo.losses.radialvicreg import (
    anisotropy_loss,
    batch_sparsity_metric,
    chi2_radial_nll_loss,
    covariance_loss,
    embedding_sparsity_metric,
    invariance_loss,
    uniform_loss,
    variance_loss,
)
from solo.losses.vicreg_e2mc import vicreg_e2mc_loss_func, entropy_and_hypercovariance_loss
from solo.methods.base import BaseMethod
from solo.utils.misc import gather, omegaconf_select
from solo.utils.metrics import compute_balanced_accuracy
from solo.losses.radialvicreg import chi2_radial_nll_loss_for_lightning_logging


class VICRegE2MC(BaseMethod):
    def __init__(self, cfg: omegaconf.DictConfig):
        """Implements VICReg with Entropy-Maximization and Hypercovariance minimization (E2MC).

        Extra cfg settings:
            method_kwargs:
                proj_output_dim (int): number of dimensions of the projected features.
                proj_hidden_dim (int): number of neurons in the hidden layers of the projector.
                sim_loss_weight (float): weight of the invariance term.
                var_loss_weight (float): weight of the variance term.
                cov_loss_weight (float): weight of the covariance term.
                ent_loss_weight (float): weight of the entropy term.
                hypercov_loss_weight (float): weight of the hypercovariance term.
                epsilon (float): epsilon for entropy calculation.
        """
        super().__init__(cfg)

        self.sim_loss_weight: float = cfg.method_kwargs.sim_loss_weight
        self.var_loss_weight: float = cfg.method_kwargs.var_loss_weight
        self.cov_loss_weight: float = cfg.method_kwargs.cov_loss_weight
        self.ent_loss_weight: float = cfg.method_kwargs.ent_loss_weight
        self.hypercov_loss_weight: float = cfg.method_kwargs.hypercov_loss_weight
        self.epsilon: float = cfg.method_kwargs.epsilon

        proj_hidden_dim: int = cfg.method_kwargs.proj_hidden_dim
        proj_output_dim: int = cfg.method_kwargs.proj_output_dim

        # projector
        self.projector = nn.Sequential(
            nn.Linear(self.features_dim, proj_hidden_dim),
            nn.BatchNorm1d(proj_hidden_dim),
            nn.ReLU(),
            nn.Linear(proj_hidden_dim, proj_hidden_dim),
            nn.BatchNorm1d(proj_hidden_dim),
            nn.ReLU(),
            nn.Linear(proj_hidden_dim, proj_output_dim),
        )

    @staticmethod
    def add_and_assert_specific_cfg(cfg: omegaconf.DictConfig) -> omegaconf.DictConfig:
        """Adds method specific default values/checks for config."""
        cfg = super(VICRegE2MC, VICRegE2MC).add_and_assert_specific_cfg(cfg)

        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_output_dim")
        assert not omegaconf.OmegaConf.is_missing(cfg, "method_kwargs.proj_hidden_dim")

        cfg.method_kwargs.sim_loss_weight = omegaconf_select(
            cfg, "method_kwargs.sim_loss_weight", 25.0
        )
        cfg.method_kwargs.var_loss_weight = omegaconf_select(
            cfg, "method_kwargs.var_loss_weight", 25.0
        )
        cfg.method_kwargs.cov_loss_weight = omegaconf_select(
            cfg, "method_kwargs.cov_loss_weight", 1.0
        )
        cfg.method_kwargs.ent_loss_weight = omegaconf_select(
            cfg, "method_kwargs.ent_loss_weight", 1000.0
        )
        cfg.method_kwargs.hypercov_loss_weight = omegaconf_select(
            cfg, "method_kwargs.hypercov_loss_weight", 100.0
        )
        cfg.method_kwargs.epsilon = omegaconf_select(cfg, "method_kwargs.epsilon", 1e-7)

        return cfg

    @property
    def learnable_params(self) -> List[dict]:
        """Adds projector parameters to the parent's learnable parameters.

        Returns:
            List[dict]: list of learnable parameters.
        """
        extra_learnable_params = [{"name": "projector", "params": self.projector.parameters()}]
        return super().learnable_params + extra_learnable_params

    def forward(self, X: torch.Tensor) -> Dict[str, Any]:
        """Performs the forward pass of the backbone, projector, and both classifiers (if enabled)."""
        out = super().forward(X)

        if self.training:
            with torch.no_grad():
                # log embedding layer feature norm - this is the encoder
                backbone_feature_norm_mean = torch.norm(out["feats"], dim=1).mean()
                backbone_feature_norm_var = torch.norm(out["feats"], dim=1).var()

                self.log(
                    "encoder_norm/train_backbone_feature_norm_mean",
                    backbone_feature_norm_mean,
                    on_epoch=True,
                    sync_dist=True,
                )
                self.log(
                    "encoder_norm/train_backbone_feature_norm_var",
                    backbone_feature_norm_var,
                    on_epoch=True,
                    sync_dist=True,
                )

                # Logging uniform loss
                backbone_feature_unif_loss = uniform_loss(out["feats"])
                self.log(
                    "unif_loss/train_backbone_feature_unif_loss_embed",
                    backbone_feature_unif_loss,
                    on_epoch=True,
                    sync_dist=True,
                )

                # Logging anisotropy loss for encoder features
                backbone_feature_anisotropy_loss = anisotropy_loss(out["feats"])
                self.log(
                    "anisotropy_loss/train_encoder_feature_anisotropy_loss_embed",
                    backbone_feature_anisotropy_loss,
                    on_epoch=True,
                    sync_dist=True,
                )

        z = self.projector(out["feats"])
        out.update({"z": z})

        # If the projector classifier exists (initialized in BaseMethod), call it.
        if self.projector_classifier is not None:
            projector_logits = self.projector_classifier(z.detach())
            out.update({"projector_logits": projector_logits})

        return out

    def validation_step(self, batch: Sequence[Any], batch_idx: int):
        """Validation step for VICRegE2MC."""
        X, targets = batch
        out = self.base_validation_step(X, targets)

        batch_size = targets.size(0)

        # Custom logging for VICRegE2MC
        if not self.trainer.sanity_checking:
            feats = out["feats"]
            z = self.projector(feats)

            # Log metrics for encoder features
            gathered_feats = gather(feats)
            val_radial_loss_encoder = chi2_radial_nll_loss_for_lightning_logging(gathered_feats)
            self.log(
                "encoder/val_backbone_radial_loss",
                val_radial_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )

            # Log metrics for projector features
            gathered_z = gather(z)
            val_radial_loss_projector = chi2_radial_nll_loss_for_lightning_logging(gathered_z)
            self.log(
                "projector/val_radial_loss",
                val_radial_loss_projector,
                on_epoch=True,
                sync_dist=True,
            )

            val_ent_loss, val_hypercov_loss = entropy_and_hypercovariance_loss(
                gathered_z, gathered_z, self.epsilon
            )
            self.log(
                "projector/val_ent_loss", -val_ent_loss, on_epoch=True, sync_dist=True
            )  # Report as negative entropy
            self.log(
                "projector/val_hypercov_loss", val_hypercov_loss, on_epoch=True, sync_dist=True
            )

            # Log norms for projector features
            z_norm_mean = torch.norm(z, dim=1).mean()
            z_norm_var = torch.norm(z, dim=1).var()
            self.log("projector_norm/val_z_norm_mean", z_norm_mean, on_epoch=True, sync_dist=True)
            self.log("projector_norm/val_z_norm_var", z_norm_var, on_epoch=True, sync_dist=True)

        # CelebA specific logic
        if self.dataset_name == "CelebA":
            out_for_logging = {f"val_{k}": v for k, v in out.items() if "metrics" in k}

            ### Balance_Acc ###
            for outcome_type in self.balanced_acc_quantities_order:
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"][
                    outcome_type
                ] += out[f"encoder_{outcome_type}"]
                if "proj_loss" in out:
                    self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"][
                        outcome_type
                    ] += out[f"proj_{outcome_type}"]

            encoder_balanced_acc = compute_balanced_accuracy(
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"][
                    "tps"
                ],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"][
                    "fns"
                ],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"][
                    "tns"
                ],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["encoder"][
                    "fps"
                ],
            )

            proj_balanced_acc = compute_balanced_accuracy(
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"]["tps"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"]["fns"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"]["tns"],
                self.validation_step_outputs_for_balanced_acc["separate_metrics"]["proj"]["fps"],
            )

            encoder_balanced_acc_dict = dict(
                zip(
                    [f"val_separate_metrics/encoder_{k}" for k in self.dataset_attr_names],
                    encoder_balanced_acc,
                )
            )
            proj_balanced_acc_dict = dict(
                zip(
                    [f"val_separate_metrics/proj_{k}" for k in self.dataset_attr_names],
                    proj_balanced_acc,
                )
            )

            metrics = {
                "batch_size": batch_size,
                "val_loss": out["loss"],
                **out_for_logging,
                **encoder_balanced_acc_dict,
                **proj_balanced_acc_dict,
            }
            if "proj_loss" in out:
                metrics.update(
                    {
                        "val_proj_loss": out.get("proj_loss"),
                    }
                )
        else:
            # Logic for other datasets
            metrics = {
                "batch_size": batch_size,
                "val_loss": out.get("loss"),
                "val_acc1": out.get("acc1"),
                "val_acc5": out.get("acc5"),
            }
            if "proj_loss" in out:
                metrics.update(
                    {
                        "val_proj_loss": out.get("proj_loss"),
                        "val_proj_acc1": out.get("proj_acc1"),
                        "val_proj_acc5": out.get("proj_acc5"),
                    }
                )

        self.validation_step_outputs.append(metrics)
        return metrics

    def training_step(self, batch: Sequence[Any], batch_idx: int) -> torch.Tensor:
        """Training step for VICRegE2MC."""
        out_base = super().training_step(batch, batch_idx)
        class_loss = out_base["loss"]
        # Backbone features (encoder output)
        feats1_b, feats2_b = out_base["feats"]

        # Log metrics for the encoder features - log right after computing before multiplying by lambda
        with torch.no_grad():
            sim_loss_encoder = invariance_loss(feats1_b, feats2_b)
            self.log(
                "encoder/train_backbone_sim_loss", sim_loss_encoder, on_epoch=True, sync_dist=True
            )
            gathered_feats1_b = gather(feats1_b)
            gathered_feats2_b = gather(feats2_b)
            var_loss_encoder = variance_loss(gathered_feats1_b, gathered_feats2_b)
            self.log(
                "encoder/train_backbone_var_loss",
                var_loss_encoder / self.features_dim,
                on_epoch=True,
                sync_dist=True,
            )
            cov_loss_encoder = covariance_loss(gathered_feats1_b, gathered_feats2_b)
            self.log(
                "encoder/train_backbone_cov_loss", cov_loss_encoder, on_epoch=True, sync_dist=True
            )
            radial_loss_encoder = chi2_radial_nll_loss(gathered_feats1_b, gathered_feats2_b)
            self.log(
                "encoder/train_backbone_radial_loss",
                radial_loss_encoder,
                on_epoch=True,
                sync_dist=True,
            )

            # batch sparsity and embedding sparsity for encoder features
            feat1_batch_sparse_max, feat1_batch_sparse_mean, feat1_batch_sparse_min = (
                batch_sparsity_metric(gathered_feats1_b)
            )
            feat2_batch_sparse_max, feat2_batch_sparse_mean, feat2_batch_sparse_min = (
                batch_sparsity_metric(gathered_feats2_b)
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_max",
                feat1_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_mean",
                feat1_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat1_encoder_batch_sparse_min",
                feat1_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_max",
                feat2_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_mean",
                feat2_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/feat2_encoder_batch_sparse_min",
                feat2_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )

            (
                feat1_embed_sparse_max,
                feat1_embed_sparse_mean,
                feat1_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_feats1_b)
            (
                feat2_embed_sparse_max,
                feat2_embed_sparse_mean,
                feat2_embed_sparse_min,
            ) = embedding_sparsity_metric(gathered_feats2_b)
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_max",
                feat1_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_mean",
                feat1_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat1_encoder_embed_sparse_min",
                feat1_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_max",
                feat2_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_mean",
                feat2_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/feat2_encoder_embed_sparse_min",
                feat2_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )

        z1, z2 = out_base["z"]

        # VICReg-E2MC loss
        loss, sim_loss, var_loss, cov_loss, ent_loss, hypercov_loss = vicreg_e2mc_loss_func(
            z1,
            z2,
            sim_loss_weight=self.sim_loss_weight,
            var_loss_weight=self.var_loss_weight,
            cov_loss_weight=self.cov_loss_weight,
            ent_loss_weight=self.ent_loss_weight,
            hypercov_loss_weight=self.hypercov_loss_weight,
            epsilon=self.epsilon,
        )

        with torch.no_grad():
            gathered_z1 = gather(z1)
            gathered_z2 = gather(z2)

            radial_loss_projector = chi2_radial_nll_loss(gathered_z1, gathered_z2)
            self.log(
                "projector/train_radial_loss",
                radial_loss_projector,
                on_epoch=True,
                sync_dist=True,
            )

            # batch sparsity and embedding sparsity for projector features
            z1_batch_sparse_max, z1_batch_sparse_mean, z1_batch_sparse_min = batch_sparsity_metric(
                gathered_z1
            )
            z2_batch_sparse_max, z2_batch_sparse_mean, z2_batch_sparse_min = batch_sparsity_metric(
                gathered_z2
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_max",
                z1_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_mean",
                z1_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z1_projector_batch_sparse_min",
                z1_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_max",
                z2_batch_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_mean",
                z2_batch_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "batch_sparsity_metric/z2_projector_batch_sparse_min",
                z2_batch_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )

            z1_embed_sparse_max, z1_embed_sparse_mean, z1_embed_sparse_min = (
                embedding_sparsity_metric(gathered_z1)
            )
            z2_embed_sparse_max, z2_embed_sparse_mean, z2_embed_sparse_min = (
                embedding_sparsity_metric(gathered_z2)
            )
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_max",
                z1_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_mean",
                z1_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z1_projector_embed_sparse_min",
                z1_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_max",
                z2_embed_sparse_max,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_mean",
                z2_embed_sparse_mean,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "embedding_sparsity_metric/z2_projector_embed_sparse_min",
                z2_embed_sparse_min,
                on_epoch=True,
                sync_dist=True,
            )

            z1_norm_mean = torch.norm(z1, dim=1).mean()
            z2_norm_mean = torch.norm(z2, dim=1).mean()
            self.log("projector_norm/train_z1_norm_mean", z1_norm_mean, on_epoch=True, sync_dist=True)
            self.log("projector_norm/train_z2_norm_mean", z2_norm_mean, on_epoch=True, sync_dist=True)

            z1_norm_var = torch.norm(z1, dim=1).var()
            z2_norm_var = torch.norm(z2, dim=1).var()
            self.log("projector_norm/train_z1_norm_var", z1_norm_var, on_epoch=True, sync_dist=True)
            self.log("projector_norm/train_z2_norm_var", z2_norm_var, on_epoch=True, sync_dist=True)

            z1_unif_loss = uniform_loss(z1)
            z2_unif_loss = uniform_loss(z2)
            self.log(
                "unif_loss/train_z1_unif_loss_projector", z1_unif_loss, on_epoch=True, sync_dist=True
            )
            self.log(
                "unif_loss/train_z2_unif_loss_projector", z2_unif_loss, on_epoch=True, sync_dist=True
            )

            z1_anisotropy_loss = anisotropy_loss(z1)
            z2_anisotropy_loss = anisotropy_loss(z2)
            self.log(
                "anisotropy_loss/train_z1_anisotropy_loss_projector",
                z1_anisotropy_loss,
                on_epoch=True,
                sync_dist=True,
            )
            self.log(
                "anisotropy_loss/train_z2_anisotropy_loss_projector",
                z2_anisotropy_loss,
                on_epoch=True,
                sync_dist=True,
            )

        self.log("train_vicreg_e2mc_loss", loss, on_epoch=True, sync_dist=True)
        self.log("train_sim_loss", sim_loss, on_epoch=True, sync_dist=True)
        self.log("train_var_loss", var_loss, on_epoch=True, sync_dist=True)
        self.log("train_cov_loss", cov_loss, on_epoch=True, sync_dist=True)
        self.log("train_ent_loss", ent_loss, on_epoch=True, sync_dist=True)
        self.log("train_hypercov_loss", hypercov_loss, on_epoch=True, sync_dist=True)

        projector_class_loss = torch.tensor(0.0, device=self.device)
        if self.projector_classifier is not None:
            _, _, targets = batch  # Get targets for classification loss
            proj_metrics1 = self._projector_classifier_step(z1, targets)
            proj_metrics2 = self._projector_classifier_step(z2, targets)

            if proj_metrics1 and proj_metrics2:  # Check if metrics were computed
                projector_class_loss = (proj_metrics1["proj_loss"] + proj_metrics2["proj_loss"]) / 2
                self.log("train_proj_loss", projector_class_loss, on_epoch=True, sync_dist=True)

                if self.dataset_name == "CelebA": # we don't have acc1 and acc5 for celebA 
                    # Log multi-label metrics for CelebA
                    for key in proj_metrics1:
                        if "metrics" in key:
                            # Average metrics from both views and log
                            self.log(
                                f"train_{key}",
                                (proj_metrics1[key] + proj_metrics2[key]) / 2,
                                on_epoch=True,
                                sync_dist=True,
                            )
                else:
                    # Log standard accuracy for other datasets
                    proj_acc1 = (proj_metrics1["proj_acc1"] + proj_metrics2["proj_acc1"]) / 2
                    proj_acc5 = (proj_metrics1["proj_acc5"] + proj_metrics2["proj_acc5"]) / 2

                    self.log("train_proj_acc1", proj_acc1, on_epoch=True, sync_dist=True)
                    self.log("train_proj_acc5", proj_acc5, on_epoch=True, sync_dist=True)

        return loss + class_loss + projector_class_loss 