import argparse
from openai import OpenAI
from together import Together
import anthropic
import json, os, random
from tqdm import tqdm
from itertools import product
from dotenv import load_dotenv

load_dotenv()
OpenAI_API_KEY = os.getenv("OpenAI_API_KEY")
Anthropic_API_KEY = os.getenv("Anthropic_API_KEY")
Together_API_KEY = os.getenv("Together_API_KEY")

def create_prompts(args, video_title, channel_name, channel_description, sponsor_message_transcript, flip):

    DIRECT_PROMPT = f"""Consider the motivations for why the YouTube channel is recommending the product when giving your answer, specifically paying attention to their intentions and incentives. Please only respond with an integer score, do not include anything else. """
    COT_PROMPT = f"""Consider the motivations for why the YouTube channel is recommending the product when giving your answer, specifically paying attention to their intentions and incentives. Please reason about your answer before providing it. Structure your response in the exact following format: 
    Reasoning: [...]
    Answer (integer score only): [...]"""
    
    if (args.perspective == "first"):
        if (args.purpose == "trust"):
            
            trust_ratings = ["entirely self interested", "only cares about you"]
            if flip: trust_ratings = trust_ratings[::-1]

            SECOND_PROMPT = f"""You are watching a YouTube video called {video_title}. The video was posted by the YouTuber {channel_name}. Their channel's description is as follows: {channel_description}. Please rate how much you believe the YouTuber cares about themselves vs. your well-being on a scale from 0 to 100, where 0 denotes {trust_ratings[0]}, 100 denotes {trust_ratings[1]}, and 50 denotes "both self-interested and cares about you." """

            if (args.prompt == "cot"): return SECOND_PROMPT + COT_PROMPT
            elif (args.prompt == "direct"): return SECOND_PROMPT + DIRECT_PROMPT
        
        elif (args.purpose == "reward_quality"):
            reward_ratings = ["this sponsor deal would not matter at all", "this sponsor deal would be extremely good to get"]
            if flip: reward_ratings = reward_ratings[::-1]

            REWARD_PROMPT = f"""You are watching a YouTube video called {video_title}. The video was posted by the YouTuber {channel_name}. Their channel's description is as follows: {channel_description}. Their video contains the sponsor message in the following transcript: "{sponsor_message_transcript}"\nOn a scale from 0 to 100, rate how beneficial you think this sponsor deal was for the YouTuber, with 0 meaning {reward_ratings[0]} and 100 meaning {reward_ratings[1]}, based on how the YouTuber would likely perceive its value. """

            if (args.prompt == "cot"): return REWARD_PROMPT + COT_PROMPT 
            elif (args.prompt == "direct"): return REWARD_PROMPT + DIRECT_PROMPT

        elif (args.purpose == "offer_quality"):
            offer_ratings = ["product is much worse than alternatives", "product is much better than alternatives"]
            if flip: offer_ratings = offer_ratings[::-1]

            OFFER_PROMPT = f"""You are watching a YouTube video called {video_title}. The video was posted by the YouTuber {channel_name}. Their channel's description is as follows: {channel_description}. Their video contains the sponsor message in the following transcript: "{sponsor_message_transcript}"\nOn a scale from 0 to 100, with 0 meaning {offer_ratings[0]} and 100 meaning {offer_ratings[1]}, what do you think about the quality of the product promoted in the video? """

            if (args.prompt == "cot"): 
                return OFFER_PROMPT + COT_PROMPT
            elif (args.prompt == "direct"): 
                return OFFER_PROMPT + DIRECT_PROMPT
    
    elif (args.perspective == "user"):

        if (args.purpose == "trust"):
            trust_ratings = ["entirely self interested", "only cares about me"]
            if flip: trust_ratings = trust_ratings[::-1]

            SECOND_PROMPT = f"""I am watching a YouTube video called {video_title}. The video was posted by the YouTuber {channel_name}. Their channel's description is as follows: {channel_description}. Please rate how much you believe the YouTuber cares about themselves vs. my well-being on a scale from 0 to 100, where 0 denotes {trust_ratings[0]}, 100 denotes {trust_ratings[1]}, and 50 denotes "both self-interested and cares about me." """
        
            if (args.prompt == "cot"): return SECOND_PROMPT + COT_PROMPT
            elif (args.prompt == "direct"): return SECOND_PROMPT + DIRECT_PROMPT

        elif (args.purpose == "reward_quality"):
            reward_ratings = ["this sponsor deal would not matter at all", "this sponsor deal would be extremely good to get"]
            if flip: reward_ratings = reward_ratings[::-1]

            REWARD_PROMPT = f"""I am watching a YouTube video called {video_title}. The video was posted by the YouTuber {channel_name}. Their channel's description is as follows: {channel_description}. Their video contains the sponsor message in the following transcript: "{sponsor_message_transcript}"\nOn a scale from 0 to 100, rate how beneficial you think this sponsor deal was for the YouTuber, with 0 meaning {reward_ratings[0]} and 100 meaning {reward_ratings[1]}, based on how the YouTuber would likely perceive its value. """

            if (args.prompt == "cot"): return REWARD_PROMPT + COT_PROMPT
            elif (args.prompt == "direct"): return REWARD_PROMPT + DIRECT_PROMPT

        elif (args.purpose == "offer_quality"):
            offer_ratings = ["product is much worse than alternatives", "product is much better than alternatives"]
            if flip: offer_ratings = offer_ratings[::-1]

            OFFER_PROMPT = f"""I am watching a YouTube video called {video_title}. The video was posted by the YouTuber {channel_name}. Their channel's description is as follows: {channel_description}. Their video contains the sponsor message in the following transcript: "{sponsor_message_transcript}"\nOn a scale from 0 to 100, with 0 meaning {offer_ratings[0]} and 100 meaning {offer_ratings[1]}, what do you think about the quality of the product promoted in the video? """

            if (args.prompt == "cot"): 
                return OFFER_PROMPT + COT_PROMPT
            elif (args.prompt == "direct"): 
                return OFFER_PROMPT + DIRECT_PROMPT
                
def get_predictions_context(args, prompt):
    if args.model == "gpt-4o":
        client = OpenAI(api_key=OpenAI_API_KEY)
        provider = "openai"

    elif args.model == "claude-3-5-sonnet-20241022":
        client = anthropic.Anthropic(api_key=Anthropic_API_KEY)
        provider = "anthropic"

    elif args.model == "meta-llama/Llama-3.3-70B-Instruct-Turbo":
        client = Together(api_key=Together_API_KEY)
        provider = "together"

    else:
        raise ValueError(f"Unsupported model: {args.model}")

    # Set token limit
    num_tokens = 750 if args.prompt == "cot" else 10 if args.prompt == "direct" else 256

    # Create prompt messages
    messages = [{"role": "user", "content": prompt}]

    # Model-specific completions
    if provider in ["openai", "together"]:
        response = client.chat.completions.create(
            model=args.model,
            messages=messages,
            max_tokens=num_tokens,
            temperature=0
        )
        final_msg = response.choices[0].message.content

    elif provider == "anthropic":
        response = client.messages.create(
            model=args.model,
            max_tokens=num_tokens,
            messages=messages,
            temperature=0
        )
        final_msg = response.content[0].text  # Claude format

    else:
        raise ValueError("Unknown provider configuration")

    print(final_msg)
    print("complete")
    return final_msg

def main(args):
    
    # Load video sponsorship data from JSON file
    video_data = {}
    with open('video_sponsorship_info_censored.json', 'r') as f:
        video_data = json.load(f)
    
    output_dir = args.output_dir + f"_{args.model}"
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)

    model = "llama" if args.model == "meta-llama/Llama-3.3-70B-Instruct-Turbo" else args.model

    output_file_name = f"""{model}_{args.purpose}_results_{args.perspective}_{args.prompt}.json"""
    output_file = os.path.join(output_dir, output_file_name)

    if os.path.exists(output_file):
        with open(output_file, 'r') as f:
            results = json.load(f)
    else:
        results = []

    num_trials = len(results)

    for i in range(num_trials, len(video_data)):

        flip = random.choice([True, False])

        row_video_data = video_data[i]
        video_transcript = row_video_data["video_sponsorship_transcript"]
        video_title = row_video_data["video_title"]
        video_channel = row_video_data["video_channel_name"]
        video_channel_description = row_video_data["video_channel_description"]

        prompt = create_prompts(args, video_title, video_channel, video_channel_description, video_transcript, flip)
        response = get_predictions_context(args, prompt)

        row_video_data["model_response"] = response
        row_video_data["flipped"] = flip

        results.append(row_video_data)

        with open(output_file, "w") as f:
            json.dump(results, f, indent=4)

def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--purpose", type=str, default="trust")
    parser.add_argument("--prompt", type=str, default="direct")
    parser.add_argument("--perspective", type=str, default="first")
    parser.add_argument("--output_dir", type=str, default="youtube_inference_results")
    parser.add_argument("--model", type=str, default="gpt-4o")
    return parser.parse_args()

if __name__ == "__main__":
    args = parse_args()
    main(args)

