import os
import random
import math
from PIL import Image, ImageDraw, ImageFilter, ImageEnhance
import numpy as np

PLAYER_1_FOLDER = "player_1_images"
PLAYER_2_FOLDER = "player_2_images"

os.makedirs(PLAYER_1_FOLDER, exist_ok=True)
os.makedirs(PLAYER_2_FOLDER, exist_ok=True)


def add_adversarial_effects(image, circles, radius, intensity="normal"):
    """Applies adversarial transformations while ensuring occlusions don't fully cover circles.
    
    intensity="normal" -> Used for Player 1 (mild distortions)
    intensity="cooked" -> Used for Player 2 (harder distortions)
    """

    # Blur: Normal vs. Cooked
    blur_radius = random.uniform(1.0, 2.5) if intensity == "normal" else random.uniform(4.0, 6.0)  # Even more blur
    image = image.filter(ImageFilter.GaussianBlur(blur_radius))

    # Add Random Noise: Normal vs. Cooked
    noise_strength = 20 if intensity == "normal" else 50  # Extreme noise for cooked images
    noise_array = np.random.randint(-noise_strength, noise_strength, (image.height, image.width, 3), dtype="int")
    image_array = np.array(image, dtype="int") + noise_array
    image_array = np.clip(image_array, 0, 255).astype("uint8")
    image = Image.fromarray(image_array)

    # Add Color Warping (for Player 2)
    if intensity == "cooked":
        enhancer = ImageEnhance.Color(image)
        image = enhancer.enhance(random.uniform(0.5, 1.5))  # Shift color balance slightly

    # Add Semi-Transparent Occlusions
    draw = ImageDraw.Draw(image, "RGBA")  # Enable transparency
    num_occlusions = random.randint(2, 5) if intensity == "normal" else random.randint(6, 10)  # More occlusions for cooked

    for _ in range(num_occlusions):
        max_attempts = 10
        while max_attempts > 0:
            x1 = random.randint(0, image.width - 50)
            y1 = random.randint(0, image.height - 50)
            x2 = x1 + random.randint(20, 50)
            y2 = y1 + random.randint(20, 50)

            # Check if occlusion overlaps a circle
            covers_circle = any(
                (cx - radius < x1 < cx + radius and cy - radius < y1 < cy + radius) or
                (cx - radius < x2 < cx + radius and cy - radius < y2 < cy + radius)
                for cx, cy in circles
            )

            if not covers_circle:  # Only place if it does NOT fully occlude a circle
                alpha = 100 if intensity == "normal" else 180  # More opaque in cooked images
                color = (random.randint(50, 200), random.randint(50, 200), random.randint(50, 200), alpha)
                draw.rectangle([x1, y1, x2, y2], fill=color)
                break  # Stop retrying once we find a good occlusion

            max_attempts -= 1  # Reduce attempts to avoid infinite loops

    # Apply Swirl Effect (for Player 2)
    if intensity == "cooked":
        image = image.rotate(random.uniform(-5, 5))  # Small random rotation
        enhancer = ImageEnhance.Contrast(image)
        image = enhancer.enhance(random.uniform(0.7, 1.3))  # Make contrast inconsistent

    return image


def generate_non_overlapping_circles(image_size=(500, 500), x=5, y=5, radius=20, intensity="normal"):
    """Generates an image with x blue and y yellow circles, ensuring non-overlapping placement.
    
    intensity="normal" -> Used for Player 1
    intensity="cooked" -> Used for Player 2
    """
    width, height = image_size
    image = Image.new("RGB", image_size, "white")
    draw = ImageDraw.Draw(image)

    circles = []

    def is_overlapping(new_pos):
        """Checks if a new circle overlaps with any existing circles."""
        for cx, cy in circles:
            if math.dist((cx, cy), new_pos) < 2 * radius:
                return True
        return False

    def random_non_overlapping_position():
        """Finds a valid random position without overlapping."""
        for _ in range(1000):
            pos = (random.randint(radius, width - radius), random.randint(radius, height - radius))
            if not is_overlapping(pos):
                return pos
        raise ValueError("Could not place circles without overlapping. Reduce the number of circles.")

    # Generate blue circles
    for _ in range(x):
        pos = random_non_overlapping_position()
        circles.append(pos)
        draw.ellipse([pos[0] - radius, pos[1] - radius, pos[0] + radius, pos[1] + radius], fill="blue", outline="black")

    # Generate yellow circles
    for _ in range(y):
        pos = random_non_overlapping_position()
        circles.append(pos)
        draw.ellipse([pos[0] - radius, pos[1] - radius, pos[0] + radius, pos[1] + radius], fill="yellow", outline="black")

    # Apply adversarial perturbations (normal for Player 1, cooked for Player 2)
    image = add_adversarial_effects(image, circles, radius, intensity=intensity)

    return image


# Generate exactly 20 unique images
unique_image_filenames = set()  # Track unique (x, y) pairs

while len(unique_image_filenames) < 20:
    x = random.randint(1, 10)
    y = random.randint(1, 10)

    if (x, y) not in unique_image_filenames:  # Ensure uniqueness
        unique_image_filenames.add((x, y))

        # Player 1 image (normal difficulty)
        player_1_image = generate_non_overlapping_circles(x=x, y=y, intensity="normal")
        player_1_filename = f"{PLAYER_1_FOLDER}/{y}_yellow_{x}_blue.png"
        player_1_image.save(player_1_filename)

        # Player 2 image (extra cooked)
        player_2_image = generate_non_overlapping_circles(x=x + 2, y=y + 2, intensity="cooked")
        player_2_filename = f"{PLAYER_2_FOLDER}/{y + 2}_yellow_{x + 2}_blue.png"
        player_2_image.save(player_2_filename)

        print(f"Generated: {player_1_filename} & {player_2_filename}")

print("Adversarial images successfully generated.")
