'''
Base code taken from https://github.com/facebookresearch/open_lth/blob/main/models/cifar_resnet.py
'''

import torch.nn as nn
import torch.nn.functional as F

from models.base_model import BaseModel
from layers import ModuleInjection

class Model(BaseModel):
    """A residual neural network as originally designed for CIFAR-10/100."""

    class Block(nn.Module):
        """A ResNet block."""
        def __init__(self, f_in: int, f_out: int, downsample=False):
            super(Model.Block, self).__init__()

            stride = 2 if downsample else 1
            self.conv1 = nn.Conv2d(f_in, f_out, kernel_size=3, stride=stride, padding=1, bias=False)
            self.bn1 = nn.BatchNorm2d(f_out)
            self.conv2 = nn.Conv2d(f_out, f_out, kernel_size=3, stride=1, padding=1, bias=False)
            self.bn2 = nn.BatchNorm2d(f_out)

            self.conv1, self.bn1, self.act1 = ModuleInjection.get_conv_bn_act(self.conv1, self.bn1)
            self.conv2, self.bn2, self.act2 = ModuleInjection.get_conv_bn_act(self.conv2, self.bn2)

            # No parameters for shortcut connections.
            if downsample or f_in != f_out:
                skip_conv = nn.Conv2d(f_in, f_out, kernel_size=1, stride=2, bias=False)
                skip_bn = nn.BatchNorm2d(f_out)

                skip_conv, skip_bn, _ = ModuleInjection.get_conv_bn_act(skip_conv, skip_bn, is_skip_connection=True)
                self.shortcut = nn.Sequential(
                    skip_conv,
                    skip_bn
                )
            else:
                self.shortcut = nn.Sequential()

        def forward(self, x):
            out = self.act1(self.bn1(self.conv1(x)))
            out = self.bn2(self.conv2(out))
            out += self.shortcut(x)
            return self.act2(out)

    def __init__(self, plan, outputs=None):
        super(Model, self).__init__()
        outputs = outputs or 10

        # Initial convolution.
        current_filters = plan[0][0]
        self.conv = nn.Conv2d(3, current_filters, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn = nn.BatchNorm2d(current_filters)
        
        self.conv, self.bn, self.act = ModuleInjection.get_conv_bn_act(self.conv, self.bn, is_first_layer=True)

        # The subsequent blocks of the ResNet.
        blocks = []
        for segment_index, (filters, num_blocks) in enumerate(plan):
            for block_index in range(num_blocks):
                downsample = segment_index > 0 and block_index == 0
                blocks.append(Model.Block(current_filters, filters, downsample))
                current_filters = filters

        self.blocks = nn.Sequential(*blocks)

        # Final fc layer. Size = number of filters in last segment.
        self.fc = nn.Linear(plan[-1][0], outputs)

        # self.init_weights()
        self.prev_module[self.bn] = None

        prev = self.bn
        for b in self.blocks:
            self.prev_module[b.bn1] = prev
            self.prev_module[b.bn2] = b.bn1
            if len(b.shortcut)>0:
                self.prev_module[b.shortcut[1]] = prev
                prev = (b.shortcut[1], b.bn2)
            else:
                prev = (prev, b.bn2) 

    def forward(self, x):
        out = self.act(self.bn(self.conv(x)))
        out = self.blocks(out)
        out = F.avg_pool2d(out, out.size()[3])
        out = out.view(out.size(0), -1)
        out = self.fc(out)
        return out

    @property
    def output_layer_names(self):
        return ['fc.weight', 'fc.bias']

    @staticmethod
    def is_valid_model_name(model_name):
        return (model_name.startswith('cifar_resnet_') and
                5 > len(model_name.split('_')) > 2 and
                all([x.isdigit() and int(x) > 0 for x in model_name.split('_')[2:]]) and
                (int(model_name.split('_')[2]) - 2) % 6 == 0 and
                int(model_name.split('_')[2]) > 2)

    @staticmethod
    def get_model_from_config(config):
        """The naming scheme for a ResNet is 'cifar_resnet_N[_W]'.

        The ResNet is structured as an initial convolutional layer followed by three "segments"
        and a linear output layer. Each segment consists of D blocks. Each block is two
        convolutional layers surrounded by a residual connection. Each layer in the first segment
        has W filters, each layer in the second segment has 32W filters, and each layer in the
        third segment has 64W filters.

        The name of a ResNet is 'cifar_resnet_N[_W]', where W is as described above.
        N is the total number of layers in the network: 2 + 6D.
        The default value of W is 16 if it isn't provided.

        For example, ResNet-20 has 20 layers. Exclusing the first convolutional layer and the final
        linear layer, there are 18 convolutional layers in the blocks. That means there are nine
        blocks, meaning there are three blocks per segment. Hence, D = 3.
        The name of the network would be 'cifar_resnet_20' or 'cifar_resnet_20_16'.
        """
        model_name = config.model_name
        outputs = config.dataset_num_classes
        ModuleInjection.update_model_compression_strategy(config.model_compression_strategy)

        if not Model.is_valid_model_name(model_name):
            raise ValueError('Invalid model name: {}'.format(model_name))

        name = model_name.split('_')
        W = 16 if len(name) == 3 else int(name[3])
        D = int(name[2])
        if (D - 2) % 3 != 0:
            raise ValueError('Invalid ResNet depth: {}'.format(D))
        D = (D - 2) // 6
        plan = [(W, D), (2*W, D), (4*W, D)]

        net = Model(plan, outputs)
        net.prunable_modules = ModuleInjection.prunable_modules

        return net

    @staticmethod
    def get_default_args(training_config):

        if training_config.dataset_name.startswith('cifar'):

            training_config.epochs = 160
            training_config.training_batch_size = 128
            training_config.test_batch_size = 256
            training_config.train_shuffle = True
            training_config.test_shuffle = False

            training_config.scheduler_type = "multi_step"
            training_config.scheduler_gamma = 0.1
            training_config.scheduler_milestones = [80, 120]

            training_config.optimizer_type = "sgd"
            training_config.optimizer_lr = 0.1
            training_config.optimizer_momentum = 0.9
            training_config.optimizer_weight_decay = 1e-4
            training_config.optimizer_no_decay = ["bias"]

        else:
            raise ValueError('Invalid dataset name: {}'.format(training_config.dataset_name))

        return training_config