package authoringassistant.llm;

import com.fasterxml.jackson.databind.ObjectMapper;
import authoringassistant.paragraph.Expression;
import it.unisa.cluelab.lllm.llm.LLMEvaluatorAgent;
import it.unisa.cluelab.lllm.llm.prompt.Prompt;
import it.unisa.cluelab.lllm.llm.prompt.PromptList;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.logging.Logger;

public class FluidOpenAIGpt5Agent extends LLMEvaluatorAgent<Expression> {
    public static Logger logger = Logger.getLogger(FluidOpenAIGpt5Agent.class.getName());
    private final HttpClient httpClient;
    private final ObjectMapper objectMapper;
    private final String token;
    private final double temperature;
    private final int ctx;

    public FluidOpenAIGpt5Agent(JSONObject settings) {
        super(settings);
        this.token = settings.getString("openai-token");
        this.temperature = (double)settings.getFloat("temperature");
        this.ctx = settings.getInt("num_ctx");
        this.httpClient = HttpClient.newBuilder()
                .connectTimeout(Duration.ofSeconds(30))
                .build();
        this.objectMapper = new ObjectMapper();
    }

    @Override
    public Expression evaluate(PromptList prompts, String grid) throws IOException {
        logger.info("Execution of the FluidOpenAIGpt5Agent");

        try {
            JSONObject requestBody = buildRequestBody(prompts);
            String response = makeHttpRequest(requestBody);
            return parseResponse(response);

        } catch (Exception e) {
            logger.severe("Error calling GPT-5 API: " + e.getMessage());
            return null;
        }
    }

    private JSONObject buildRequestBody(PromptList prompts) {
        JSONObject requestBody = new JSONObject();
        requestBody.put("model", "gpt-5");
        requestBody.put("reasoning_effort", "minimal");
        requestBody.put("temperature", this.temperature);
        requestBody.put("max_completion_tokens", this.ctx);

        JSONArray messages = new JSONArray();
        for (Prompt p : prompts) {
            JSONObject message = new JSONObject();
            message.put("role", p.getRole());
            message.put("content", p.getContent());
            messages.put(message);
        }

        requestBody.put("messages", messages);

        return requestBody;
    }

    private String makeHttpRequest(JSONObject requestBody) throws IOException, InterruptedException {
        String requestBodyString = requestBody.toString();

        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create("https://api.openai.com/v1/chat/completions"))
                .header("Content-Type", "application/json")
                .header("Authorization", "Bearer " + token)
                .timeout(Duration.ofSeconds(90))
                .POST(HttpRequest.BodyPublishers.ofString(requestBodyString))
                .build();

        HttpResponse<String> response = httpClient.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() != 200) {
            throw new IOException(STR."HTTP error: \{response.statusCode()} - \{response.body()}");
        }

        return response.body();
    }

    private Expression parseResponse(String responseBody) throws IOException {
        JSONObject response = new JSONObject(responseBody);
        JSONArray choices = response.getJSONArray("choices");
        JSONObject choice = choices.getJSONObject(0);
        JSONObject message = choice.getJSONObject("message");

        String content = message.optString("content", "");
        return new Expression(content, "", null);
    }
}