**This (anonymised, trimmed down) version of the code is taken from the dev branch for this project. It has been tested to work for the problems provided, but may still have sharp edges.**

# PyTorch Implementation of Differentiable SDE Solvers
This codebase provides [stochastic differential equation (SDE)](https://en.wikipedia.org/wiki/Stochastic_differential_equation) solvers with GPU support and efficient sensitivity analysis.
Similar to [torchdiffeq](https://github.com/rtqichen/torchdiffeq), algorithms in this repository are fully supported to run on GPUs.

## Installation
```shell script
pip install git+https://github.com/[redacted].git
```

## Usage
The central functions of interest are `sdeint` and `sdeint_adjoint`. They can be imported as follows:
```Python
from torchsde import sdeint, sdeint_adjoint
```

### Integrating SDEs
SDEs are defined with two vector fields, named respectively the *drift* and *diffusion* functions.
The drift is analogous to the vector field in ODEs, whereas the diffusion controls how the [Brownian motion](https://en.wikipedia.org/wiki/Brownian_motion) affects the system.

By default, solvers in the codebase integrate SDEs that are `torch.nn.Module` objects with the drift function `f` and diffusion function `g`.
For instance, the [Geometric Brownian motion](https://en.wikipedia.org/wiki/Geometric_Brownian_motion) can be integrated as follows:
```Python
import torch
from torchsde import sdeint


class SDE(torch.nn.Module):

    def __init__(self, mu, sigma):
        super().__init__()
        self.noise_type="diagonal"
        self.sde_type = "ito"

        self.mu = mu
        self.sigma = sigma

    def f(self, t, y):
        return self.mu * y

    def g(self, t, y):
        return self.sigma * y

batch_size, d, m = 4, 1, 1  # State dimension d, Brownian motion dimension m.
geometric_bm = SDE(mu=0.5, sigma=1)
y0 = torch.zeros(batch_size, d).fill_(0.1)  # Initial state.
ts = torch.linspace(0, 1, 20)
ys = sdeint(geometric_bm, y0, ts)
```
To ensure the solvers work, `f` and `g` must take in the time `t` and state `y` in the specific order.
**Most importantly, the SDE class must have the attributes `sde_type` and `noise_type`.**
The noise type of the SDE determines what numerical algorithms may be used.

#### Possible noise types and explanations when state dimension=d
- `diagonal`: The diffusion `g` is element-wise and has output size (d,). The Brownian motion is d-dimensional.
- `additive`: The diffusion `g` is constant w.r.t. `y` and has output size (d, m). The Brownian motion is m-dimensional.
- `scalar`: The diffusion `g` has output size (d, 1). The Brownian motion is 1-dimensional.
- `general`: The diffusion `g` has output size (d, m). The Brownian motion is m-dimensional.

### Keyword arguments of `sdeint`
- `bm`: A `BrownianPath`, `BrownianTree` or `BrownianInterval` object. Optionally include to seed the solver's computation.
- `method`: One of the solvers listed below.
- `dt`: A float for the constant step size or initial step size for adaptive time-stepping.
- `adaptive`: If True, use adaptive time-stepping.
- `rtol`: Relative tolerance.
- `atol`: Absolute tolerance.
- `dt_min`: Minimum step size.

#### List of SDE solvers
Ito solvers:
- `euler`
- `milstein`
- `srk`
  
Stratonovich solvers:
- `euler_heun`
- `midpoint`
- `heun`