"""
Common utility functions used throughout the project.
"""
import os
import torch
import numpy as np
from typing import Dict, Any, Optional
import json
import logging


def setup_logging(log_dir: str, name: str) -> logging.Logger:
    """Set up logging configuration.
    
    Args:
        log_dir: Directory to save logs
        name: Name of the logger
        
    Returns:
        Configured logger
    """
    os.makedirs(log_dir, exist_ok=True)
    logger = logging.getLogger(name)
    logger.setLevel(logging.INFO)
    
    # File handler
    fh = logging.FileHandler(os.path.join(log_dir, f"{name}.log"))
    fh.setLevel(logging.INFO)
    
    # Console handler
    ch = logging.StreamHandler()
    ch.setLevel(logging.INFO)
    
    # Formatter
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    fh.setFormatter(formatter)
    ch.setFormatter(formatter)
    
    logger.addHandler(fh)
    logger.addHandler(ch)
    
    return logger


def save_config(config: Dict[str, Any], path: str):
    """Save configuration to a JSON file.
    
    Args:
        config: Configuration dictionary
        path: Path to save the configuration
    """
    with open(path, 'w') as f:
        json.dump(config, f, indent=4)


def load_config(path: str) -> Dict[str, Any]:
    """Load configuration from a JSON file.
    
    Args:
        path: Path to load the configuration from
        
    Returns:
        Configuration dictionary
    """
    with open(path, 'r') as f:
        return json.load(f)


def set_seed(seed: int):
    """Set random seed for reproducibility.
    
    Args:
        seed: Random seed
    """
    torch.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)
    np.random.seed(seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False 