import numpy as np

def simulate_multitask_dataset(
    n_samples=100,
    d_numeric=16,
    d_text=16,
    D=32,
    noise_std=0.1,
    random_state=42
):
    rng = np.random.RandomState(random_state)
    # Generate features
    X_numeric = rng.uniform(-1, 1, size=(n_samples, d_numeric))
    X_text = rng.uniform(-1, 1, size=(n_samples, d_text))

    # Random Fourier Features for phi (text) and psi (numeric)
    W_phi = rng.normal(0, 1, size=(D, d_text))
    b_phi = rng.uniform(0, 2 * np.pi, size=(D,))
    W_psi = rng.normal(0, 1, size=(D, d_numeric))
    b_psi = rng.uniform(0, 2 * np.pi, size=(D,))
    def phi(x_text):
        return np.sqrt(2 / D) * np.cos(x_text @ W_phi.T + b_phi)
    def psi(x_num):
        return np.sqrt(2 / D) * np.cos(x_num @ W_psi.T + b_psi)

    # Sample parameters
    alpha1 = rng.normal(0, 1, size=(d_numeric,))
    beta1 = rng.normal(0, 1, size=(D,))
    omega1 = rng.normal(0, 1, size=(d_numeric,))
    gamma1 = rng.uniform(0.5, 1.5)
    alpha2 = rng.normal(0, 1, size=(d_text,))
    beta2 = rng.normal(0, 1, size=(D,))
    omega2 = rng.normal(0, 1, size=(d_text,))
    gamma2 = rng.uniform(0.5, 1.5)

    # Compute feature maps
    phi_X_text = phi(X_text)
    psi_X_num = psi(X_numeric)

    # Compute targets
    y1 = (
        X_numeric @ alpha1
         + phi_X_text @ beta1
         + gamma1 * np.sin(X_numeric @ omega1)
        + rng.normal(0, noise_std, size=n_samples)
    )
    y2 = (
        X_text @ alpha2
         + psi_X_num @ beta2
         + gamma2 * np.cos(X_text @ omega2)
        + rng.normal(0, noise_std, size=n_samples)
    )

    return {
        "X_numeric": X_numeric,
        "X_text": X_text,
        "phi_X_text": phi_X_text,
        "psi_X_num": psi_X_num,
        "y1": y1,
        "y2": y2,
        "params": {
            "alpha1": alpha1,
            "beta1": beta1,
            "omega1": omega1,
            "gamma1": gamma1,
            "alpha2": alpha2,
            "beta2": beta2,
            "omega2": omega2,
            "gamma2": gamma2,
            "W_phi": W_phi,
            "b_phi": b_phi,
            "W_psi": W_psi,
            "b_psi": b_psi,
            "noise_std": noise_std,
            "random_state": random_state,
            "D": D,
            "d_numeric": d_numeric,
            "d_text": d_text
        }
    }