# -----------------------------------------------------------------------------------
# SemanIR: Sharing Key Semantics in Transformer Makes Efficient Image Restoration
# -----------------------------------------------------------------------------------


import torch
import torch.nn as nn
import torch.nn.functional as F
from fairscale.nn import checkpoint_wrapper
from omegaconf import OmegaConf
from model.upsample import Upsample, UpsampleOneStep
from model.ops import (
    _fill_window,
)
from model.common import (
    build_last_conv, _parse_list, LayerNorm
)
from model.semanir_block import SemanticTransformerBlock
from model.semanir_block_v2 import SemanticTransformerBlockV2

from timm.models.layers import to_2tuple, trunc_normal_


def calculate_mask(input_resolution, window_size, shift_size):
    """
    Use case: 1)
    """
    # calculate attention mask for SW-MSA
    if isinstance(shift_size, int):
        shift_size = to_2tuple(shift_size)
    mask_windows = _fill_window(input_resolution, window_size, shift_size)

    attn_mask = mask_windows.unsqueeze(1) - mask_windows.unsqueeze(2)
    attn_mask = attn_mask.masked_fill(attn_mask != 0, float("-inf")).masked_fill(
        attn_mask == 0, float(0.0)
    )  # nW, window_size**2, window_size**2

    return attn_mask


class TransformerStage(nn.Module):
    """Residual Swin Transformer Block (RSTB).
    Args:
        dim (int): Number of input channels.
        input_resolution (tuple[int]): Input resolution.
        depth (int): Number of blocks.
        num_heads (int): Number of attention heads.
        window_size (int): Local window size.
        mlp_ratio (float): Ratio of mlp hidden dim to embedding dim.
        qkv_bias (bool, optional): If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float | None, optional): Override default qk scale of head_dim ** -0.5 if set.
        drop (float, optional): Dropout rate. Default: 0.0
        attn_drop (float, optional): Attention dropout rate. Default: 0.0
        drop_path (float | tuple[float], optional): Stochastic depth rate. Default: 0.0
        conv_type: The convolutional block before residual connection.
    """

    def __init__(
        self,
        dim,
        input_resolution,
        depth,
        num_heads,
        window_size,
        mlp_ratio=2.0,
        top_k=32,
        qkv_bias=True,
        qk_scale=None,
        drop=0.0,
        attn_drop=0.0,
        drop_path=0.0,
        conv_type="1conv",
        args=None,
    ):
        super().__init__()

        self.dim = dim
        self.input_resolution = input_resolution

        self.blocks = nn.ModuleList()

        for i in range(depth):
            if args.version == "v1":
                block = SemanticTransformerBlock(
                    dim=dim,
                    input_resolution=input_resolution,
                    num_heads=num_heads,
                    window_size=window_size,
                    shift_size=0 if (i % 2 == 0) else window_size // 2,
                    mlp_ratio=mlp_ratio,
                    top_k=top_k,
                    qkv_bias=qkv_bias,
                    qk_scale=qk_scale,
                    drop=drop,
                    attn_drop=attn_drop,
                    drop_path=drop_path[i] if isinstance(drop_path, list) else drop_path,
                    use_pe=args.use_pe,
                    graph_type=args.graph_type,
                    args=args,
                )
            else:
                block = SemanticTransformerBlockV2(
                    dim=dim,
                    input_resolution=input_resolution,
                    num_heads=num_heads,
                    window_size=window_size,
                    shift_size=0 if (i % 2 == 0) else window_size // 2,
                    global_size=args.global_size,
                    mlp_ratio=mlp_ratio,
                    top_k=top_k,
                    qkv_bias=qkv_bias,
                    qk_scale=qk_scale,
                    drop=drop,
                    attn_drop=attn_drop,
                    drop_path=drop_path[i] if isinstance(drop_path, list) else drop_path,
                    use_pe=args.use_pe,
                    graph_type=args.graph_type,
                    args=args,
                )
            if args.fairscale_checkpoint:
                block = checkpoint_wrapper(block, offload_to_cpu=args.offload_to_cpu)
            self.blocks.append(block)

        self.conv = build_last_conv(conv_type, dim)

    def forward(self, x, mask=None):
        res = x
        sim_measure = {}
        for i, blk in enumerate(self.blocks):
            m = None if i % 2 == 0 else mask
            if i <= 1:
                res, measure = blk(res, m)
                sim_measure[i % 2 == 0] = measure
            else:
                res, measure = blk(res, m, sim_measure[i % 2 == 0])
        res = self.conv(res)

        return res + x

    def flops(self):
        flops = 0
        flops += self.blocks.flops()
        H, W = self.input_resolution
        flops += H * W * self.dim * self.dim * 9

        return flops


class SemanIR(nn.Module):
    r"""SemanIR
    Args:
        img_size (int | tuple(int)): Input image size. Default 64
        in_chans (int): Number of input image channels. Default: 3
        embed_dim (int): Patch embedding dimension. Default: 96
        depths (tuple(int)): Depth of each Swin Transformer layer.
        num_heads (tuple(int)): Number of attention heads in different layers.
        window_size (int): Window size. Default: 7
        mlp_ratio (float): Ratio of mlp hidden dim to embedding dim. Default: 4
        qkv_bias (bool): If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float): Override default qk scale of head_dim ** -0.5 if set. Default: None
        drop_rate (float): Dropout rate. Default: 0
        attn_drop_rate (float): Attention dropout rate. Default: 0
        drop_path_rate (float): Stochastic depth rate. Default: 0.1
        norm_layer (nn.Module): Normalization layer. Default: nn.LayerNorm.
        upscale: Upscale factor. 2/3/4/8 for image SR, 1 for denoising and compress artifact reduction
        img_range: Image range. 1. or 255.
        upsampler: The reconstruction reconstruction module. 'pixelshuffle'/'pixelshuffledirect'/'nearest+conv'/None
        conv_type: The convolutional block before residual connection. '1conv'/'3conv'
    """

    def __init__(
        self,
        img_size=64,
        in_channels=3,
        out_channels=None,
        embed_dim=96,
        depths="6+6+6+6+6+6",
        num_heads="6+6+6+6+6+6",
        window_size=8,
        global_size=64,
        mlp_ratio=2.0,
        top_k=32,
        qkv_bias=True,
        qk_scale=None,
        drop_rate=0.0,
        attn_drop_rate=0.0,
        drop_path_rate=0.1,
        upscale=2,
        img_range=1.0,
        upsampler="",
        conv_type="1conv",
        fairscale_checkpoint=False,  # fairscale activation checkpointing
        offload_to_cpu=False,
        use_pe=True,  #  investigate positional encoding
        graph_type="mask",
        version="v1",
        **kwargs,
    ):
        # TODO: add triton implementation
        # Select top_k and window_size during inference

        super(SemanIR, self).__init__()
        # Process the input arguments
        out_channels = out_channels or in_channels
        num_out_feats = 64
        self.embed_dim = embed_dim
        depths = _parse_list(depths)
        num_heads = _parse_list(num_heads)
        self.upscale = upscale
        self.upsampler = upsampler
        self.window_size = window_size

        self.img_range = img_range
        if in_channels == 3:
            rgb_mean = (0.4488, 0.4371, 0.4040)
            self.mean = torch.Tensor(rgb_mean).view(1, 3, 1, 1)
        else:
            self.mean = torch.zeros(1, 1, 1, 1)

        self.pad_size = window_size
        args = OmegaConf.create(
            {
                "fairscale_checkpoint": fairscale_checkpoint,
                "offload_to_cpu": offload_to_cpu,
                "use_pe": use_pe,
                "graph_type": graph_type,
                "global_size": global_size,
                "version": version,
            }
        )

        # Head of the network. First convolution.
        self.conv_first = nn.Conv2d(in_channels, embed_dim, 3, 1, 1)

        # Body of the network
        self.norm_start = LayerNorm(embed_dim, 1)
        self.pos_drop = nn.Dropout(p=drop_rate)

        # stochastic depth
        dpr = [x.item() for x in torch.linspace(0, drop_path_rate, sum(depths))]
        # stochastic depth decay rule

        self.layers = nn.ModuleList()
        for i_layer in range(len(depths)):
            layer = TransformerStage(
                dim=embed_dim,
                input_resolution=to_2tuple(img_size),
                depth=depths[i_layer],
                num_heads=num_heads[i_layer],
                window_size=window_size,
                mlp_ratio=mlp_ratio,
                top_k=top_k,
                qkv_bias=qkv_bias,
                qk_scale=qk_scale,
                drop=drop_rate,
                attn_drop=attn_drop_rate,
                drop_path=dpr[
                    sum(depths[:i_layer]) : sum(depths[: i_layer + 1])
                ],  # no impact on SR results
                conv_type=conv_type,
                args=args,
            )
            self.layers.append(layer)
        self.norm_end = LayerNorm(embed_dim, 1)

        # Tail of the network
        self.conv_after_body = build_last_conv(conv_type, embed_dim)

        #####################################################################################################
        ################################ 3, high quality image reconstruction ################################
        if self.upsampler == "pixelshuffle":
            # for classical SR
            self.conv_before_upsample = nn.Sequential(
                nn.Conv2d(embed_dim, num_out_feats, 3, 1, 1), nn.LeakyReLU(inplace=True)
            )
            self.upsample = Upsample(upscale, num_out_feats)
            self.conv_last = nn.Conv2d(num_out_feats, out_channels, 3, 1, 1)
        elif self.upsampler == "pixelshuffledirect":
            # for lightweight SR (to save parameters)
            self.upsample = UpsampleOneStep(
                upscale,
                embed_dim,
                out_channels,
            )
        elif self.upsampler == "nearest+conv":
            # for real-world SR (less artifacts)
            assert self.upscale == 4, "only support x4 now."
            self.conv_before_upsample = nn.Sequential(
                nn.Conv2d(embed_dim, num_out_feats, 3, 1, 1), nn.LeakyReLU(inplace=True)
            )
            self.conv_up1 = nn.Conv2d(num_out_feats, num_out_feats, 3, 1, 1)
            self.conv_up2 = nn.Conv2d(num_out_feats, num_out_feats, 3, 1, 1)
            self.conv_hr = nn.Conv2d(num_out_feats, num_out_feats, 3, 1, 1)
            self.conv_last = nn.Conv2d(num_out_feats, out_channels, 3, 1, 1)
            self.lrelu = nn.LeakyReLU(negative_slope=0.2, inplace=True)
        else:
            # for image denoising and JPEG compression artifact reduction
            self.conv_last = nn.Conv2d(embed_dim, out_channels, 3, 1, 1)

        self.apply(self._init_weights)
        # if init_method in ["l", "w"] or init_method.find("t") >= 0:
        #     for layer in self.layers:
        #         layer._init_weights()

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            # Only used to initialize linear layers
            # weight_shape = m.weight.shape
            # if weight_shape[0] > 256 and weight_shape[1] > 256:
            #     std = 0.004
            # else:
            #     std = 0.02
            # print(f"Standard deviation during initialization {std}.")
            trunc_normal_(m.weight, std=0.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, (nn.LayerNorm, LayerNorm)):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    @torch.jit.ignore
    def no_weight_decay(self):
        return {"absolute_pos_embed"}

    @torch.jit.ignore
    def no_weight_decay_keywords(self):
        return {"relative_position_bias_table"}

    def check_image_size(self, x):
        _, _, h, w = x.size()
        mod_pad_h = (self.pad_size - h % self.pad_size) % self.pad_size
        mod_pad_w = (self.pad_size - w % self.pad_size) % self.pad_size
        x = F.pad(x, (0, mod_pad_w, 0, mod_pad_h), "reflect")
        return x

    def forward_features(self, x):
        x = self.norm_start(x)
        x = self.pos_drop(x)

        mask = calculate_mask(
            x.shape[2:], to_2tuple(self.window_size), to_2tuple(self.window_size // 2)
        ).to(dtype=x.dtype, device=x.device)

        for layer in self.layers:
            x = layer(x, mask)

        x = self.norm_end(x)  # B L C
        return x

    def forward(self, x):
        H, W = x.shape[2:]  # [B, 3, H(64), W(64)]

        x = self.check_image_size(x)

        self.mean = self.mean.type_as(x)
        x = (x - self.mean) * self.img_range

        if self.upsampler == "pixelshuffle":
            # for classical SR
            x = self.conv_first(x)  # [B, 180, H(64), W(64)]
            x = self.conv_after_body(self.forward_features(x)) + x
            x = self.conv_before_upsample(x)
            x = self.conv_last(self.upsample(x))
        elif self.upsampler == "pixelshuffledirect":
            # for lightweight SR
            x = self.conv_first(x)
            x = self.conv_after_body(self.forward_features(x)) + x
            x = self.upsample(x)
        elif self.upsampler == "nearest+conv":
            # for real-world SR
            x = self.conv_first(x)
            x = self.conv_after_body(self.forward_features(x)) + x
            x = self.conv_before_upsample(x)
            x = self.lrelu(
                self.conv_up1(
                    torch.nn.functional.interpolate(x, scale_factor=2, mode="nearest")
                )
            )
            x = self.lrelu(
                self.conv_up2(
                    torch.nn.functional.interpolate(x, scale_factor=2, mode="nearest")
                )
            )
            x = self.conv_last(self.lrelu(self.conv_hr(x)))
        else:
            # for image denoising and JPEG compression artifact reduction
            x_first = self.conv_first(x)
            res = self.conv_after_body(self.forward_features(x_first)) + x_first
            x = x + self.conv_last(res)

        x = x / self.img_range + self.mean

        return x[:, :, : H * self.upscale, : W * self.upscale]

    def flops(self):
        flops = 0
        H, W = self.patches_resolution
        flops += H * W * 3 * self.embed_dim * 9
        flops += self.patch_embed.flops()
        for i, layer in enumerate(self.layers):
            flops += layer.flops()
        flops += H * W * 3 * self.embed_dim * self.embed_dim
        flops += self.upsample.flops()
        return flops

    def convert_checkpoint(self, state_dict):
        for k in list(state_dict.keys()):
            if (
                k.find("relative_coords_table") >= 0
                or k.find("relative_position_index") >= 0
            ):
                state_dict.pop(k)
                print(k)
        return state_dict


def model_analysis(model, height, width, print_flops=False, print_run_time=True, num_iter=100):

    print(model)

    # number of parameters
    num_params = 0
    for p in model.parameters():
        if p.requires_grad:
            num_params += p.numel()
    print(f"Params: {num_params / 10 ** 6: 0.2f} M")

    x = torch.randn((1, 3, height, width))

    # Flops
    if print_flops:
        from fvcore.nn import FlopCountAnalysis, parameter_count_table, flop_count_table
        flops = FlopCountAnalysis(model, x)
        print(flop_count_table(flops))
        total_flops = flops.total() / 10 ** 9
        print(f"FLOPs: {total_flops} G")
        print(parameter_count_table(model))

    # Runtime
    if print_run_time:
        x = x.cuda()
        model = model.to(torch.device("cuda"))
        model_runtime = []
        start = torch.cuda.Event(enable_timing=True)
        end = torch.cuda.Event(enable_timing=True)
        for i in range(num_iter):
            start.record()
            model(x)
            end.record()
            torch.cuda.synchronize()
            model_runtime.append(start.elapsed_time(end))
        avg_runtime = sum(model_runtime[1:]) / (len(model_runtime) - 1)
        print(f"Avg runtime: {avg_runtime} ms")
        print(model_runtime)

if __name__ == "__main__":
    upscale = 4
    window_size = 8
    height = (256 // upscale // window_size) * window_size
    width = (256 // upscale // window_size) * window_size
    img_size = 64

    model = SemanIR(
        window_size=window_size,
        top_k=32,
        depths=[6, 6, 6, 6, 6, 6],
        embed_dim=180,
        num_heads=[6, 6, 6, 6, 6, 6],
        mlp_ratio=2,
        upscale=upscale,
        upsampler="pixelshuffle",
        conv_type="1conv",
        img_size=img_size,
        img_range=1.0,
        fairscale_checkpoint=False,
        offload_to_cpu=False,
        version="v2"
    )

    # for k, v in model.state_dict().items():
    #     print(k, v.shape)
    # image_path = "/Users/jinliang/Documents/n01440764_10042.JPEG"
    # import imageio
    # import numpy as np
    # import torchvision.transforms as transforms
    #
    # image = imageio.imread(image_path)
    # image = np.ascontiguousarray(image)
    # image = transforms.functional.to_tensor(image).unsqueeze(0)[:, :, :img_size, :img_size]

    # print(y.shape)

    # for k, v in s_new['params'].items():
    #     print(k, v.shape)
    # model.load_state_dict(s_new['params'])

    model_analysis(model, height, width)
