# -----------------------------------------------------------------------------------
# SemanIR: Sharing Key Semantics in Transformer Makes Efficient Image Restoration
# -----------------------------------------------------------------------------------


from typing import Tuple
from math import prod

import torch
import torch.nn as nn
from timm.models.layers import DropPath, to_2tuple
from model.common import get_relative_coords_table, get_relative_position_index, Mlp, LayerNorm, grid_partition, grid_reverse


def window_partition(x, window_size: Tuple[int, int]):
    """
    Args:
        x: (B, C, H, W)
        window_size (tuple[int, int]): window size

    Returns:
        windows: (num_windows*B, window_size * window_size, C) or B, N, C
    """
    B, C, H, W = x.shape
    x = x.view(
        B, C, H // window_size[0], window_size[0], W // window_size[1], window_size[1]
    )
    windows = (
        x.permute(0, 2, 4, 3, 5, 1)
        .contiguous()
        .view(-1, window_size[0] * window_size[1], C)
    )
    return windows


def window_reverse(windows, window_size: Tuple[int, int], img_size: Tuple[int, int]):
    """
    Args:
        windows: (num_windows * B, window_size[0] * window_size[1], C)
        window_size (Tuple[int, int]): Window size
        img_size (Tuple[int, int]): Image size

    Returns:
        x: (B, C, H, W)
    """
    C = windows.shape[-1]
    H, W = img_size
    x = windows.view(
        -1, H // window_size[0], W // window_size[1], window_size[0], window_size[1], C
    )
    x = x.permute(0, 5, 1, 3, 2, 4).contiguous().view(-1, C, H, W)
    return x


class GraphAttention(nn.Module):
    r"""Window based multi-head self attention (W-MSA) module with relative position bias.
    It supports both of shifted and non-shifted window.
    Args:
        dim (int): Number of input channels.
        window_size (tuple[int]): The height and width of the window.
        num_heads (int): Number of attention heads.
        qkv_bias (bool, optional):  If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float | None, optional): Override default qk scale of head_dim ** -0.5 if set
        attn_drop (float, optional): Dropout ratio of attention weight. Default: 0.0
        proj_drop (float, optional): Dropout ratio of output. Default: 0.0
    """

    def __init__(
        self,
        dim,
        window_size,
        shift_size,
        global_size,
        num_heads,
        qkv_bias=True,
        qk_scale=None,
        attn_drop=0.0,
        proj_drop=0.0,
        use_pe=True,
        top_k=32,
        graph_type="mask",
    ):
        """
        Args:
        dim (int): Number of input channels.
        graph_type (str): Graph type, one of 'mask','gather' and 'triton'.
            'mask': apply masks to the attention maps. Benefit: easy implementation.
            'gather': gather tenors according to the results of KNN. Benefit: no redundant computation. Drawback: large memory footprint.
            'triton': triton implementation. Benefit: save a lot of GPU memory. Drawback: slow inference. Difficulty in back propagation.
        """
        super().__init__()
        self.dim = dim
        self.window_size = window_size  # Wh, Ww
        self.shift_size = shift_size
        self.global_size = global_size
        self.num_heads = num_heads
        self.head_dim = dim // num_heads
        self.scale = qk_scale or self.head_dim**-0.5
        self.use_pe = use_pe
        self.top_k = top_k
        self.graph_type = graph_type

        # mlp to generate continuous relative position bias
        if self.use_pe:
            # mlp to generate continuous relative position bias
            self.cpb_mlp = nn.Sequential(
                nn.Linear(2, 512, bias=True),
                nn.ReLU(inplace=True),
                nn.Linear(512, num_heads, bias=False),
            )
            table = get_relative_coords_table(self.window_size)
            index = get_relative_position_index(self.window_size)
            self.register_buffer("relative_coords_table", table)
            self.register_buffer("relative_position_index", index)

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)

        self.proj_drop = nn.Dropout(proj_drop)

        self.softmax = nn.Softmax(dim=-1)

    def forward(self, x, mask=None, measure=None):
        """
        Args:
            x: input features with shape of (B_=num_windows*B, N, C)
            mask: shifted window attention mask
            measure: (num_windows*B, H, N, k) or (num_windows*B, H, N, N)
        """
        B_, N, C = x.shape

        # qkv projection
        # [B_, N, C] -> [B_, N, 3C] -> [B_, N, 3, head, (C/head)] -> [3, B_, head, N, (C/head)]
        qkv = self.qkv(x).reshape(B_, N, 3, self.num_heads, -1).permute(2, 0, 3, 1, 4)
        # [B_, head, N, (C/head)]
        q, k, v = qkv[0], qkv[1], qkv[2]

        # attention map
        q = q * self.scale
        # [B_, head, N, N]
        attn = q @ k.transpose(-2, -1)

        # positional encoding
        if self.use_pe:
            bias_table = self.cpb_mlp(self.relative_coords_table)
            bias_table = bias_table.view(-1, self.num_heads)

            bias = bias_table[self.relative_position_index.view(-1)]
            bias = bias.view(N, N, -1).permute(2, 0, 1).contiguous()
            # nH, Wh*Ww, Wh*Ww
            bias = 16 * torch.sigmoid(bias)
            attn = attn + bias.unsqueeze(0)

        # dynamic attention mask
        # attn[measure] = float("-inf")
        attn[measure == False] = float("-inf")

        # shift attention mask
        # if mask is not None:
        #     nW = mask.shape[0]
        #     mask = mask.unsqueeze(1).unsqueeze(0)
        #     attn = attn.view(B_ // nW, nW, self.num_heads, N, N) + mask
        #     attn = attn.view(-1, self.num_heads, N, N)
        # Use two masks could cause nan values.

        # attention
        # print("1", torch.sum(attn))
        attn = self.softmax(attn)
        # print("2", measure[0, 0, 0, 0], measure.dtype, measure.device, measure.requires_grad, torch.sum(measure == False), measure.numel(), torch.sum(attn), prod(attn.shape[:3]))
        attn = self.attn_drop(attn)
        x = (attn @ v).transpose(1, 2).reshape(B_, N, C)

        # output projection
        x = self.proj(x)
        x = self.proj_drop(x)
        return x

    def extra_repr(self) -> str:
        return f"dim={self.dim}, window_size={self.window_size}, num_heads={self.num_heads}"

    def flops(self, N):
        # calculate flops for 1 window with token length of N
        flops = 0
        # qkv = self.qkv(x)
        flops += N * self.dim * 3 * self.dim
        # attn = (q @ k.transpose(-2, -1))
        flops += self.num_heads * N * (self.dim // self.num_heads) * N
        #  x = (attn @ v)
        flops += self.num_heads * N * N * (self.dim // self.num_heads)
        # x = self.proj(x)
        flops += N * self.dim * self.dim
        return flops


class GraphAttentionWrapperV2(GraphAttention):
    def __init__(self, **kwargs):
        super(GraphAttentionWrapperV2, self).__init__(**kwargs)

    def calc_sim_measure(self, x, K, return_mask=True):
        # x: B_, N, C
        with torch.no_grad():
            B_, N, C = x.shape
            x = x.view(B_, N, self.num_heads, C // self.num_heads).permute(
                0, 2, 1, 3
            )  # B_, num_head, N, head_dim
            similarity = torch.matmul(x, x.transpose(2, 3))
            topk_values, topk_indices = torch.topk(similarity, K, dim=-1)

        if return_mask:
            kthvalue = torch.min(topk_values, dim=-1, keepdim=True).values
            mask = similarity >= kthvalue
            # mask = mask.to(dtype)
            return mask
        else:
            return topk_indices

    def forward(self, x, mask=None, measure=None):
        """
        Args:
            x: B, C, H ,W
            mask: attention mask for shifted windows. This is calculated in the main network.
            measure: topk_values or mask. This is calculated by the first two blocks.
                It should also be returned and used by the following blocks
        """
        B, C, H, W = x.shape

        # window partition
        if self.shift_size == 0:
            x = window_partition(x, self.window_size)  # nW*B, wh * ww, C
        else:
            x = grid_partition(x, self.window_size[0], self.global_size)
        # establish the dynamic graph
        if measure is None:
            measure = self.calc_sim_measure(x, self.top_k)

        # window attention
        x = super(GraphAttentionWrapperV2, self).forward(
            x, mask, measure
        )  # nW*B, wh*ww, C

        # merge/reverse windows
        if self.shift_size == 0:
            x = window_reverse(x, self.window_size, (H, W))  # [B, C, H, W]
        else:
            x = grid_reverse(x, self.window_size[0], (B, C, H, W), self.global_size)

        return x, measure


class SemanticTransformerBlockV2(nn.Module):
    r"""Swin Transformer Block.
    Args:
        block_id (int): the block id
        dim (int): Number of input channels.
        input_resolution (tuple[int]): Input resulotion.
        num_heads (int): Number of attention heads.
        window_size (int): Window size.
        shift_size (int): Shift size for SW-MSA.
        mlp_ratio (float): Ratio of mlp hidden dim to embedding dim.
        qkv_bias (bool, optional): If True, add a learnable bias to query, key, value. Default: True
        qk_scale (float | None, optional): Override default qk scale of head_dim ** -0.5 if set.
        drop (float, optional): Dropout rate. Default: 0.0
        attn_drop (float, optional): Attention dropout rate. Default: 0.0
        drop_path (float, optional): Stochastic depth rate. Default: 0.0
        act_layer (nn.Module, optional): Activation layer. Default: nn.GELU
    """

    def __init__(
        self,
        dim,
        input_resolution,
        num_heads,
        window_size=8,
        shift_size=0,
        mlp_ratio=2.0,
        top_k=32,
        qkv_bias=True,
        qk_scale=None,
        drop=0.0,
        attn_drop=0.0,
        drop_path=0.0,
        act_layer=nn.GELU,
        use_pe=True,
        graph_type="mask",
        global_size=64,
        args=None,
    ):
        super().__init__()
        self.dim = dim
        self.input_resolution = input_resolution
        self.num_heads = num_heads
        self.window_size = window_size
        self.shift_size = shift_size
        self.global_size = global_size
        self.mlp_ratio = mlp_ratio
        self.top_k = top_k
        # if min(self.input_resolution) <= self.window_size:
        #     # if window size is larger than input resolution, we don't partition windows
        #     self.shift_size = 0
        #     self.window_size = min(self.input_resolution)
        # assert (
        #     0 <= self.shift_size < self.window_size
        # ), "shift_size must in 0-window_size"

        self.norm1 = LayerNorm(dim, 1)
        self.attn = GraphAttentionWrapperV2(
            dim=dim,
            window_size=to_2tuple(self.window_size),
            shift_size=self.shift_size,
            global_size=self.global_size,
            num_heads=num_heads,
            qkv_bias=qkv_bias,
            qk_scale=qk_scale,
            attn_drop=attn_drop,
            proj_drop=drop,
            use_pe=use_pe,
            top_k=top_k,
            graph_type=graph_type,
        )

        self.drop_path = DropPath(drop_path) if drop_path > 0.0 else nn.Identity()

        self.norm2 = LayerNorm(dim, 1)
        self.mlp = Mlp(
            in_features=dim,
            hidden_features=int(dim * mlp_ratio),
            act_layer=act_layer,
            drop=drop,
        )

    def forward(self, x, mask=None, measure=None):
        """
        Args:
            x: input tensor with shape of B, C, H, W
        Returns:
            output: tensor shape B, C, H, W
            measure: similarity measure used by the following blocks
        """

        # self-attention
        res, measure = self.attn(self.norm1(x), mask, measure)
        x = x + self.drop_path(res)

        # MLP
        x = x + self.drop_path(self.mlp(self.norm2(x)))

        return x, measure

    def extra_repr(self) -> str:
        return (
            f"dim={self.dim}, input_resolution={self.input_resolution}, num_heads={self.num_heads}, "
            f"window_size={self.window_size}, shift_size={self.shift_size}, mlp_ratio={self.mlp_ratio}"
        )

    def flops(self):
        flops = 0
        H, W = self.input_resolution
        # norm1
        flops += self.dim * H * W
        # W-MSA/SW-MSA
        nW = H * W / self.window_size / self.window_size
        flops += nW * self.attn.flops(self.window_size * self.window_size)
        # mlp
        flops += 2 * H * W * self.dim * self.dim * self.mlp_ratio
        # norm2
        flops += self.dim * H * W
        return flops
