# Copyright (2024) Tsinghua University, Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import argparse

import torch
from transformers import WhisperFeatureExtractor

from config import Config
from models.salmonn import SALMONN
from utils import prepare_one_sample

import pandas as pd
from collections import defaultdict

import evaluate
import os
import tqdm

parser = argparse.ArgumentParser()
parser.add_argument("--cfg-path", type=str, required=True, help='path to configuration file')
parser.add_argument("--device", type=str, default="cuda")
parser.add_argument(
    "--options",
    nargs="+",
    help="override some settings in the used config, the key-value pair "
    "in xxx=yyy format will be merged into config file (deprecate), "
    "change to --cfg-options instead.",
)


def main(path,output_test_path,promt):
    args = parser.parse_args()
    cfg = Config(args)

    model = SALMONN.from_config(cfg.config.model)
    model.to(args.device)
    model.eval()

    wav_processor = WhisperFeatureExtractor.from_pretrained(cfg.config.model.whisper_path)


    df = pd.read_csv(path)

    dict_output = {
        "gt_latex":[],
        "pr_latex":[],
        "pron":[],
        "audio_path":[]
    }

    for _,row in tqdm.tqdm(df.iterrows()):
        try:

            wav_path = row["audio_path"]
            gt_latex = row["latex"]
            pron = row["pronunciation"]
            
            samples = prepare_one_sample(wav_path, wav_processor)
            _prompt = [
                cfg.config.model.prompt_template.format("<Speech><SpeechHere></Speech> " + promt.strip())
            ]

            predict_latex = ""
            with torch.cuda.amp.autocast(dtype=torch.float16):
                predict_latex = model.generate(samples, cfg.config.generate, prompts=_prompt)[0].replace("</s>", "")
            
            print(f"\npredict_latex {predict_latex}")
            print(f"gt_latex      {gt_latex}")
            print(pron)
            print(wav_path,"\n")

            dict_output["gt_latex"].append(gt_latex)
            dict_output["pr_latex"].append(predict_latex)
            dict_output["pron"].append(pron)
            dict_output["audio_path"].append(wav_path)


        except Exception as e:
            print(e)

    pd.DataFrame(dict_output).to_excel(output_test_path,index=False)


if __name__ == "__main__":
    path = "/home/jovyan/Nikita/speech2latex/salmonn_exps/SALMONN/test/data/test_ru_eng_2000.csv"

    output_test_path = "/home/jovyan/Nikita/speech2latex/salmonn_exps/SALMONN/test/output/test_predicted_ru_eng.xlsx"
    prompt = "Recognize the speech and convert the content into text. Any mathematical expressions should be transcribed in LaTeX format."

    main(path, output_test_path, prompt)