#!/usr/bin/env python3
"""
Comparative Analysis of Job Posting Frequency Experiments
"""

import json
import matplotlib.pyplot as plt
import numpy as np
from pathlib import Path

def load_analysis_results(file_path):
    """Load analysis results from JSON file"""
    with open(file_path, 'r') as f:
        return json.load(f)

def generate_comparative_analysis():
    """Generate comprehensive comparative analysis"""
    
    # Load both experiments
    exp1_data = load_analysis_results('analysis_results/experiment_1/analysis_results.json')
    exp2_data = load_analysis_results('analysis_results/experiment_2/analysis_results.json')
    
    print("=== COMPARATIVE ANALYSIS: Job Posting Frequency Impact ===\n")
    
    print("Experiment Configuration:")
    print("- Experiment 1 (Less Frequent): 3-10 turn cooldown for job posting")
    print("- Experiment 2 (More Frequent): 2-7 turn cooldown for job posting")
    print()
    
    # Basic statistics comparison
    print("1. MARKET SCALE COMPARISON")
    print("-" * 40)
    exp1_overview = exp1_data['simulation_overview']
    exp2_overview = exp2_data['simulation_overview']
    
    scale_metrics = [
        ('Total Jobs Posted', exp1_overview['total_jobs'], exp2_overview['total_jobs']),
        ('Total Bids Submitted', exp1_overview['total_bids'], exp2_overview['total_bids']),
        ('Avg Bids per Job', exp1_overview['total_bids']/exp1_overview['total_jobs'], 
         exp2_overview['total_bids']/exp2_overview['total_jobs']),
        ('Avg Bids per Freelancer', exp1_data['persona_analysis']['avg_bids_per_freelancer'],
         exp2_data['persona_analysis']['avg_bids_per_freelancer'])
    ]
    
    for metric, exp1_val, exp2_val in scale_metrics:
        change = ((exp2_val - exp1_val) / exp1_val) * 100
        print(f"{metric}:")
        print(f"  Exp1 (Less Frequent): {exp1_val:.1f}")
        print(f"  Exp2 (More Frequent): {exp2_val:.1f}")
        print(f"  Change: {change:+.1f}%")
        print()
    
    # Market efficiency comparison
    print("2. MARKET EFFICIENCY COMPARISON")
    print("-" * 40)
    
    efficiency_metrics = [
        ('Job Fill Rate', exp1_data['market_efficiency']['avg_job_fill_rate'],
         exp2_data['market_efficiency']['avg_job_fill_rate']),
        ('Average Competition', exp1_data['market_efficiency']['avg_competition'],
         exp2_data['market_efficiency']['avg_competition']),
        ('Success Rate (bids to jobs)', exp1_data['persona_analysis']['success_rate'],
         exp2_data['persona_analysis']['success_rate'])
    ]
    
    for metric, exp1_val, exp2_val in efficiency_metrics:
        change = ((exp2_val - exp1_val) / exp1_val) * 100
        print(f"{metric}:")
        print(f"  Exp1: {exp1_val:.3f}")
        print(f"  Exp2: {exp2_val:.3f}")
        print(f"  Change: {change:+.1f}%")
        print()
    
    # Category performance comparison
    print("3. CATEGORY PERFORMANCE COMPARISON")
    print("-" * 40)
    
    categories = set(exp1_data['skill_analysis']['category_coverage'].keys()) & \
                set(exp2_data['skill_analysis']['category_coverage'].keys())
    
    print("Top categories by fill rate improvement (Exp2 vs Exp1):")
    category_improvements = []
    
    for category in categories:
        exp1_fill = exp1_data['skill_analysis']['category_coverage'][category]['fill_rate']
        exp2_fill = exp2_data['skill_analysis']['category_coverage'][category]['fill_rate']
        improvement = exp2_fill - exp1_fill
        category_improvements.append((category, improvement, exp1_fill, exp2_fill))
    
    # Sort by improvement
    category_improvements.sort(key=lambda x: x[1], reverse=True)
    
    for i, (category, improvement, exp1_fill, exp2_fill) in enumerate(category_improvements[:5]):
        print(f"{i+1}. {category}: {exp1_fill:.3f} → {exp2_fill:.3f} ({improvement:+.3f})")
    
    print("\nWorst performing categories (biggest decline):")
    for i, (category, improvement, exp1_fill, exp2_fill) in enumerate(category_improvements[-3:]):
        print(f"   {category}: {exp1_fill:.3f} → {exp2_fill:.3f} ({improvement:+.3f})")
    
    print()
    
    # Bidding intensity comparison
    print("4. BIDDING INTENSITY COMPARISON")
    print("-" * 40)
    
    print("Average bids per job by category:")
    category_bid_changes = []
    for category in categories:
        exp1_bids = exp1_data['skill_analysis']['category_coverage'][category]['avg_bids_per_job']
        exp2_bids = exp2_data['skill_analysis']['category_coverage'][category]['avg_bids_per_job']
        change = exp2_bids - exp1_bids
        category_bid_changes.append((category, change, exp1_bids, exp2_bids))
    
    category_bid_changes.sort(key=lambda x: x[1], reverse=True)
    
    print("Biggest increases in competition:")
    for i, (category, change, exp1_bids, exp2_bids) in enumerate(category_bid_changes[:3]):
        print(f"{i+1}. {category}: {exp1_bids:.1f} → {exp2_bids:.1f} ({change:+.1f})")
    
    print("\nBiggest decreases in competition:")
    for i, (category, change, exp1_bids, exp2_bids) in enumerate(category_bid_changes[-3:]):
        print(f"{i+1}. {category}: {exp1_bids:.1f} → {exp2_bids:.1f} ({change:+.1f})")
    
    print()
    
    # Rate analysis
    print("5. RATE IMPACT ANALYSIS")
    print("-" * 40)
    
    print("Premium categories (>$80/hr average bid rate):")
    for category in categories:
        exp1_rate = exp1_data['skill_analysis']['category_rates'][category]['avg_bid_rate']
        exp2_rate = exp2_data['skill_analysis']['category_rates'][category]['avg_bid_rate']
        
        if exp1_rate > 80 or exp2_rate > 80:
            rate_change = exp2_rate - exp1_rate
            print(f"  {category}: ${exp1_rate:.0f} → ${exp2_rate:.0f} ({rate_change:+.0f})")
    
    return {
        'scale_comparison': scale_metrics,
        'efficiency_comparison': efficiency_metrics,
        'category_improvements': category_improvements,
        'category_bid_changes': category_bid_changes
    }

def create_comparative_visualizations(analysis_data):
    """Create comparative visualizations"""
    
    # Load data again for visualization
    exp1_data = load_analysis_results('analysis_results/experiment_1/analysis_results.json')
    exp2_data = load_analysis_results('analysis_results/experiment_2/analysis_results.json')
    
    # Create figure with subplots
    fig, axes = plt.subplots(2, 2, figsize=(15, 12))
    fig.suptitle('Job Posting Frequency Impact: Comparative Analysis', fontsize=16, fontweight='bold')
    
    # 1. Market Scale Metrics
    ax1 = axes[0, 0]
    metrics = ['Jobs Posted', 'Total Bids', 'Avg Bids/Job', 'Avg Bids/Freelancer']
    exp1_vals = [
        exp1_data['simulation_overview']['total_jobs'],
        exp1_data['simulation_overview']['total_bids'],
        exp1_data['simulation_overview']['total_bids']/exp1_data['simulation_overview']['total_jobs'],
        exp1_data['persona_analysis']['avg_bids_per_freelancer']
    ]
    exp2_vals = [
        exp2_data['simulation_overview']['total_jobs'],
        exp2_data['simulation_overview']['total_bids'],
        exp2_data['simulation_overview']['total_bids']/exp2_data['simulation_overview']['total_jobs'],
        exp2_data['persona_analysis']['avg_bids_per_freelancer']
    ]
    
    x = np.arange(len(metrics))
    width = 0.35
    
    # Normalize values for comparison (different scales)
    norm_exp1 = [exp1_vals[0]/500, exp1_vals[1]/3000, exp1_vals[2], exp1_vals[3]/15]
    norm_exp2 = [exp2_vals[0]/500, exp2_vals[1]/3000, exp2_vals[2], exp2_vals[3]/15]
    
    ax1.bar(x - width/2, norm_exp1, width, label='Less Frequent (3-10 turn)', alpha=0.8)
    ax1.bar(x + width/2, norm_exp2, width, label='More Frequent (2-7 turn)', alpha=0.8)
    ax1.set_ylabel('Normalized Values')
    ax1.set_title('Market Scale Comparison')
    ax1.set_xticks(x)
    ax1.set_xticklabels(metrics, rotation=45, ha='right')
    ax1.legend()
    ax1.grid(True, alpha=0.3)
    
    # 2. Fill Rate by Category
    ax2 = axes[0, 1]
    categories = list(set(exp1_data['skill_analysis']['category_coverage'].keys()) & 
                     set(exp2_data['skill_analysis']['category_coverage'].keys()))
    categories = sorted(categories)[:8]  # Show top 8 for readability
    
    exp1_fills = [exp1_data['skill_analysis']['category_coverage'][cat]['fill_rate'] for cat in categories]
    exp2_fills = [exp2_data['skill_analysis']['category_coverage'][cat]['fill_rate'] for cat in categories]
    
    x = np.arange(len(categories))
    ax2.bar(x - width/2, exp1_fills, width, label='Less Frequent', alpha=0.8)
    ax2.bar(x + width/2, exp2_fills, width, label='More Frequent', alpha=0.8)
    ax2.set_ylabel('Fill Rate')
    ax2.set_title('Fill Rate by Category')
    ax2.set_xticks(x)
    ax2.set_xticklabels([cat.replace(' & ', '\n& ') for cat in categories], rotation=45, ha='right')
    ax2.legend()
    ax2.grid(True, alpha=0.3)
    
    # 3. Competition Intensity
    ax3 = axes[1, 0]
    exp1_competition = [exp1_data['skill_analysis']['category_coverage'][cat]['avg_bids_per_job'] for cat in categories]
    exp2_competition = [exp2_data['skill_analysis']['category_coverage'][cat]['avg_bids_per_job'] for cat in categories]
    
    ax3.bar(x - width/2, exp1_competition, width, label='Less Frequent', alpha=0.8)
    ax3.bar(x + width/2, exp2_competition, width, label='More Frequent', alpha=0.8)
    ax3.set_ylabel('Avg Bids per Job')
    ax3.set_title('Competition Intensity by Category')
    ax3.set_xticks(x)
    ax3.set_xticklabels([cat.replace(' & ', '\n& ') for cat in categories], rotation=45, ha='right')
    ax3.legend()
    ax3.grid(True, alpha=0.3)
    
    # 4. Market Efficiency Overview
    ax4 = axes[1, 1]
    efficiency_metrics = ['Fill Rate', 'Success Rate', 'Avg Competition']
    exp1_eff = [
        exp1_data['market_efficiency']['avg_job_fill_rate'],
        exp1_data['persona_analysis']['success_rate'],
        exp1_data['market_efficiency']['avg_competition'] / 10  # Scale down for comparison
    ]
    exp2_eff = [
        exp2_data['market_efficiency']['avg_job_fill_rate'],
        exp2_data['persona_analysis']['success_rate'],
        exp2_data['market_efficiency']['avg_competition'] / 10  # Scale down for comparison
    ]
    
    x = np.arange(len(efficiency_metrics))
    ax4.bar(x - width/2, exp1_eff, width, label='Less Frequent', alpha=0.8)
    ax4.bar(x + width/2, exp2_eff, width, label='More Frequent', alpha=0.8)
    ax4.set_ylabel('Normalized Values')
    ax4.set_title('Market Efficiency Metrics')
    ax4.set_xticks(x)
    ax4.set_xticklabels(efficiency_metrics)
    ax4.legend()
    ax4.grid(True, alpha=0.3)
    
    plt.tight_layout()
    
    # Save the plot
    Path('analysis_results').mkdir(exist_ok=True)
    plt.savefig('analysis_results/comparative_analysis_plots.png', dpi=300, bbox_inches='tight')
    print("✅ Comparative visualization saved to analysis_results/comparative_analysis_plots.png")
    
    plt.show()

if __name__ == "__main__":
    analysis = generate_comparative_analysis()
    create_comparative_visualizations(analysis)
