# ——————————————————————————————————————
# Online updating
# ——————————————————————————————————————

#   X_all: n-length vector
#   Y_all: n-length response vector
#   gamma0: initial value of learning rate
#   gamma_exp: gamma_i=gamma0*i^(-gamma_exp)
#   kernel: a function kernel(x, z)
#   eps_privacy, delta_privacy: parameters of LDP
#   loss_type: 'huber', 'L2'
#   privacy_type: 'DP', 'non' 

# ——————————————————————————————————————

#   PH-FSGD: loss_type='huber', privacy_type='DP'
#   H-FSGD: loss_type='huber', privacy_type='non'
#   L2-FSGD: loss_type='L2', privacy_type='non'

# ——————————————————————————————————————

online_updating <- function(X_all, Y_all, gamma0, gamma_exp, kernel, inner_product_H, grid, eps_privacy, delta_privacy, tau, n_drop, loss_type=c('huber', 'L2'), privacy_type=c('DP', 'non')) {
  n <- length(X_all)
  f_hat_pre <- rep(0, lent)
  f_hat_bar_pre <- rep(0, lent)
  record_id <- 1
  f_est_list <- list()
  
  if (privacy_type == "DP") {
    c_dp <- 8 * tau^2 * log(2 / delta_privacy) / (eps_privacy^2)
    chol_dp <- chol(c_dp * K_grid)
  }
  
  for (i in 1:n) {
    X <- X_all[i]
    Y <- Y_all[i]
    
    gamma_i <- gamma0 * i ^ (-gamma_exp)
    
    K_xn <- kernel(X, grid)
    ip_val <- inner_product_H(f_hat_pre, K_xn, Kinv_grid)
    resn <- Y - ip_val
    
    if (loss_type == 'huber') {
      cap    <- pmax(-tau, pmin(resn, tau))
      grad   <- cap * K_xn
    } else {
      grad <- resn * K_xn
    }
    
    if (privacy_type == "DP") {
      z <- rnorm(lent)
      noise_dp <- crossprod(chol_dp, z)
    } else {
      noise_dp <- numeric(lent)
    }
    
    f_hat_current <- f_hat_pre + gamma_i * (grad + noise_dp)
    f_hat_pre <- f_hat_current
    
    if (i > n_drop) {
      f_hat_bar_current <- ((i - n_drop - 1) * f_hat_bar_pre + f_hat_current) / (i - n_drop)
      f_hat_bar_pre <- f_hat_bar_current
    }
    
    if (i %in% record_size) {
      f_est_list[[record_id]] <- f_hat_bar_current
      record_id <- record_id + 1
    }
  }
    
  # return: f_hat_bar at record_id
  f_est_list
}


