#include "algs.h"

using namespace std;
using namespace mygraph;

resultsHandler allResults;
vector<double> alpha;
vector<bool> emptyBoolVector;
vector<bool> emptySetVector;
vector<double> emptyDoubleVector;
vector<size_t> emptySize_tVector;

uniform_real_distribution<double> unidist = uniform_real_distribution<double>(0, 1);

struct revgainLT revgainLTobj;
struct gainLT gainLTobj;

void init_alpha(tinyGraph &g)
{
   alpha.assign(g.n, 0.0);
   mt19937 gen(0); // same sequence each time
   #pragma omp parallel for
   for (node_id u = 0; u < g.n; ++u)
   {
      alpha[u] = unidist(gen);
   }
}

#ifdef IMG
signed long marge(size_t &nEvals, tinyGraph &g, node_id u, vector<bool> &set)
{
   if (set[u])
      return 0;
   ++nEvals;

   signed long m;
   double mx = 2 * g.getWeightedDegreeMinusSet(u, set);
   double my = g.getWeightedDegree(u);
   m = (mx - my);
   return m;
}


size_t compute_valSet(size_t &nEvals, tinyGraph &g, vector<bool> &set)
{
   ++nEvals;
   size_t val = 0;
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {
      vector<tinyEdge> &neis = g.adjList[u].neis;
      for (size_t j = 0; j < neis.size(); ++j)
      {
         node_id v = neis[j].target;
         if ((set[u] && !set[v]) || (!set[u] && set[v]))
            val += neis[j].weight;
      }
   }

   return val / 2;
}

double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &set_id)
{
   vector<bool> set(g.n, false);
   if(set_id.size()==0) return 0;
   #pragma omp parallel for
   for (size_t i = 0; i < set_id.size(); ++i)
   {
      set[set_id[i]] = true;
   }
   ++nEvals;
   double val = 0;
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {     
      double max=0;
      for(node_id v=0;v <g.n; v++)
      {
         if(set[u] && set[v])
         {
            val-=(double)g.getEdgeWeight(u,v)/(double)g.n;
         }
         if(set[u]==false && set[v]==true)
         {
            double tmp=g.getEdgeWeight(u,v);
            if(tmp>max)
               max=tmp;
         }
      }
      val+=max;
   }

   return val;
}
double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &set_id, vector<node_id> &set_id2)
{
   vector<bool> set(g.n, false);
   #pragma omp parallel for
   for (size_t i = 0; i < set_id.size(); ++i)
   {
      set[set_id[i]] = true;
   }
   #pragma omp parallel for
   for (size_t i = 0; i < set_id2.size(); ++i)
   {
      set[set_id2[i]] = true;
   }
   ++nEvals;
   double val = 0;
   
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {     
      double max=0;
      for(node_id v=0;v <g.n; v++)
      {
         if(set[u] && set[v])
         {
            val-=(double)g.getEdgeWeight(u,v)/(double)g.n;
         }
         if(set[u]==false && set[v]==true)
         {
            double tmp=g.getEdgeWeight(u,v);
            if(tmp>max)
               max=tmp;
         }
      }
      val+=max;
   }
   return val;
}
#endif

#ifdef REVMAX
signed long marge(size_t &nEvals, tinyGraph &g, node_id u, vector<bool> &set)
{

   if (set[u])
      return 0;

   ++nEvals;

   signed long m;
   double mx = 2 * g.getWeightedDegreeMinusSet(u, set);
   double my = g.getWeightedDegree(u);

   m = (mx - my);

   return m;
}

size_t compute_valSet(size_t &nEvals, tinyGraph &g, vector<bool> &set)
{
   ++nEvals;
   size_t val = 0;
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {
      vector<tinyEdge> &neis = g.adjList[u].neis;
      for (size_t j = 0; j < neis.size(); ++j)
      {
         node_id v = neis[j].target;
         if ((set[u] && !set[v]) || (!set[u] && set[v]))
            val += neis[j].weight;
      }
   }

   return val / 2;
}

double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &sset)
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> set(g.n, false);
   #pragma omp parallel for
   for (size_t i = 0; i < sset.size(); ++i)
   {
      set[sset[i]] = true;
   }

   ++nEvals;
   size_t val = 0;
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {
      vector<tinyEdge> &neis = g.adjList[u].neis;
      for (size_t j = 0; j < neis.size(); ++j)
      {
         node_id v = neis[j].target;
         if ((set[u] && !set[v]) || (!set[u] && set[v]))
            val += neis[j].weight;
      }
   }
   return val / 2;
}
double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &sset, vector<node_id> &sset2)
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> set(g.n, false);
   #pragma omp parallel for
   for (size_t i = 0; i < sset.size(); ++i)
   {
      set[sset[i]] = true;
   }
   #pragma omp parallel for
   for (size_t i = 0; i < sset2.size(); ++i)
   {
      set[sset2[i]] = true;
   }
   ++nEvals;
   size_t val = 0;
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {
      vector<tinyEdge> &neis = g.adjList[u].neis;
      for (size_t j = 0; j < neis.size(); ++j)
      {
         node_id v = neis[j].target;
         if ((set[u] && !set[v]) || (!set[u] && set[v]))
            val += neis[j].weight;
      }
   }
   return val / 2;
}
#endif

#ifdef MAXCUT
double compute_valSet(size_t &nEvals, tinyGraph &g, vector<bool> &set,
                      vector<bool> &cov)
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }

   ++nEvals;
   cov.assign(g.n, false);
   double val = 0;
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {
      if (!set[u])
      {
         vector<tinyEdge> &neis = g.adjList[u].neis;
         double valU = 0.0;
         for (size_t j = 0; j < neis.size(); ++j)
         {
            node_id v = neis[j].target;
            if (set[v])
            {
               valU += neis[j].weight;
            }
         }
         valU = pow(valU, alpha[u]);
         val += valU;
      }
   }

   return val;
}

double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &sset)
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> set(g.n, false);
   #pragma omp parallel for
   for (size_t i = 0; i < sset.size(); ++i)
   {
      set[sset[i]] = true;
   }

   ++nEvals;

   double val = 0;
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {
      if (!set[u])
      {
         vector<tinyEdge> &neis = g.adjList[u].neis;
         double valU = 0.0;
         for (size_t j = 0; j < neis.size(); ++j)
         {
            node_id v = neis[j].target;
            if (set[v])
            {
               valU += neis[j].weight;
            }
         }
         valU = pow(valU, alpha[u]);
         val += valU;
      }
   }

   return val;
}
double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &sset, vector<node_id> &sset2)
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> set(g.n, false);
   #pragma omp parallel for
   for (size_t i = 0; i < sset.size(); ++i)
   {
      set[sset[i]] = true;
   }
   #pragma omp parallel for
   for (size_t i = 0; i < sset2.size(); ++i)
   {
      set[sset2[i]] = true;
   }
   ++nEvals;

   double val = 0;
   #pragma omp parallel for reduction(+:val)
   for (node_id u = 0; u < g.n; ++u)
   {
      if (!set[u])
      {
         vector<tinyEdge> &neis = g.adjList[u].neis;
         double valU = 0.0;
         for (size_t j = 0; j < neis.size(); ++j)
         {
            node_id v = neis[j].target;
            if (set[v])
            {
               valU += neis[j].weight;
            }
         }
         valU = pow(valU, alpha[u]);
         val += valU;
      }
   }

   return val;
}
double marge(size_t &nEvals, tinyGraph &g, node_id x, vector<bool> &set,
             vector<bool> &cov)
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   if (set[x])
      return 0;

   double loss = 0.0;

   vector<tinyEdge> &neis = g.adjList[x].neis;
   double valX = 0.0;
   #pragma omp parallel for reduction(+:valX)
   for (size_t j = 0; j < neis.size(); ++j)
   {
      node_id v = neis[j].target;
      if (set[v])
      {
         valX += neis[j].weight;
      }
   }

   valX = pow(valX, alpha[x]);

   loss = valX;

   double gain = 0.0;
   #pragma omp parallel for reduction(+:gain)
   for (size_t j = 0; j < neis.size(); ++j)
   {
      node_id v = neis[j].target;
      vector<tinyEdge> &neisV = g.adjList[v].neis;
      double valV = 0.0;
      double valVwithX = 0.0;
      for (size_t k = 0; k < neisV.size(); ++k)
      {
         node_id w = neisV[k].target;
         if (w != x)
         {
            if (set[w])
            {
               valV += neisV[k].weight;
               valVwithX += neisV[k].weight;
            }
         }
         else
         {
            valVwithX += neisV[k].weight;
         }
      }

      gain += pow(valVwithX, alpha[v]) - pow(valV, alpha[v]);
   }

   ++nEvals;
   return gain - loss;
}
#endif

void eval_multi_worker(vector<double> &x, tinyGraph &g, double &val, size_t nsamps, mt19937 &gen)
{
   size_t tmpEvals = 0;

   val = 0.0;
   vector<bool> set(g.n, false);
   for (size_t i = 0; i < nsamps; ++i)
   {
      for (size_t j = 0; j < g.n; ++j)
      {
         set[j] = false;
         if (unidist(gen) < x[j])
         {
            set[j] = true;
         }
      }

      val += compute_valSet(tmpEvals, g, set);
   }
}

double eval_multilinear_base(size_t &nEvals, tinyGraph &g, vector<double> &x, size_t nsamps, size_t nThreads, vector<mt19937> &vgens)
{

   thread *wThreads = new thread[nThreads];
   vector<double> vals(nThreads);
   size_t nSampsPerThread = nsamps / nThreads + 1;
   size_t totSamps = nSampsPerThread * nThreads;
   double val = 0.0;

   for (size_t i = 0; i < nThreads; ++i)
   {
      wThreads[i] = thread(eval_multi_worker, ref(x), ref(g), ref(vals[i]), nSampsPerThread, ref(vgens[i]));
   }
   for (size_t i = 0; i < nThreads; ++i)
   {
      wThreads[i].join();
      val += vals[i];
   }
   delete[] wThreads;

   val /= totSamps;
   nEvals += totSamps;
   return val;
}

void reportResults(size_t nEvals, double obj, size_t rounds)
{
   allResults.add("obj", obj);
   allResults.add("nEvals", nEvals);
   allResults.add("rounds", rounds);
}
void reportResults(size_t nEvals, double obj, size_t rounds,size_t nEvals_Adapt)
{
   allResults.add("obj", obj);
   allResults.add("nEvals", nEvals);
   allResults.add("rounds", rounds);
   allResults.add("nEvals_Adapt", nEvals_Adapt);
}
void filter(size_t &nEvals,
            tinyGraph &g,
            vector<bool> &A,
            vector<bool> &S,
            double tau,
            vector<size_t> &idsA)
{
   vector<size_t> newIdsA;
   for (size_t i = 0; i < idsA.size(); ++i)
   {
      size_t x = idsA[i];
      if (S[x])
      {
         // filter x
         A[x] = false;
      }
      else
      {
         if (marge(nEvals, g, x, S) < tau)
         {
            // filter x
            A[x] = false;
         }
         else
         {
            // keep x
            newIdsA.push_back(x);
         }
      }
   }
   idsA.swap(newIdsA);
}

void random_set(tinyGraph &g, vector<bool> &C, vector<size_t> &A)
{
   C.assign(g.n, false);
   uniform_int_distribution<size_t> dist(0, A.size() - 1);
   size_t size_of_c = dist(gen);
   double prob = 1.0 / size_of_c;

   for (size_t i = 0; i < A.size(); ++i)
   {
      if (unidist(gen) < prob)
      {
         C[A[i]] = true;
      }
   }
}

void random_set(tinyGraph &g, vector<size_t> &C, vector<size_t> &A)
{
   C.clear();
   uniform_int_distribution<size_t> dist(0, A.size() - 1);
   size_t size_of_c = dist(gen);
   double prob = 1.0 / size_of_c;

   for (size_t i = 0; i < A.size(); ++i)
   {
      if (unidist(gen) < prob)
      {
         C.push_back(A[i]);
      }
   }
}

void unc_max(size_t &nEvals, tinyGraph &g, vector<size_t> &A, double epsi, double delta, vector<size_t> &result)
{
   result.clear();
   size_t ell = (log(1 / delta)) / (log(1 + (4.0 / 3) * epsi)) + 1;
   vector<bool> tmp;
   vector<bool> max;
   double tmpVal = 0;
   double maxVal = 0;
   for (size_t i = 0; i < ell; ++i)
   {
      random_set(g, tmp, A);
      tmpVal = compute_valSet(nEvals, g, tmp);
      if (tmpVal >= maxVal)
      {
         max = tmp;
         maxVal = tmpVal;
      }
   }

   for (size_t i = 0; i < g.n; ++i)
   {
      if (max[i])
         result.push_back(i);
   }
}

void unc_max(size_t &nEvals, tinyGraph &g, vector<size_t> &A, double epsi, double delta, vector<bool> &result)
{
   result.assign(g.n, false);
   size_t ell = (log(1 / delta)) / (log(1 + (4.0 / 3) * epsi)) + 1;
   vector<bool> tmp;
   double tmpVal = 0;
   double maxVal = 0;
   for (size_t i = 0; i < ell; ++i)
   {
      random_set(g, tmp, A);
      tmpVal = compute_valSet(nEvals, g, tmp);
      if (tmpVal >= maxVal)
      {
         result = tmp;
         maxVal = tmpVal;
      }
   }
}

void sampleUt(vector<size_t> &R, vector<size_t> &A, size_t t)
{
   // g.logg << "Starting sampleUX..." << endL;
   R.clear();
   uniform_int_distribution<size_t> dist(0, A.size() - 1);
   vector<bool> alreadySampled(A.size(), false);

   for (size_t i = 0; i < t; ++i)
   {
      size_t pos;
      do
      {
         pos = dist(gen);
      } while ((alreadySampled[pos]));

      alreadySampled[pos] = true;
      R.push_back(A[pos]);
   }
}

unsigned samp_Dt(size_t &nEvals,
                 tinyGraph &g,
                 vector<size_t> &idsA,
                 vector<size_t> &idsS,
                 size_t t,
                 double tau)
{
   vector<size_t> idsT;
   vector<bool> T(g.n, false);
   uniform_int_distribution<size_t> dist(0, idsA.size() - 1);
   while (idsT.size() < t - 1)
   {
      size_t pos;
      pos = dist(gen);
      if (!T[idsA[pos]])
      {
         T[idsA[pos]] = true;
         idsT.push_back(idsA[pos]);
      }
   }

   size_t x = 0;
   do
   {
      size_t pos = dist(gen);
      x = idsA[pos];
   } while (T[x]);

   vector<bool> ScupT(g.n, false);

   for (size_t i = 0; i < idsS.size(); ++i)
   {
      ScupT[idsS[i]] = true;
   }
   for (size_t i = 0; i < idsT.size(); ++i)
   {
      ScupT[idsT[i]] = true;
   }

   if (marge(nEvals, g, x, ScupT) >= tau)
   {

      return 1;
   }

   return 0;
}

double reduced_mean_dagum(size_t &nEvals,
                          tinyGraph &g,
                          vector<size_t> &idsA,
                          vector<size_t> &idsS,
                          size_t t,
                          double tau,
                          double epsi,
                          double delta,
                          bool fast)
{
   // g.logg << DEBUG << "RMD: Chernoff requires: " << 16 * (log(2 / delta) / epsi / epsi + 1) << endL;
   // g.logg << DEBUG << "t: " << t << endL;

   // double lambda = exp(1) - 2.0;
   // double upsilon = 4.0 * lambda * log(2.0 / delta) / pow(epsi, 2);
   double upsilon_1;
   if (!fast)
      // upsilon_1 = 1.0 + (1.0 + epsi) * upsilon;
      upsilon_1 = 16 * ceil(log(2 / delta) / pow(epsi, 2));
   else
      upsilon_1 = 100;

   size_t sum = upsilon_1;
   size_t nSamps = 0;
   size_t goal = upsilon_1;

   while (nSamps < goal)
   {
      nSamps += 1;
      sum -= 1 - samp_Dt(nEvals, g, idsA, idsS, t, tau);

      if (fast && static_cast<double>(sum) / goal < (1.0 - 1.5 * epsi))
      {
         break;
      }
   }

   // g.logg << DEBUG << "Dagum required: " << goal << endL;
   // g.logg << DEBUG << "Dagum returning: " << static_cast<double>(sum) / goal << endL;
   return static_cast<double>(sum) / goal;
}

double reduced_mean_chernoff(size_t &nEvals,
                             tinyGraph &g,
                             vector<size_t> &idsA,
                             vector<size_t> &idsS,
                             size_t t,
                             double tau,
                             double epsi,
                             double delta,
                             bool fast)
{

   g.logg << DEBUG << "Chernoff requires: " << 16 * (log(2 / delta) / epsi / epsi + 1) << endL;
   size_t ell = 0;
   if (fast)
      ell = 100;
   else
      ell = 16 * (log(2 / delta) / epsi / epsi + 1);

   double ubar = 0;
   for (size_t l = 0; l < ell; ++l)
   {
      ubar += samp_Dt(nEvals, g, idsA, idsS, t, tau);
   }
   ubar /= ell;

   return ubar;
}

void report_rounds(vector<bool> &S,
                   vector<double> &valRounds,
                   tinyGraph &g)
{
   // size_t tmp = 0;
   // if (valRounds.size() == 0) {
   //    valRounds.push_back( compute_valSet( tmp, g, S ) );
   //    return;
   // }

   // size_t tmpVal = valRounds[ valRounds.size() - 1 ];
   // size_t tmpVal2 = compute_valSet( tmp, g, S );
   // if (tmpVal2 > tmpVal) {
   //    valRounds.push_back( tmpVal2 );
   // } else {
   //    valRounds.push_back( tmpVal );
   // }
   valRounds.push_back(0);
}

void threshold_sample(size_t &nEvals,
                      tinyGraph &g,
                      vector<bool> &S,       // sampled set, may not be empty
                      vector<size_t> &idsS,  // index of sampled set
                      vector<bool> &exclude, // exclude set
                      bool bexcl,            // if exclude set
                      size_t k,
                      double tau,
                      double epsi,
                      double delta,
                      bool fast,
                      bool reportRounds,
                      vector<double> &valRounds)
{
   g.logg << DEBUG << "ThreSam: k = " << k << ", initial size of S = " << idsS.size() << endL;
   double hatepsi;
   if (!fast)
   {
      hatepsi = epsi / 3;
   }
   else
   {
      hatepsi = epsi;
   }
   // g.logg << DEBUG << "hatepsi = " << hatepsi << endL;
   size_t r = log(2 * g.n / delta) / log(1 / (1 - hatepsi)) + 1;
   size_t m = log(k) / hatepsi + 1;
   double hatdelta = delta / (2 * r * (m + 1));
   size_t adaRounds = 0;

   vector<bool> A(g.n, true); // candidate set
   vector<size_t> idsA;       // index of candidate set
   // vector< size_t > idsS;

   vector<size_t> idsT;
   for (size_t i = 0; i < g.n; ++i)
   {
      if (bexcl)
      {
         if (!exclude[i])
            idsA.push_back(i);
         else
            A[i] = false;
      }
      else
         idsA.push_back(i);
   }

   for (unsigned j = 0; j < r; ++j)
   {
      filter(nEvals, g, A, S, tau, idsA);
      // nothing added during filter
      if (reportRounds)
      {
         report_rounds(S, valRounds, g);
         ++adaRounds;
      }
      g.logg << DEBUG << "-------ThreSam: " << j << "th iteration-------" << endL;
      g.logg << DEBUG << "ThreSam: size of V: " << idsA.size() << endL;
      //	 g.logg << DEBUG << "(1 - hatepsi): " << 1 - hatepsi << endL;
      //  g.logg << DEBUG << "log n = " << log( g.n ) << endL;

      if (idsA.size() == 0)
      {
         return;
      }
      else
      {
         if (idsA.size() == 1)
         {

            S[idsA[0]] = true;
            idsS.push_back(idsA[0]);
            return;
         }
      }

      unsigned t = 0;
      unsigned oldT = 0;

      if (reportRounds)
      {
         report_rounds(S, valRounds, g);
         ++adaRounds;
      }
      for (unsigned i = 0; i < m; ++i)
      {
         if (oldT == idsA.size())
         {
            break;
         }
         t = (unsigned)pow(1 + hatepsi, i);
         if (t > k - idsS.size())
         {
            g.logg << DEBUG << "prefix being checked larger than k - |S|" << endL;
            break;
         }
         else if (t > idsA.size())
         {
            t = idsA.size();
         }

         if (t == oldT)
         {
            continue;
         }

         double ubar = 0;

         ubar = reduced_mean_dagum(nEvals,
                                   g,
                                   idsA,
                                   idsS,
                                   t,
                                   tau,
                                   hatepsi,
                                   hatdelta,
                                   fast);

         if (ubar <= 1 - 1.5 * hatepsi)
            break;

         if (t == idsA.size())
         {
            // all possible elements are good in expectation
            // can break
            break;
         }

         oldT = t;
      }

      if (t > k - idsS.size())
         t = k - idsS.size();

      sampleUt(idsT, idsA, t);
      g.logg << DEBUG << "ThreSam: prefix = : " << t;
      for (size_t i = 0; i < t; ++i)
      {
         if (!S[idsT[i]])
         {
            S[idsT[i]] = true;
            idsS.push_back(idsT[i]);
         }
      }
      g.logg << DEBUG << ", size of solution = " << idsS.size() << endL;

      if (idsS.size() >= k)
      {
         return;
      }
   }
}

void threshold_seq(size_t &nEvals,
                   tinyGraph &g,
                   vector<bool> &A,           // sampled set A, may not be empty
                   vector<bool> &Aprime,      // sampled set A', may not be empty
                   vector<size_t> &idsA,      // index of sampled set A
                   vector<size_t> &idsAprime, // index of sampled set A'
                   vector<bool> &exclude,     // exclude set
                   bool bexcl,                // if exclude set
                   size_t k,
                   double tau,
                   double epsi,
                   double delta,
                   bool fast,
                   bool reportRounds,
                   vector<double> &valRounds)
{
   if(idsA.size() == k) return;
   size_t l = 4.0 * (2 / epsi * log(g.n) + log(g.n / delta));
   size_t adaRounds = 0;
   vector<bool> V(g.n, true); // candidate set
   vector<size_t> idsV;       // index of candidate set
   size_t s;                  // prefix of candidate to be checked
   size_t iStar;
   size_t numTrue;
   signed long tempMarge;
   vector<bool> AT;
   vector<bool> B;
   //g.logg<<"ThreSeq: k = "<<k<<", epsi = "<<epsi<<", delta = "<<delta<<endL;
   //g.logg<< "k = " << k << ", initial size of A = " << idsA.size() << endL;

   for (size_t i = 0; i < g.n; ++i)
   { // update candidate set if need to exclude any set
      if (bexcl)
      {
         if (!exclude[i])
            idsV.push_back(i);
         else
            V[i] = false;
      }
      else
         idsV.push_back(i);
   }

   for (unsigned j = 0; j < l; ++j)
   { // outer for loop
      filter(nEvals, g, V, A, tau, idsV);
      //g.logg<< "-------ThreSeq: " << j << "th iteration-------" << endL;
      //g.logg<< "ThreSeq: size of V: " << idsV.size() << endL;

      if (reportRounds)
      {
         //g.logg<< "1"<<endL;
         report_rounds(A, valRounds, g);
         ++adaRounds;
         //g.logg<< "2"<<endL;
      }
      //g.logg<< "Current size of A: " << idsA.size() << ", ";

      if (idsV.size() == 0)
      {
         return;
      }
      else
      {
         random_shuffle(idsV.begin(), idsV.end());
         if (idsV.size() < k - idsA.size())
         {
            s = idsV.size();
         }
         else
         {
            s = k - idsA.size();
         }
         // g.logg << DEBUG << "s = " << idsA.size() << ", ";
         iStar = 0;
         numTrue = 0;
         B.assign(s, true);
         AT = A;

         for (unsigned i = 0; i < s; ++i)
         {
            tempMarge = marge(nEvals, g, idsV[i], AT);
            // g.logg << DEBUG << "marginal gain: " << tempMarge << endL;
            if (tempMarge > tau)
            {
               ++numTrue;
            }
            else if (tempMarge < 0)
            {
               B[i] = false;
            }

            if (numTrue >= (1 - epsi) * (i + 1))
            {
               iStar = i;
            }
            AT[idsV[i]] = true;
         }
         //g.logg << "ThreSam: prefix = " << iStar;
         if (reportRounds)
         {
            report_rounds(A, valRounds, g);
            ++adaRounds;
         }

         for (size_t i = 0; i <= iStar; i++)
         {
            A[idsV[i]] = true;
            idsA.push_back(idsV[i]);
            if (B[i])
            {
               Aprime[idsV[i]] = true;
               idsAprime.push_back(idsV[i]);
            }
         }
         //g.logg << ", size of solution = " << idsA.size() << endL;

         if (idsA.size() == k)
         {
            return;
         }
      }
   }
}

void threshold_seq_ama(size_t &nEvals,
                       tinyGraph &g,
                       vector<bool> &S,       // sampled set
                       vector<size_t> &idsS,  // index of sampled set
                       vector<bool> &exclude, // exclude set
                       bool bexcl,            // if exclude set
                       size_t k,
                       double tau,
                       double epsi,
                       double delta,
                       bool fast,
                       bool reportRounds,
                       vector<double> &valRounds)
{
   int ctr = 0;
   vector<bool> V(g.n, true); // candidate set
   vector<size_t> idsV;       // index of candidate set
   int l = ceil(1 / pow(epsi, 2));
   // g.logg << DEBUG << "l = " << l << endL;

   size_t adaRounds = 0;

   for (size_t i = 0; i < g.n; ++i)
   {
      if (bexcl)
      {
         if (!exclude[i])
            idsV.push_back(i);
         else
            V[i] = false;
      }
      else
         idsV.push_back(i);
   }
   // g.logg << DEBUG << "k = " << k << endL;
   // g.logg << DEBUG << "Size of V before filter: " << idsV.size() << endL;
   filter(nEvals, g, V, S, tau, idsV);
   // g.logg << DEBUG << "Size of V after filter: " << idsV.size() << endL;
   // nothing added during filter
   if (reportRounds)
   {
      report_rounds(S, valRounds, g);
      ++adaRounds;
   }

   vector<bool> good_element(g.n, false);
   vector<size_t> good_element_ids;
   vector<size_t> good_element_ids_star;
   vector<signed long> good_element_val;
   vector<signed long> bad_element_val;
   signed long tempMarge;

   vector<bool> A;
   vector<size_t> idsA;
   unsigned kStar;
   int j = -1;
   // g.logg << DEBUG << "Post filter size of V: " << idsV.size() << endL;
   // g.logg << DEBUG << "Post filter size of S: " << idsS.size() << endL;
   while (idsV.size() > 0 && ctr < l)
   {
      ++j;
      g.logg << DEBUG << "-------AmaThreSeq: " << j << "th iteration-------" << endL;
      g.logg << DEBUG << "AmaThreSeq: size of V: " << idsV.size() << endL;

      random_shuffle(idsV.begin(), idsV.end());
      kStar = 0;
      good_element_ids_star.clear();
      A = S;
      idsA = idsS;
      if (reportRounds)
      {
         report_rounds(S, valRounds, g);
         ++adaRounds;
      }
      // g.logg << DEBUG << "Length of sampled seq: " << min(k, idsV.size()) << endL;
      for (unsigned t = 0; t < min(k - idsS.size(), idsV.size()); ++t)
      {
         A[idsV[t]] = true;
         idsA.push_back(idsV[t]);
         good_element_ids.clear();
         good_element_val.clear();
         bad_element_val.clear();
         for (unsigned i = t + 1; i < idsV.size(); ++i)
         {
            tempMarge = marge(nEvals, g, idsV[i], A);
            // g.logg << DEBUG << "marginal gain: " << tempMarge << endL;
            if (tempMarge >= tau)
            {
               good_element[idsV[i]] = true;
               good_element_ids.push_back(idsV[i]);
               good_element_val.push_back(tempMarge);
            }
            else if (tempMarge < 0)
            {
               bad_element_val.push_back(tempMarge);
            }
         }
         if (kStar == 0)
         {
            if (good_element_ids.size() <= (1 - epsi) * idsV.size())
            {
               good_element_ids_star = good_element_ids;
               kStar = t;
            }
            if (epsi * accumulate(good_element_val.begin(), good_element_val.end(), 0) <= -accumulate(bad_element_val.begin(), bad_element_val.end(), 0))
            {
               good_element_ids_star = good_element_ids;
               kStar = t;
               ctr++;
            }
         }
      }

      g.logg << DEBUG << "AmaThreSeq: prefix = " << kStar;
      // g.logg << DEBUG << "ctr = " << ctr << endL;

      for (unsigned t = 0; t <= kStar; ++t)
      {
         idsS.push_back(idsV[t]);
         S[idsV[t]] = true;
      }
      g.logg << DEBUG << ", size of solution = " << idsS.size() << endL;
      // g.logg << DEBUG << "ThreSeq current Val: " << compute_valSet(nEvals, g, S) << endL;

      V.assign(g.n, false);
      idsV.clear();
      for (unsigned t = 0; t < good_element_ids_star.size(); ++t)
      {
         idsV.push_back(good_element_ids_star[t]);
         V[good_element_ids_star[t]] = true;
      }
      // g.logg << DEBUG << "size of V: " << idsV.size() << endL;

      if (idsS.size() == k)
      {
         return;
      }
   }
}

void threshold_seq_ama2(size_t &nEvals,
                        tinyGraph &g,
                        vector<bool> &S,       // sampled set
                        vector<size_t> &idsS,  // index of sampled set
                        vector<bool> &exclude, // exclude set
                        bool bexcl,            // if exclude set
                        size_t k,
                        double tau,
                        double epsi,
                        double delta,
                        bool fast,
                        bool reportRounds,
                        vector<double> &valRounds)
{
   g.logg << DEBUG << "Size of S at beginning of TS: " << idsS.size() << endL;
   int ctr = 0;
   vector<bool> V(g.n, true); // candidate set
   vector<size_t> idsV;       // index of candidate set
   int l = ceil(1 / pow(epsi, 2));
   // g.logg << DEBUG << "l = " << l << endL;

   size_t adaRounds = 0;

   for (size_t i = 0; i < g.n; ++i)
   {
      if (bexcl)
      {
         if (!exclude[i])
            idsV.push_back(i);
         else
            V[i] = false;
      }
      else
         idsV.push_back(i);
   }
   g.logg << DEBUG << "k = " << k << endL;
   g.logg << DEBUG << "Size of V before filter: " << idsV.size() << endL;
   filter(nEvals, g, V, S, tau, idsV);

   if (reportRounds)
   {
      report_rounds(S, valRounds, g);
      ++adaRounds;
   }
   g.logg << DEBUG << "Size of V after filter: " << idsV.size() << endL;
   // nothing added during filter

   vector<bool> good_element(g.n, false);
   vector<size_t> good_element_ids;
   vector<size_t> good_element_ids_star;
   vector<signed long> good_element_val;
   vector<signed long> bad_element_val;
   signed long tempMarge;

   vector<bool> A;
   vector<size_t> idsA;
   bool ctr_plus; // if increase str
   unsigned t;    // index of prefix to be checked
   unsigned t_start;
   unsigned t_end;
   // g.logg << DEBUG << "Post filter size of V: " << idsV.size() << endL;
   // g.logg << DEBUG << "Post filter size of S: " << idsS.size() << endL;
   while (idsV.size() > 0 && ctr < l)
   {

      // g.logg << DEBUG << "Post filter size of V: " << idsV.size() << endL;
      random_shuffle(idsV.begin(), idsV.end());
      good_element_ids_star.clear();
      ctr_plus = false;
      t_start = 0;
      t_end = min(k - idsS.size(), idsV.size()); // 4
      // g.logg << DEBUG << "Length of sampled seq: " << min(k, idsV.size()) << endL;

      while (t_start < t_end)
      {
         if (reportRounds)
         {
            report_rounds(S, valRounds, g);
            ++adaRounds;
         }
         t = floor((t_start + t_end) / 2);
         g.logg << DEBUG << "start = " << t_start << ", end = " << t_end << ", current t = " << t << endL;
         A = S;
         idsA = idsS;
         for (unsigned j = 0; j <= t; j++)
         {
            A[idsV[j]] = true;
            idsA.push_back(idsV[j]);
         }
         good_element_ids.clear();
         good_element_val.clear();
         bad_element_val.clear();
         for (unsigned i = t + 1; i < idsV.size(); ++i)
         {
            tempMarge = marge(nEvals, g, idsV[i], A);
            // g.logg << DEBUG << "marginal gain: " << tempMarge << endL;
            if (tempMarge >= tau)
            {
               good_element[idsV[i]] = true;
               good_element_ids.push_back(idsV[i]);
               good_element_val.push_back(tempMarge);
            }
            else if (tempMarge < 0)
            {
               bad_element_val.push_back(tempMarge);
            }
         }
         g.logg << DEBUG << "Condition one is " << (good_element_ids.size() <= (1 - epsi) * idsV.size())
                << ", condition two is " << (epsi * accumulate(good_element_val.begin(), good_element_val.end(), 0) <= -accumulate(bad_element_val.begin(), bad_element_val.end(), 0)) << endL;
         if (good_element_ids.size() <= (1 - epsi) * idsV.size())
         {
            t_end = t;
            ctr_plus = false;
            good_element_ids_star = good_element_ids;
         }
         else if (epsi * accumulate(good_element_val.begin(), good_element_val.end(), 0) <= -accumulate(bad_element_val.begin(), bad_element_val.end(), 0))
         {
            t_end = t;
            ctr_plus = true;
            good_element_ids_star = good_element_ids;
         }
         else
         {
            t_start = t + 1;
         }
      }

      if (ctr_plus)
      {
         ++ctr;
      }
      if (t_start == min(k - idsS.size(), idsV.size()))
      {
         t = t_start - 1;
      }
      else
      {
         t = t_start;
      }

      g.logg << DEBUG << "prefix being selected is " << t << endL;
      // g.logg << DEBUG << "ctr = " << ctr << endL;

      for (unsigned i = 0; i <= t; ++i)
      {
         idsS.push_back(idsV[i]);
         S[idsV[i]] = true;
      }
      g.logg << DEBUG << "size of S: " << idsS.size() << endL;
      // g.logg << DEBUG << "ThreSeq current Val: " << compute_valSet(nEvals, g, S) << endL;

      V.assign(g.n, false);
      idsV.clear();
      for (unsigned i = 0; i < good_element_ids_star.size(); ++i)
      {
         idsV.push_back(good_element_ids_star[i]);
         V[good_element_ids_star[i]] = true;
      }
      // g.logg << DEBUG << "size of V: " << idsV.size() << endL;

      if (idsS.size() == k)
      {
         return;
      }
   }
}

double Ig::leastBenefit(node_id u, vector<bool> &set)
{
   set[u] = false;
   double m = marge(nEvals, g, u, set);
   set[u] = true;
   return m;
}

void Ig::run()
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> A(g.n, false);
   vector<bool> B(g.n, false);
   vector<bool> C(g.n, false);
   vector<bool> D(g.n, false);
   vector<bool> E(g.n, false);

   double valA = 0;
   double valB = 0;
   double valD;
   double valE;
   int rounds=0;
   node_id maxSingle;
   for (size_t i = 0; i < k; ++i)
   {
      size_t maxAid = 0;
      double maxMargeA = 0;
      rounds++;
      #pragma omp parallel for
      for (node_id u = 0; u < g.n; ++u)
      {
         if (!(B[u] || A[u]))
         {
            if (marge(nEvals, g, u, A) >= maxMargeA)
            {
               #pragma omp critical
               {
                  maxMargeA = marge(nEvals, g, u, A);
                  maxAid = u;
               }
            }
         }
      }

      if (maxMargeA > 0)
      {
         A[maxAid] = true;

         valA += maxMargeA;
      }

      if (i == 0)
         maxSingle = maxAid;

      size_t maxBid = 0;
      double maxMargeB = 0;
      rounds++;
      #pragma omp parallel for
      for (node_id u = 0; u < g.n; ++u)
      {
         if (!(B[u] || A[u]))
         {
            if (marge(nEvals, g, u, B) >= maxMargeB)
            {
               #pragma omp critical
               {
                  maxMargeB = marge(nEvals, g, u, B);
                  maxBid = u;
               }
            }
         }
      }

      if (maxMargeB > 0)
      {
         B[maxBid] = true;
         valB += maxMargeB;
      }
   }

   g.logg << "IG: First interlacing complete." << endL;

   // Begin second interlacing
   g.logg << "IG: Adding maxSingle to D,E: " << maxSingle << endL;
   valD = marge(nEvals, g, maxSingle, D);
   valE = valD;
   D[maxSingle] = true;
   E[maxSingle] = true;
   for (size_t i = 0; i < k - 1; ++i)
   {
      size_t maxDid = 0;
      long maxMargeD = 0;
      rounds++;
      #pragma omp parallel for
      for (node_id u = 0; u < g.n; ++u)
      {
         if (!(D[u] || E[u]))
         {
            if (marge(nEvals, g, u, D) > maxMargeD)
            {
               #pragma omp critical
               {
                  maxMargeD = marge(nEvals, g, u, D);
                  maxDid = u;
               }
            }
         }
      }

      if (maxMargeD > 0)
      {
         D[maxDid] = true;
         valD += maxMargeD;
      }

      size_t maxEid = 0;
      double maxMargeE = 0;
      rounds++;
      #pragma omp parallel for
      for (node_id u = 0; u < g.n; ++u)
      {
         if (!(E[u] || D[u]))
         {
            if (marge(nEvals, g, u, E) > maxMargeE)
            {
               #pragma omp critical
               {
                  maxMargeE = marge(nEvals, g, u, E);
                  maxEid = u;
               }
            }
         }
      }
      if (maxMargeE > 0)
      {
         E[maxEid] = true;
         valE += maxMargeE;
      }
   }

   g.logg << "IG: Second interlacing complete." << endL;

   valA = compute_valSet(nEvals, g, A);
   valB = compute_valSet(nEvals, g, B);
   valD = compute_valSet(nEvals, g, D);
   valE = compute_valSet(nEvals, g, E);
   vector<double> vC;
   vC.push_back(valA);
   vC.push_back(valB);
   vC.push_back(valD);
   vC.push_back(valE);

   double valC = 0;
   size_t jj = 0;
   for (size_t i = 0; i < vC.size(); ++i)
   {
      if (vC[i] > valC)
      {
         valC = vC[i];
         jj = i;
      }
   }

   if (jj == 0)
      C = A;
   if (jj == 1)
      C = B;
   if (jj == 2)
      C = D;
   if (jj == 3)
      C = E;
   g.logg << "C: " << compute_valSet(nEvals, g, C) << endL;

   // steal
   if (steal)
   {
      vector<MyPair> possibleGain;
      vector<MyPair> Cbenefits;
      MyPair tmp;
      rounds++;
      for (size_t i = 0; i < g.n; ++i)
      {
         if (A[i] || B[i] || D[i] || E[i])
         {
            tmp.u = i;
            tmp.gain = marge(nEvals, g, i, C);

            possibleGain.push_back(tmp);
         }

         if (C[i])
         {
            tmp.u = i;
            tmp.gain = leastBenefit(i, C);

            Cbenefits.push_back(tmp);
         }
      }

      // g.logg << "IG: Sorting..." << endL;
      std::sort(Cbenefits.begin(), Cbenefits.end(), gainLT());
      std::sort(possibleGain.begin(), possibleGain.end(), revgainLT());

      // Attempt to replace elements of C
      size_t nStolen = 0;
      for (size_t i = 0; i < Cbenefits.size(); ++i)
      {

         if (Cbenefits[i].gain < possibleGain[i].gain)
         {

            if (C[possibleGain[i].u])
            {
               C[possibleGain[i].u] = false;
            }
            else
            {
               ++nStolen;
               C[Cbenefits[i].u] = false;
               C[possibleGain[i].u] = true;
            }
         }
      }
      g.logg << "IG: Stealing complete: " << nStolen << " stolen." << endL;
   }
   g.logg << "C: " << compute_valSet(nEvals, g, C) << endL;
   g.logg << "Evals: " << nEvals << endL;
   reportResults(nEvals, compute_valSet(nEvals, g, C),rounds);
}

double Fig::leastBenefit(node_id u, vector<bool> &set, vector<double> &valRounds)
{
   set[u] = false;
   double m = marge(nEvals, g, u, set);
   set[u] = true;

   if (reportRounds)
   {
      report_rounds(set, valRounds, g);
   }
   return m;
}

bool Fig::swap(node_id u, node_id v, vector<bool> &set)
{
   double init = compute_valSet(nEvals, g, set);
   set[u] = false;
   set[v] = true;
   double m = compute_valSet(nEvals, g, set);
   if (m > init)
   {
      return true;
   }
   set[u] = true;
   set[v] = false;
   return false;
}

size_t Fig::sizeSet(vector<bool> &S)
{
   size_t ssize = 0;
   for (size_t i = 0; i < g.n; ++i)
   {
      if (S[i])
         ++ssize;
   }
   return ssize;
}

void Fig::add(vector<bool> &S, vector<bool> &T, node_id &j, double &tau, vector<double> &valRounds)
{
   if (sizeSet(S) == k)
   {
      j = 0;
      tau = stopGain;
      return;
   }

   while (tau > stopGain)
   {
      for (node_id x = j; x < g.n; ++x)
      {
         if (!T[x])
         {
            if (reportRounds)
            {
               report_rounds(S, valRounds, g);
            }
            if (marge(nEvals, g, x, S) >= (tau))
            {
               S[x] = true;
               j = x;
               return;
            }
         }
      }
      tau = (1 - epsi) * tau;
      j = 0;
   }
   j = 0;
   return;
}

void Fig::run()
{
   vector<bool> A(g.n, false);
   vector<bool> B(g.n, false);
   vector<bool> C(g.n, false);
   vector<bool> D(g.n, false);
   vector<bool> E(g.n, false);

   g.logg << "FIG: epsi = " << epsi << ", k = " << k << endL;

   // Get max singleton
   g.logg << "FIG: Determining max singleton..." << endL;
   double M = 0;
   node_id a0;
   for (size_t x = 0; x < g.n; ++x)
   {
      if (marge(nEvals, g, x, A) > (M))
      {
         a0 = x;
         M = marge(nEvals, g, x, A);
      }
   }

   vector<double> valRounds;

   g.logg << "FIG: M = " << M << endL;
   g.logg << "FIG: Stopping condition: " << stopGain << endL;

   g.logg << "FIG: Starting first interlacing..." << endL;
   double tauA = M;
   double tauB = M;
   stopGain = epsi * M / g.n;
   node_id a = 0;
   node_id b = 0;
   while (tauA > stopGain || tauB > stopGain)
   {
      // g.logg << "FIG: tauA = " << tauA << ", tauB = " << tauB << endL;
      add(A, B, a, tauA, valRounds);
      add(B, A, b, tauB, valRounds);
   }

   // g.logg << "FIG: First interlacing complete." << endL;
   g.logg << "FIG: Starting second interlacing..." << endL;
   double tauD = M;
   double tauE = M;
   node_id d = 0;
   node_id e = 0;
   D[a0] = true;
   E[a0] = true;
   while (tauD > stopGain || tauE > stopGain)
   {
      // g.logg << "FIG: tauD = " << tauD << ", tauE = " << tauE << endL;
      add(D, E, d, tauD, valRounds);
      add(E, D, e, tauE, valRounds);
   }

   double valA = compute_valSet(nEvals, g, A);
   double valB = compute_valSet(nEvals, g, B);
   double valD = compute_valSet(nEvals, g, D);
   double valE = compute_valSet(nEvals, g, E);
   vector<double> vC;
   vC.push_back(valA);
   vC.push_back(valB);
   vC.push_back(valD);
   vC.push_back(valE);

   double valC = 0;
   size_t jj = 0;
   for (size_t i = 0; i < vC.size(); ++i)
   {
      if (vC[i] > valC)
      {
         valC = vC[i];
         jj = i;
      }
   }

   if (jj == 0)
      C = A;
   if (jj == 1)
      C = B;
   if (jj == 2)
      C = D;
   if (jj == 3)
      C = E;
   g.logg << "FIG: f(C) = " << compute_valSet(nEvals, g, C) << endL;

   // steal
   if (steal)
   {
      vector<MyPair> possibleGain;
      vector<MyPair> Cbenefits;
      MyPair tmp;
      for (size_t i = 0; i < g.n; ++i)
      {
         if (A[i] || B[i] || D[i] || E[i])
         {
            tmp.u = i;
            tmp.gain = marge(nEvals, g, i, C);

            possibleGain.push_back(tmp);
         }

         if (reportRounds)
         {
            report_rounds(C, valRounds, g);
         }

         if (C[i])
         {
            tmp.u = i;
            tmp.gain = leastBenefit(i, C, valRounds);

            Cbenefits.push_back(tmp);
         }
      }

      std::sort(Cbenefits.begin(), Cbenefits.end(), gainLT());
      std::sort(possibleGain.begin(), possibleGain.end(), revgainLT());

      // Attempt to replace elements of C
      size_t nStolen = 0;
      for (size_t i = 0; i < Cbenefits.size(); ++i)
      {

         if (Cbenefits[i].gain < possibleGain[i].gain)
         {

            if (C[possibleGain[i].u])
            {
               C[possibleGain[i].u] = false;
            }
            else
            {
               if (this->swap(Cbenefits[i].u,
                              possibleGain[i].u,
                              C))
               {
                  ++nStolen;
                  // C[ Cbenefits[i].u ] = false;
                  // C[ possibleGain[i].u ] = true;
               }

               if (reportRounds)
               {
                  report_rounds(C, valRounds, g);
               }
            }
         }
      }
      g.logg << "FIG: Stealing complete: " << nStolen << " stolen." << endL;
      g.logg << "FIG: f(C) = " << compute_valSet(nEvals, g, C) << endL;
   }

   g.logg << "FIG: # evals = " << nEvals << endL;
   reportResults(nEvals, compute_valSet(nEvals, g, C), valRounds.size());

   if (reportRounds)
   {
      for (size_t j = 0; j < valRounds.size(); ++j)
      {
         // allResults.add( to_string( j ), valRounds[ j ] );
      }
   }
}

double Rg::leastBenefit(node_id u, vector<bool> &set)
{
   ++nEvals;
   set[u] = false;
   double m = marge(nEvals, g, u, set);
   set[u] = true;
   return m;
}

void Rg::run()
{
   vector<bool> A(g.n, false);
   vector<MyPair> margeGains;
   MyPair tmp;

   vector<double> valRounds;
   for (size_t i = 0; i < k; ++i)
   {
      if (reportRounds)
      {
         report_rounds(A, valRounds, g);
      }

      margeGains.clear();
      #pragma omp parallel for
      for (node_id u = 0; u < g.n; ++u)
      {
         if (!(A[u]))
         {
            #pragma omp critical
            {
               tmp.gain = marge(nEvals, g, u, A);
               tmp.u = u;
               margeGains.push_back(tmp);
            }
         }
      }

      std::sort(margeGains.begin(), margeGains.end(), revgainLT());
      uniform_int_distribution<size_t> dist(0, k - 1);
      size_t rand = dist(gen);
      node_id u = margeGains[rand].u;
      A[u] = true;
   }

   g.logg << "A: " << compute_valSet(nEvals, g, A) << endL;
   g.logg << "Evals: " << nEvals << endL;

   reportResults(nEvals, compute_valSet(nEvals, g, A), k);

   if (reportRounds)
   {
      // for (size_t j = 0; j < valRounds.size(); ++j) {
      //    allResults.add( to_string( j ), valRounds[ j ] );
      // }
   }
}

double Frg::leastBenefit(node_id u, vector<bool> &set)
{
   ++nEvals;
   set[u] = false;
   double m = marge(nEvals, g, u, set);
   set[u] = true;
   return m;
}

void Frg::fillM(vector<node_id> &M, vector<bool> &S)
{

   while (w > epsi * W / k)
   {
      for (node_id x = 0; x < g.n; ++x)
      {
         if (marge(nEvals, g, x, S) > (1 - epsi) * w)
         {
            M.push_back(x);
            if (M.size() >= k)
               return;
         }
      }

      w = (1 - epsi) * w;
   }
}

void Frg::run()
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   runRandom();
}

void Frg::randomSampling(double p, size_t s, vector<bool> &A)
{
   size_t rho = p * g.n + 1;
   vector<bool> M;
   vector<MyPair> margeGains;
   MyPair tmp;
   for (size_t i = 0; i < k; ++i)
   {
      sampleUnifSize(M, rho);

      margeGains.clear();
      for (node_id u = 0; u < g.n; ++u)
      {
         if (M[u])
         {
            tmp.gain = marge(nEvals, g, u, A);
            tmp.u = u;
            margeGains.push_back(tmp);
         }
      }

      std::sort(margeGains.begin(), margeGains.end(), revgainLT());
      uniform_int_distribution<size_t> dist(0, s - 1);
      size_t rand = dist(gen);
      node_id u = margeGains[rand].u;
      if (marge(nEvals, g, u, A) >= 0.0)
      {
         A[u] = true;
      }
   }
}

void Frg::sampleUnifSize(vector<bool> &R, size_t Size)
{

   uniform_int_distribution<size_t> dist(0, g.n - 1);
   R.assign(g.n, false);

   for (size_t i = 0; i < Size; ++i)
   {
      size_t pos;
      do
      {
         pos = dist(gen);
      } while (R[pos]);
      R[pos] = true;
   }
}

void Frg::runRandom()
{
   double p = 8.0 / (k * epsi * epsi) * log(2 / (epsi));
   g.logg << "FastRandom: p = " << p << endL;
   if (p >= 1.0)
   {
      // run RandomGreedy
      g.logg << "FastRandom: Running RandomGreedy..." << endL;
      Rg rg(myArgs);
      rg.run();
   }
   else
   {
      g.logg << "FastRandom: Running RandomSampling..." << endL;
      vector<bool> S(g.n, false);
      size_t rho = p * g.n + 1;
      size_t s = rho * k / g.n;

      randomSampling(p, s, S);

      g.logg << "S: " << compute_valSet(nEvals, g, S) << endL;
      g.logg << "Evals: " << nEvals << endL;

      reportResults(nEvals, compute_valSet(nEvals, g, S), k);
   }
}

void Frg::runSimple()
{
   vector<node_id> M;
   vector<node_id> newM;
   vector<bool> S(g.n, false);

   // Get max singleton
   g.logg << "FRG: Determining max singleton..." << endL;
   W = 0;

   for (size_t x = 0; x < g.n; ++x)
   {
      if (marge(nEvals, g, x, S) > W)
      {
         W = marge(nEvals, g, x, S);
      }
   }
   w = W;

   for (size_t i = 0; i < k; ++i)
   {
      g.logg << "FRG: iteration " << i << endL;
      fillM(M, S);
      uniform_int_distribution<size_t> dist(0, k - 1);
      size_t rand = dist(gen);
      if (rand >= M.size())
         continue;

      node_id u = M[rand];
      S[u] = true;
      // M.erase( M.begin() + u );
      for (size_t j = 0; j < M.size(); ++j)
      {
         if (marge(nEvals, g, M[j], S) <= (1 - epsi) * w)
         {
            //
         }
         else
         {
            newM.push_back(M[j]);
         }
      }
      M = newM;
   }

   g.logg << "S: " << compute_valSet(nEvals, g, S) << endL;
   g.logg << "Evals: " << nEvals << endL;

   reportResults(nEvals, compute_valSet(nEvals, g, S));
}

void Frg::runImproved()
{
   vector<node_id> M;
   vector<bool> S(g.n, false);

   // Get max singleton
   g.logg << "FRG: Determining max singleton..." << endL;
   W = 0;

   for (size_t x = 0; x < g.n; ++x)
   {
      if (marge(nEvals, g, x, S) > static_cast<signed long>(W))
      {
         W = marge(nEvals, g, x, S);
      }
   }

   // w = W;

   fillM(M, S);

   for (size_t i = 0; i < k; ++i)
   {
      uniform_int_distribution<size_t> dist(0, M.size() - 1);
      size_t rand = dist(gen);
      node_id u = M[rand];
      if (marge(nEvals, g, u, S) > (1 - epsi) * w)
      {
         S[u] = true;
      }
      else
      {
         vector<node_id> newM;
         for (size_t j = 0; j < M.size(); ++j)
         {
            if (marge(nEvals, g, M[j], S) <= (1 - epsi) * w)
            {
               //
            }
            else
            {
               newM.push_back(M[j]);
            }
         }
         M = newM;
         size_t sizeOld = M.size();
         fillM(M, S);
         size_t sizeInc = M.size() - sizeOld;
         if (sizeInc > 0)
         {
            uniform_int_distribution<size_t> dist(0, sizeInc - 1);
            size_t rand = dist(gen);
            node_id u = M[sizeOld + rand];
            S[u] = true;
         }
      }
   }

   g.logg << "S: " << compute_valSet(nEvals, g, S) << endL;
   g.logg << "Evals: " << nEvals << endL;

   reportResults(nEvals, compute_valSet(nEvals, g, S));
}

long Sg::leastBenefit(node_id u, vector<bool> &set)
{
   set[u] = false;
   long m = marge(nEvals, g, u, set);
   set[u] = true;
   return m;
}

void Sg::run()
{
   vector<bool> A(g.n, false);

   double maxGain;
   node_id maxIdx;
   MyPair tmp;

   vector<double> valRounds;

   for (size_t i = 0; i < k; ++i)
   {
      if (reportRounds)
      {
         report_rounds(A, valRounds, g);
      }

      maxGain = 0;
      for (node_id u = 0; u < g.n; ++u)
      {

         if (marge(nEvals, g, u, A) > maxGain)
         {
            maxIdx = u;
            maxGain = marge(nEvals, g, u, A);
         }
      }

      if (maxGain > 0)
      {
         A[maxIdx] = true;
      }
      else
      {
         break;
      }
   }

   g.logg << "A: " << compute_valSet(nEvals, g, A) << endL;
   g.logg << "Evals: " << nEvals << endL;

   reportResults(nEvals, compute_valSet(nEvals, g, A));

   if (reportRounds)
   {
      for (size_t j = 0; j < valRounds.size(); ++j)
      {
         allResults.add(to_string(j), valRounds[j]);
      }
   }
}

size_t Anm::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}

void Anm::run()
{
   g.logg << "ANM starting run..." << endL;
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> sol(g.n, false);
   double solVal = 0;

   vector<MyPair> margeGains;
   MyPair tmp;

   margeGains.clear();
   for (node_id u = 0; u < g.n; ++u)
   {
      tmp.gain = marge(nEvals, g, u, sol);
      tmp.u = u;
      margeGains.push_back(tmp);
   }

   double topKgains = 0;
   std::sort(margeGains.begin(), margeGains.end(), revgainLT());
   for (size_t i = 0; i < k; ++i)
   {
      topKgains += margeGains[i].gain;
   }

   double M = margeGains[0].gain; // topKgains / k;

   size_t r = 0;
   if (!fast)
   {
      epsi = epsi / 6.0;
      r = log(k) / epsi + 1;
      delta = delta / (2 * (r + 1));
   }
   else
   {
      r = log(k) / epsi + 1;
   }

   double c_1 = 1.0 / 7;
   double c_3 = 3;

   g.logg << "topKgains = " << topKgains << endL;
   g.logg << "M = " << M << endL;

   double tau_i = M / (1 - epsi) * c_1;

   g.logg << "r = " << r << endL;
   double solRound = 0;
   vector<double> valueTmp;
   vector<vector<double>> valueAllRounds(r + 1, valueTmp);

   for (unsigned i = 0; i <= r; ++i)
   {
      vector<double> &valRounds = valueAllRounds[i];
      valRounds.clear();
      valRounds.push_back(0);

      vector<bool> A(g.n, false);
      vector<bool> S(g.n, false);
      vector<size_t> idsA;
      vector<size_t> idsS;

      threshold_sample(nEvals, g, S, idsS, emptyBoolVector, false, k, tau_i, epsi, delta, fast, reportRounds, valRounds);
      double tempVal = compute_valSet(nEvals, g, S);
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = S;
         solRound = i;
      }

      tau_i = tau_i * (1 - epsi);

      size_t sizeA = idsA.size();

      if (sizeA < c_3 * k)
      {
         vector<size_t> idsU;
         random_set(g, idsU, idsA);
         // unc_max( nEvals, g, idsA, epsi, delta, idsU );
         if (reportRounds)
            report_rounds(sol, valRounds, g);
         if (idsU.size() > k)
         {
            vector<size_t> idsD;
            sampleUt(idsD, idsU, k);
            idsU = idsD;
         }

         random_shuffle(idsU.begin(), idsU.end());

         vector<bool> empty(g.n, false);
         vector<bool> prefix(g.n, false);
         double tempVal = 0;

         for (size_t j = 0; j < idsU.size(); ++j)
         {
            prefix[idsU[j]] = true;

            tempVal = compute_valSet(nEvals, g, prefix);

            if (tempVal >= solVal)
            {
               solVal = tempVal;
               sol = prefix;
               solRound = i;
            }
         }

         if (reportRounds)
         {
            report_rounds(sol, valRounds, g);
         }
      }
   }

   g.logg << INFO << "ANM: solVal=" << solVal << endL;
   g.logg << INFO << "ANM: queries=" << nEvals << endL;
   g.logg << INFO << "ANM: solSize=" << get_size_set(sol) << endL;

   size_t rounds = 0;
   if (reportRounds)
   {
      g.logg << INFO << "ANM: Adaptive rounds=" << valueAllRounds[solRound].size() << endL;
      rounds = valueAllRounds[solRound].size();
      for (size_t j = 0; j < valueAllRounds[solRound].size(); ++j)
      {
         allResults.add(to_string(j), valueAllRounds[solRound][j]);
      }
   }

   reportResults(nEvals, solVal, rounds);
}

size_t Ast::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}

void Ast::run()
{
   vector<bool> A(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<bool> B(g.n, false);
   vector<bool> C(g.n, false);
   vector<bool> sol(g.n, false);
   size_t solVal = 0;
   size_t solRound = 0;

   vector<MyPair> margeGains;
   MyPair tmp;

   margeGains.clear();
   for (node_id u = 0; u < g.n; ++u)
   {
      tmp.gain = marge(nEvals, g, u, A);
      tmp.u = u;
      margeGains.push_back(tmp);
   }

   double topKgains = 0;
   std::sort(margeGains.begin(), margeGains.end(), revgainLT());
   for (size_t i = 0; i < k; ++i)
   {
      topKgains += margeGains[i].gain;
   }

   double M = topKgains / k;
   size_t m = log(1.0 / (6 * k)) / log(1 - epsi);
   if (a)
   {
      M = topKgains / (6 * k);
      m = log(1.0 / k) / log(1 - epsi);
   }

   g.logg << INFO << "AST: M = " << M;

   double tau_i = M / (1 - epsi);

   g.logg << ", number of iterations = " << m << endL;

   vector<double> valueTmp;
   vector<vector<double>> valueAllRounds(m + 1, valueTmp);

   for (unsigned i = 0; i <= m; ++i)
   {
      vector<double_t> &valueThisRound = valueAllRounds[i];
      valueThisRound.clear();
      valueThisRound.push_back(0); // Finding topKgains was first adaptive round

      g.logg << INFO << "------------AST: " << i << "th iteration------------" << endL;

      tau_i = tau_i * (1 - epsi);
      g.logg << INFO << "AST: tau_i = " << tau_i << endL;
      if (fast)
      {
         // g.logg << "ATG: Stopping condition: " << solVal * (1 - epsi) / (6 * k) << endL << INFO;

         if (tau_i < solVal * (1 - epsi) / (6 * k))
         {
            // solVal is a lower bound on OPT and
            // telling us we can stop now.
            break;
         }
      }
      A.assign(g.n, false);
      idsA.clear();
      threshold_sample(nEvals, g, A, idsA, emptyBoolVector, false, k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      B.assign(g.n, false);
      idsB.clear();
      threshold_sample(nEvals, g, B, idsB, A, true, k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      random_set(g, C, idsA);
      // unc_max(nEvals, g, idsA, epsi, delta, C);
      // auto size_C = accumulate(C.begin(), C.end(), 0);

      double tempVal = compute_valSet(nEvals, g, A);

      g.logg << DEBUG << "AST: f(A)=" << tempVal;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = A;

         solRound = i;
      }

      tempVal = compute_valSet(nEvals, g, B);
      g.logg << DEBUG << ", f(B)=" << tempVal;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = B;

         solRound = i;
      }

      tempVal = compute_valSet(nEvals, g, C);
      g.logg << DEBUG << ", f(C)=" << tempVal << endL;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = C;
         solRound = i;
      }

      g.logg << INFO << "AST: solVal=" << solVal;
      g.logg << ", solSize = " << get_size_set(sol);
      g.logg << ", queries=" << nEvals << endL;

      if (fast)
      {
         if (get_size_set(sol) == k)
         {
            // We can quit now.
            break;
         }
      }
   }

   size_t rounds = 0;

   if (reportRounds)
   {
      g.logg << "AST: rounds=" << valueAllRounds[solRound].size() << endL;
      rounds = valueAllRounds[solRound].size();
      for (size_t j = 0; j < valueAllRounds[solRound].size(); ++j)
      {
         allResults.add(to_string(j), valueAllRounds[solRound][j]);
      }
   }

   reportResults(nEvals, solVal, rounds);
}

size_t AmaAst::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}

void AmaAst::run()
{
   vector<bool> A(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<bool> B(g.n, false);
   vector<bool> C(g.n, false);
   vector<bool> sol(g.n, false);
   size_t solVal = 0;
   size_t solRound = 0;

   vector<MyPair> margeGains;
   MyPair tmp;

   margeGains.clear();
   for (node_id u = 0; u < g.n; ++u)
   {
      tmp.gain = marge(nEvals, g, u, A);
      tmp.u = u;
      margeGains.push_back(tmp);
   }

   double topKgains = 0;
   std::sort(margeGains.begin(), margeGains.end(), revgainLT());
   for (size_t i = 0; i < k; ++i)
   {
      topKgains += margeGains[i].gain;
   }

   double M = topKgains / k;
   size_t m = log(1.0 / (6 * k)) / log(1 - epsi);
   if (a)
   {
      M = topKgains / (6 * k);
      m = log(1.0 / k) / log(1 - epsi);
   }

   g.logg << INFO << "AST: M = " << M;

   double tau_i = M / (1 - epsi);

   g.logg << INFO << ", number of iterations = " << m << endL;

   vector<double> valueTmp;
   vector<vector<double>> valueAllRounds(m + 1, valueTmp);

   for (unsigned i = 0; i <= m; ++i)
   {
      vector<double_t> &valueThisRound = valueAllRounds[i];
      valueThisRound.clear();
      valueThisRound.push_back(0); // Finding topKgains was first adaptive round

      g.logg << INFO << "------------AST: " << i << "th iteration------------" << endL;

      tau_i = tau_i * (1 - epsi);
      g.logg << "AST: tau_i = " << tau_i << endL;
      if (fast)
      {
         // g.logg << "AmaATG: Stopping condition: " << solVal * (1 - epsi) / (6 * k) << endL << INFO;

         if (tau_i < solVal * (1 - epsi) / (6 * k))
         {
            // solVal is a lower bound on OPT and
            // telling us we can stop now.
            break;
         }
      }

      A.assign(g.n, false);
      idsA.clear();
      threshold_seq_ama(nEvals, g, A, idsA, emptyBoolVector, false, k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      B.assign(g.n, false);
      idsB.clear();
      threshold_seq_ama(nEvals, g, B, idsB, A, true, k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      random_set(g, C, idsA);
      // unc_max(nEvals, g, idsA, epsi, delta, C);
      // auto size_C = accumulate(C.begin(), C.end(), 0);

      double tempVal = compute_valSet(nEvals, g, A);

      g.logg << DEBUG << "f(A)=" << tempVal;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = A;

         solRound = i;
      }

      tempVal = compute_valSet(nEvals, g, B);
      g.logg << DEBUG << ", f(B)=" << tempVal;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = B;

         solRound = i;
      }

      tempVal = compute_valSet(nEvals, g, C);
      g.logg << DEBUG << ", f(C)=" << tempVal << endL;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = C;
         solRound = i;
      }

      g.logg << INFO << "AST: solVal=" << solVal;
      g.logg << ", solSize = " << get_size_set(sol);
      g.logg << ", queries=" << nEvals << endL;

      if (fast)
      {
         if (get_size_set(sol) == k)
         {
            // We can quit now.
            break;
         }
      }
   }

   size_t rounds = 0;

   if (reportRounds)
   {
      g.logg << "AST: rounds=" << valueAllRounds[solRound].size() << endL;
      rounds = valueAllRounds[solRound].size();
      for (size_t j = 0; j < valueAllRounds[solRound].size(); ++j)
      {
         allResults.add(to_string(j), valueAllRounds[solRound][j]);
      }
   }

   reportResults(nEvals, solVal, rounds);
}

size_t AmaAst2::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}

void AmaAst2::run()
{
   vector<bool> A(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<bool> B(g.n, false);
   vector<bool> C(g.n, false);
   vector<bool> sol(g.n, false);
   size_t solVal = 0;
   size_t solRound = 0;

   vector<MyPair> margeGains;
   MyPair tmp;

   margeGains.clear();
   for (node_id u = 0; u < g.n; ++u)
   {
      tmp.gain = marge(nEvals, g, u, A);
      tmp.u = u;
      margeGains.push_back(tmp);
   }

   double topKgains = 0;
   std::sort(margeGains.begin(), margeGains.end(), revgainLT());
   for (size_t i = 0; i < k; ++i)
   {
      topKgains += margeGains[i].gain;
   }

   double M = topKgains / k;
   size_t m = log(1.0 / (6 * k)) / log(1 - epsi);
   if (a)
   {
      M = topKgains / (6 * k);
      m = log(1.0 / k) / log(1 - epsi);
   }

   g.logg << INFO << "AST: M = " << M;

   double tau_i = M / (1 - epsi);

   g.logg << ", number of iterations = " << m << endL;

   vector<double> valueTmp;
   vector<vector<double>> valueAllRounds(m + 1, valueTmp);

   for (unsigned i = 0; i <= m; ++i)
   {
      vector<double_t> &valueThisRound = valueAllRounds[i];
      valueThisRound.clear();
      valueThisRound.push_back(0); // Finding topKgains was first adaptive round

      g.logg << "------------AST: " << i << "th iteration------------" << endL;

      tau_i = tau_i * (1 - epsi);
      g.logg << "AST: tau_i = " << tau_i << endL;

      if (fast)
      {
         // g.logg << "AmaATG: Stopping condition: " << solVal * (1 - epsi) / (6 * k) << endL << INFO;

         if (tau_i < solVal * (1 - epsi) / (6 * k))
         {
            // solVal is a lower bound on OPT and
            // telling us we can stop now.
            break;
         }
      }

      A.assign(g.n, false);
      idsA.clear();
      threshold_seq_ama2(nEvals, g, A, idsA, emptyBoolVector, false, k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      B.assign(g.n, false);
      idsB.clear();
      threshold_seq_ama2(nEvals, g, B, idsB, A, true, k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      random_set(g, C, idsA);
      // unc_max(nEvals, g, idsA, epsi, delta, C);
      // auto size_C = accumulate(C.begin(), C.end(), 0);

      double tempVal = compute_valSet(nEvals, g, A);

      g.logg << DEBUG << "AST: f(A)=" << tempVal << endL;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = A;

         solRound = i;
      }

      tempVal = compute_valSet(nEvals, g, B);
      g.logg << DEBUG << "f(B)=" << tempVal << endL;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = B;

         solRound = i;
      }

      tempVal = compute_valSet(nEvals, g, C);
      g.logg << DEBUG << "f(C)=" << tempVal << endL;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = C;
         solRound = i;
      }

      g.logg << INFO << "AST: solVal=" << solVal;
      g.logg << ", solSize = " << get_size_set(sol);
      g.logg << ", queries=" << nEvals << endL;

      if (fast)
      {
         if (get_size_set(sol) == k)
         {
            // We can quit now.
            break;
         }
      }

      // if (!fast) {
      //    if (solVal > topKgains / 6.0) {
      //       break;
      //    }
      // }
   }

   size_t rounds = 0;

   if (reportRounds)
   {
      g.logg << "AST: rounds=" << valueAllRounds[solRound].size() << endL;
      rounds = valueAllRounds[solRound].size();
      for (size_t j = 0; j < valueAllRounds[solRound].size(); ++j)
      {
         allResults.add(to_string(j), valueAllRounds[solRound][j]);
      }
   }

   reportResults(nEvals, solVal, rounds);
}

size_t TSAst::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}

void TSAst::run()
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> A(g.n, false);
   vector<bool> Aprime(g.n, false);
   vector<bool> B(g.n, false);
   vector<bool> Bprime(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<size_t> idsAprime;
   vector<size_t> idsBprime;
   vector<bool> C(g.n, false);
   vector<bool> sol(g.n, false);
   size_t solVal = 0;
   size_t solRound = 0;

   vector<MyPair> margeGains;
   MyPair tmp;

   // calculate objective value for each single element
   margeGains.clear();
   for (node_id u = 0; u < g.n; ++u)
   {
      tmp.gain = marge(nEvals, g, u, A);
      tmp.u = u;
      margeGains.push_back(tmp);
   }

   // calculate average of top k values as initial tau
   double topKgains = 0;
   std::sort(margeGains.begin(), margeGains.end(), revgainLT());
   for (size_t i = 0; i < k; ++i)
   {
      topKgains += margeGains[i].gain;
   }
   double M = topKgains / k;

   size_t m = log(1.0 / (6 * k)) / log(1 - epsi);
   if (a)
   {
      M = topKgains / (6 * k);
      m = log(1.0 / k) / log(1 - epsi);
   }

   g.logg << INFO << "TSATG: M = " << M;

   double tau_i = M / (1 - epsi);

   g.logg << ", number of iterations = " << m << endL;

   vector<double> valueTmp;
   vector<vector<double>> valueAllRounds(m + 1, valueTmp);

   for (unsigned i = 0; i <= m; ++i)
   {
      vector<double_t> &valueThisRound = valueAllRounds[i];
      valueThisRound.clear();
      valueThisRound.push_back(0); // Finding topKgains was first adaptive round

      g.logg << INFO << "------------AST: " << i << "th iteration------------" << endL;

      tau_i = tau_i * (1 - epsi);
      g.logg << INFO << "AST: tau_i = " << tau_i << endL;

      if (fast)
      {
         if (tau_i < solVal * (1 - epsi) / (6 * k))
         {
            // solVal is a lower bound on OPT and
            // telling us we can stop now.
            break;
         }
      }

      A.assign(g.n, false);
      Aprime.assign(g.n, false);
      idsA.clear();
      idsAprime.clear();
      threshold_seq(nEvals, g, A, Aprime, idsA, idsAprime, emptyBoolVector, false,
                    k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      B.assign(g.n, false);
      Bprime.assign(g.n, false);
      idsB.clear();
      idsBprime.clear();
      threshold_seq(nEvals, g, B, Bprime, idsB, idsBprime, A, true,
                    k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      random_set(g, C, idsA);

      double tempVal = compute_valSet(nEvals, g, Aprime);

      g.logg << DEBUG << "TSATG: f(A')=" << tempVal;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = Aprime;

         solRound = i;
      }

      tempVal = compute_valSet(nEvals, g, Bprime);
      g.logg << DEBUG << ", f(B')=" << tempVal;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = Bprime;

         solRound = i;
      }

      tempVal = compute_valSet(nEvals, g, C);
      g.logg << DEBUG << ". f(A'')=" << tempVal << endL;
      if (tempVal >= solVal)
      {
         solVal = tempVal;
         sol = C;
         solRound = i;
      }
      if (fast)
      {
         if (get_size_set(sol) == k)
         {
            // We can quit now.
            break;
         }
      }
   }
   g.logg << INFO << "AST: solVal=" << solVal;
   g.logg << ", solSize = " << get_size_set(sol);
   g.logg << ", queries=" << nEvals << endL;

   size_t rounds = 0;

   if (reportRounds)
   {
      g.logg << "AST: rounds=" << valueAllRounds[solRound].size() << endL;

      rounds = valueAllRounds[solRound].size();
      for (size_t j = 0; j < valueAllRounds[solRound].size(); ++j)
      {
         allResults.add(to_string(j), valueAllRounds[solRound][j]);
      }
   }

   reportResults(nEvals, solVal, rounds);
}

double Ene::onenorm(vector<double> &x)
{
   double val = 0.0;
   for (size_t i = 0; i < x.size(); ++i)
   {
      if (x[i] > 0)
         val += x[i];
      else
         val -= x[i];
   }
   return val;
}

/**
 *  This is multilinear formula for unweighted, undirected
 *  graph cut.
 */
double Ene::evalMultilinear(size_t &nEvals, vector<double> &x, size_t nsamps)
{
   if (nsamps > 0)
   {
      return evalMultilinearSample(nEvals, x, nsamps);
   }
   // cut-value only

   ++nEvals;

   double val = 0.0;
   for (node_id u = 0; u < g.n; ++u)
   {
      vector<tinyEdge> &neis = g.adjList[u].neis;
      for (size_t j = 0; j < neis.size(); ++j)
      {
         node_id v = neis[j].target;
         if (u < v)
         {
            val += x[u] * (1.0 - x[v]) + (1.0 - x[u]) * x[v];
         }
      }
   }

   return val;
}

double Ene::evalMultilinearSample(size_t &nEvals, vector<double> &x, size_t nsamps)
{
   if (nsamps == 0)
   {
      nsamps = nSamps;
   }

   double val;

   if (nThreads > 1)
      val = eval_multilinear_base(
          nEvals,
          g,
          x,
          nsamps,
          nThreads,
          vgens);
   else
      val = evalMultilinearOld(nEvals, x, nsamps);

   return val;
}

double Ene::evalMultilinearOld(size_t &nEvals, vector<double> &x, size_t nsamps)
{
   if (nsamps == 0)
   {
      nsamps = nSamps;
   }

   double val = 0.0;
   vector<bool> set(g.n, false);
   for (size_t i = 0; i < nsamps; ++i)
   {
      for (size_t j = 0; j < g.n; ++j)
      {
         set[j] = false;
         if (unidist(gen) < x[j])
         {
            set[j] = true;
         }
      }
      val += compute_valSet(nEvals, g, set);
   }

   return val / nsamps;
}

/*
 * Graph cut-value objective
 */
void Ene::gradientMultilinear(size_t &nEvals, vector<double> &gradient, vector<double> &x, size_t nsamps)
{
   if (nsamps > 0)
   {
      gradientMultilinearSample(nEvals, gradient, x);
      return;
   }

   ++nEvals;

   for (node_id u = 0; u < g.n; ++u)
   {
      vector<tinyEdge> &neis = g.adjList[u].neis;
      double &valu = gradient[u];
      valu = 0.0;
      for (size_t j = 0; j < neis.size(); ++j)
      {
         node_id v = neis[j].target;
         valu += 1.0 - 2 * x[v];
      }
   }
}

void Ene::gradientMultilinearSample(size_t &nEvals, vector<double> &gradient, vector<double> &x)
{
   double gamma = 0.5;
   double fx = evalMultilinearSample(nEvals, x, nSamps);
   for (size_t i = 0; i < g.n; ++i)
   {
      if (x[i] + 0.5 * gamma > 1.0)
      {
         // use backward difference
         x[i] -= gamma;
         gradient[i] = (fx - evalMultilinearSample(nEvals, x, nSamps)) / gamma;
         x[i] += gamma;
      }
      else
      {
         if (x[i] - 0.5 * gamma < 0.0)
         {
            // use forward difference
            x[i] += gamma;
            gradient[i] = (evalMultilinearSample(nEvals, x, nSamps) - fx) / gamma;
            x[i] -= gamma;
         }
         else
         {
            // use central difference
            x[i] += 0.5 * gamma;
            gradient[i] = (evalMultilinearSample(nEvals, x, nSamps));
            x[i] -= gamma;
            gradient[i] = (gradient[i] - evalMultilinearSample(nEvals, x, nSamps)) / gamma;
            x[i] += 0.5 * gamma;
         }
      }
   }
}

void Ene::computeTeta(size_t &nEvals,
                      double eta,
                      double v,
                      vector<double> &z,
                      vector<node_id> &S,
                      vector<node_id> &Seta,
                      vector<node_id> &Teta,
                      size_t j,
                      vector<double> &zstart,
                      size_t idxthread,
                      vector<double> &vec_g)
{
   vector<double> zeta(z.begin(), z.end());

   for (size_t i = 0; i < S.size(); ++i)
   {
      zeta[S[i]] = zeta[S[i]] + eta * (1.0 - zeta[S[i]]);
   }

   vector<double> grad(g.n, 0.0);
   if (idxthread == 0)
   {
      ++rounds;
   }
   gradientMultilinear(nEvals, grad, zeta, nSamps);

   Seta.clear();
   Teta.clear();

   vector<double> vec_g_eta(g.n);
   for (size_t i = 0; i < g.n; ++i)
   {
      vec_g_eta[i] = (1.0 - zeta[i]) * grad[i];
   }

   for (size_t i = 0; i < S.size(); ++i)
   {
      if (vec_g_eta[S[i]] >= v)
         Seta.push_back(S[i]);
      if (vec_g_eta[S[i]] > 0.0)
         Teta.push_back(S[i]);
   }

   //    cerr << "eta Seta S " << eta << ' ' << Seta.size() << ' ' << S.size() << ' ' << (Seta.size() >= (1.0 - epsi)*S.size()) << endl;
}

bool Ene::evalEta(size_t &nEvals, double eta, double epsi, double v,
                  vector<double> &z, vector<node_id> &S, size_t j, vector<double> &zstart, size_t threadindex,
                  vector<double> &vec_g)
{
   vector<node_id> Seta;
   vector<node_id> Teta;
   computeTeta(nEvals, eta, v, z, S, Seta, Teta, j, zstart, threadindex, vec_g);

   return (Seta.size() >= (1.0 - epsi) * S.size());
}

double Ene::findeta1(size_t &nEvals, double v, vector<double> &z, vector<double> &vec_g, vector<node_id> &S, size_t j, vector<double> &zstart, size_t threadindex)
{

   double etastart = 0.0;
   double etaend = epsi * epsi;
   // double etaend = epsi;
   while (etaend - etastart > delta)
   {
      double etatest = (etastart + etaend) / 2.0;

      if (evalEta(nEvals, etatest, epsi, v, z, S, j, zstart, threadindex, vec_g))
      {
         etastart = etatest;
      }
      else
      {
         etaend = etatest;
      }
   }

   return etastart + delta;
}

double Ene::findeta2(double epsi, size_t j, size_t k, vector<double> &z, vector<node_id> &S)
{
   double eta = epsi * j * k - onenorm(z);

   double on1 = 0.0;
   for (size_t i = 0; i < S.size(); ++i)
   {
      on1 += z[S[i]];
   }

   eta /= (S.size() - on1);

   if (epsi * epsi < eta)
      return epsi * epsi;

   return eta;
}

void Ene::runM(size_t &rounds, size_t &nEvals, double epsi, double M, double &valout, size_t i)
{
   vector<double> x(g.n, 0.0);
   vector<double> z(g.n, 0.0);
   vector<double> xstart;
   vector<double> zstart;

   for (size_t j = 1; j <= size_t(ceil(1.0 / epsi)); ++j)
   {
      //++rounds;

      xstart.assign(x.begin(), x.end());
      zstart.assign(z.begin(), z.end());
      double vstart = (pow((1 - epsi), j) - 2 * epsi) * M - evalMultilinear(nEvals, x, nSamps);
      //      double vstart = M - evalMultilinear( nEvals, x );
      if (vstart < 0)
      {
         vstart = 0;
      }
      vstart /= k;

      if (print)
      {
         cerr << "Guess " << i << ": j = " << j << endl;
      }

      // double vstart = (1 - epsi)*M - evalMultilinear( x );

      double v = vstart;
      while ((v > epsi * vstart) && (onenorm(z) < epsi * j * k))
      {
         vector<node_id> S;
         S.clear();

         vector<double> gradf(g.n);
         ++rounds;
         gradientMultilinear(nEvals, gradf, z, nSamps);
         // compute vec_g
         vector<double> vec_g(g.n);
         for (size_t i = 0; i < g.n; ++i)
         {
            vec_g[i] = (1.0 - z[i]) * gradf[i];

            if (vec_g[i] >= v)
            {
               if (z[i] <= 1.0 - pow(1.0 - epsi, j))
               {
                  if (z[i] < epsi * (1.0 - zstart[i]) + zstart[i])
                  {
                     S.push_back(i);
                  }
               }
            }
         }
         // if (print)
         // cerr << "S.size " << S.size() << endl;

         if (S.size() == 0)
         {
            // v = (1 - epsi)*v;
            v = 0.75 * v;
         }
         else
         {
            double eta1 = findeta1(nEvals, v, z, vec_g, S, j, zstart, i);
            double eta2 = findeta2(epsi, j, k, z, S);
            double eta = min(eta1, eta2);
            //	  double minStepSize = epsi*epsi / 10;
            // if (eta < minStepSize)
            // eta = minStepSize;
            // cerr << "eta1 = " << eta1 << ", " << "eta2 =" << eta2 << endl;

            vector<node_id> Teta;
            vector<node_id> Seta;
            // computeTeta( eta - delta, v, z, S, Seta, Teta );
            computeTeta(nEvals, eta - delta, v, z, S, Seta, Teta, j, zstart, i, vec_g);
            // update x,z
            for (size_t i = 0; i < Teta.size(); ++i)
            {
               x[Teta[i]] = x[Teta[i]] + eta * (1.0 - x[Teta[i]]);
            }

            for (size_t i = 0; i < S.size(); ++i)
            {
               z[S[i]] = z[S[i]] + eta * (1.0 - z[S[i]]);
            }

            if (evalMultilinear(nEvals, z, nSamps) > evalMultilinear(nEvals, x, nSamps))
            {
               x.assign(z.begin(), z.end());
            }

            // if (print) {
            //   cerr << "x: " << endl;
            //   for (size_t i = 0; i < g.n; ++i) {
            //     cerr << x[i] << ' ';
            //   }
            //   cerr << endl;

            //   cerr << "z: " << endl;
            //   for (size_t i = 0; i < g.n; ++i) {
            //     cerr << z[i] << ' ';
            //   }
            //   cerr << endl;
            // }
         }
      }
      // cerr << "v epsi*vstart " << v << ' ' << epsi*vstart << endl;
      // cerr << "onenorm(z), epsijk " << onenorm(z) << ' ' << epsi*j*k << endl;
   }

   valout = evalMultilinear(nEvals, x, 0); // evalMultilinear( x, 100000 );
   nEvals -= 1;                            // don't count this evaluation in the total nEvals
                                           // cerr << "OPT Guess " << i << " (tau, val, onenorm): " << M << ' ' << valout << ' ' << onenorm( x ) << endl;
                                           // cerr << "x: " <<endl;
                                           // for (size_t i = 0; i < g.n; ++i) {
                                           // cerr << x[i] << ' ';
                                           // }
                                           // cerr << endl;
                                           // cerr << endl;
}

void Ene::run()
{
   g.logg << "Ene: starting run..." << endL;
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   double solVal = 0.0;
   double tau = 0;
   vector<bool> A(g.n, false);
   #pragma omp parallel for reduction(max:tau)
   for (node_id u = 0; u < g.n; ++u)
   {
      if (marge(nEvals, g, u, A) >= tau)
      {
         tau = marge(nEvals, g, u, A);
      }
   }

   double tauinit = tau;
   vector<double> vtau;
   vector<size_t> vEvals;
   vector<size_t> vRounds;
   while (tau < tauinit * k)
   {
      vtau.push_back(tau);
      tau = tau * (1 + epsi);
      vEvals.push_back(0);
      vRounds.push_back(1);
      // tau = tau*(1.1);
   }

   size_t nThreadsG;
   if (nThreads == 1)
      nThreadsG = vtau.size();
   else
      nThreadsG = 1;

   g.logg << "Paralellizing OPT guesses in " << nThreadsG << " threads..." << endL;

   if (nThreadsG > 1)
   {
      thread *wThreads = new thread[nThreadsG];
      vector<double> vals(nThreadsG);
      for (size_t i = 0; i < nThreadsG; ++i)
      {
         wThreads[i] = thread(&Ene::runM, this, ref(vRounds[i]), ref(vEvals[i]), epsi, vtau[i], ref(vals[i]), i);
      }

      for (size_t i = 0; i < nThreadsG; ++i)
      {
         wThreads[i].join();
         if (vals[i] > solVal)
            solVal = vals[i];
         nEvals += vEvals[i];
         if (vRounds[i] > rounds)
            rounds = vRounds[i];
      }

      delete[] wThreads;
   }
   else
   {
      #pragma omp parallel for
      for (size_t i = 0; i < vtau.size(); ++i)
      {
         double val;
         runM(vRounds[i], nEvals, epsi, vtau[i], val, i);
         #pragma omp critical
         {
         if (val > solVal)
            solVal = val;
         if (vRounds[i] > rounds)
            rounds = vRounds[i];
         }
      }
   }

   g.logg << INFO << "Ene: solVal=" << solVal << endL;
   g.logg << INFO << "Ene: queries=" << nEvals << endL;

   if (reportRounds)
   {
      g.logg << INFO << "Ene: Adaptive rounds=" << rounds << endL;
   }

   reportResults(nEvals, solVal, rounds);
}

size_t Atg::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}

void Atg::run()
{
   vector<bool> A(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<bool> B(g.n, false);
   vector<bool> C(g.n, false);
   vector<bool> sol(g.n, false);
   double solVal = 0;

   // Get max singleton
   double M = 0;
   // node_id a0;

   for (size_t x = 0; x < g.n; ++x)
   {
      if (marge(nEvals, g, x, A) > (M))
      {
         M = marge(nEvals, g, x, A);
      }
   }

   g.logg << INFO << "M = " << M;

   size_t m = log(1.0 / (c * k)) / log(1 - epsi); // number of iterations

   double tau_i = M / (1 - epsi);

   g.logg << INFO << ", number of iterations = " << m << endL;

   if (!fast)
   {
      delta = 1.0 / (2 * m);
   }

   vector<double> valueThisRound;

   valueThisRound.push_back(0); // Finding topKgains was first adaptive round
   for (unsigned i = 0; i <= m; ++i)
   {
      g.logg << INFO << "--------------";
      g.logg << INFO << "1st for loop, iteration #" << i << endL;
      tau_i = tau_i * (1 - epsi);

      threshold_sample(nEvals, g, A, idsA, emptyBoolVector, false,
                       k, tau_i,
                       epsi, delta,
                       fast,
                       reportRounds,
                       valueThisRound);
      g.logg << INFO << "Size of A returned: " << idsA.size() << ", rounds = " << valueThisRound.size() << endL;

      if (idsA.size() == k)
         break;
   }

   tau_i = M / (1 - epsi);

   for (unsigned i = 0; i <= m; ++i)
   {
      g.logg << INFO << "--------------";
      g.logg << INFO << "2nd for loop, iteration #" << i << endL;

      tau_i = tau_i * (1 - epsi);

      threshold_sample(nEvals, g, B, idsB, A, true,
                       k,
                       tau_i,
                       epsi, delta,
                       fast,
                       reportRounds,
                       valueThisRound);
      g.logg << INFO << "Size of B returned: " << idsB.size() << ", rounds = " << valueThisRound.size() << endL;
      if (idsB.size() == k)
         break;
   }

   random_set(g, C, idsA);

   double tempVal = compute_valSet(nEvals, g, A);

   g.logg << INFO << "f(A)=" << tempVal;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = A;
   }

   tempVal = compute_valSet(nEvals, g, B);
   g.logg << INFO << ", f(B)=" << tempVal;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = B;
   }

   tempVal = compute_valSet(nEvals, g, C);
   g.logg << INFO << ", f(C)=" << tempVal << endL;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = C;
   }

   g.logg << INFO << "ATG: solVal=" << solVal;
   g.logg << INFO << ", queries=" << nEvals;
   g.logg << INFO << ", solSize=" << get_size_set(sol) << endL;

   size_t rounds = 0;

   if (reportRounds)
   {
      g.logg << INFO << "ATG: rounds=" << valueThisRound.size() << endL;
      rounds = valueThisRound.size();
      // for (size_t j = 0; j < valueThisRound.size(); ++j) {
      // allResults.add( to_string( j ), valueThisRound[ j ] );
      // }
   }

   reportResults(nEvals, solVal, rounds);
}

size_t AmaAtg::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}

void AmaAtg::run()
{
   vector<bool> A(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<bool> B(g.n, false);
   vector<bool> C(g.n, false);
   vector<bool> sol(g.n, false);
   double solVal = 0;

   // Get max singleton
   double M = 0;
   // node_id a0;

   for (size_t x = 0; x < g.n; ++x)
   {
      if (marge(nEvals, g, x, A) > (M))
      {
         M = marge(nEvals, g, x, A);
      }
   }

   g.logg << INFO << "M = " << M;

   size_t m = log(1.0 / (c * k)) / log(1 - epsi); // number of iterations

   double tau_i = M / (1 - epsi);

   g.logg << INFO << ", number of iterations = " << m << endL;

   if (!fast)
   {
      delta = 1.0 / (2 * m);
   }

   vector<double> valueThisRound;

   valueThisRound.push_back(0); // Finding topKgains was first adaptive round
   for (unsigned i = 0; i <= m; ++i)
   {
      g.logg << INFO << "--------------";
      g.logg << INFO << "1st for loop, iteration #" << i << endL;

      tau_i = tau_i * (1 - epsi);

      threshold_seq_ama(nEvals, g, A, idsA, emptyBoolVector, false, k,
                        tau_i, epsi, delta, fast, reportRounds, valueThisRound);
      g.logg << INFO << "Size of A returned: " << idsA.size() << ", rounds = " << valueThisRound.size() << endL;
      if (idsA.size() == k)
         break;
   }

   tau_i = M / (1 - epsi);

   for (unsigned i = 0; i <= m; ++i)
   {
      g.logg << INFO << "--------------";
      g.logg << INFO << "2nd for loop, iteration #" << i << endL;

      tau_i = tau_i * (1 - epsi);

      threshold_seq_ama(nEvals, g, B, idsB, A, true,
                        k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);

      g.logg << INFO << "Size of B returned: " << idsB.size() << ", rounds = " << valueThisRound.size() << endL;

      if (idsB.size() == k)
         break;
   }

   random_set(g, C, idsA);

   double tempVal = compute_valSet(nEvals, g, A);

   g.logg << INFO << "f(A)=" << tempVal;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = A;
   }

   tempVal = compute_valSet(nEvals, g, B);
   g.logg << ", f(B)=" << tempVal;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = B;
   }

   tempVal = compute_valSet(nEvals, g, C);
   g.logg << ", f(C)=" << tempVal << endL;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = C;
   }

   g.logg << INFO << "ATG: solVal=" << solVal;
   g.logg << INFO << ", queries=" << nEvals;
   g.logg << INFO << ", solSize=" << get_size_set(sol) << endL;

   size_t rounds = 0;

   if (reportRounds)
   {
      g.logg << INFO << "ATG: rounds=" << valueThisRound.size() << endL;
      rounds = valueThisRound.size();
      // for (size_t j = 0; j < valueThisRound.size(); ++j) {
      // allResults.add( to_string( j ), valueThisRound[ j ] );
      // }
   }

   reportResults(nEvals, solVal, rounds);
}

size_t AmaAtg2::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}

void AmaAtg2::run()
{
   vector<bool> A(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<bool> B(g.n, false);
   vector<bool> C(g.n, false);
   vector<bool> sol(g.n, false);
   double solVal = 0;

   // Get max singleton
   double M = 0;
   // node_id a0;

   for (size_t x = 0; x < g.n; ++x)
   {
      if (marge(nEvals, g, x, A) > (M))
      {
         M = marge(nEvals, g, x, A);
      }
   }

   g.logg << INFO << "M = " << M;

   size_t m = log(1.0 / (c * k)) / log(1 - epsi); // number of iterations

   double tau_i = M / (1 - epsi);

   g.logg << INFO << ", number of iterations = " << m << endL;

   if (!fast)
   {
      delta = 1.0 / (2 * m);
   }

   vector<double> valueThisRound;

   valueThisRound.push_back(0); // Finding topKgains was first adaptive round
   for (unsigned i = 0; i <= m; ++i)
   {
      g.logg << INFO << "--------------";
      g.logg << INFO << "1st for loop, iteration #" << i << endL;

      tau_i = tau_i * (1 - epsi);

      threshold_seq_ama2(nEvals, g, A, idsA, emptyBoolVector, false, k,
                         tau_i, epsi, delta, fast, reportRounds, valueThisRound);
      if (idsA.size() == k)
         break;
   }

   tau_i = M / (1 - epsi);

   for (unsigned i = 0; i <= m; ++i)
   {
      g.logg << INFO << "--------------";
      g.logg << INFO << "2nd for loop, iteration #" << i << endL;

      tau_i = tau_i * (1 - epsi);

      threshold_seq_ama2(nEvals, g, B, idsB, A, true,
                         k, tau_i, epsi, delta, fast, reportRounds, valueThisRound);
      g.logg << INFO << "Size of B returned: " << idsB.size() << endL;

      if (idsB.size() == k)
         break;
   }

   random_set(g, C, idsA);

   double tempVal = compute_valSet(nEvals, g, A);

   g.logg << INFO << "f(A)=" << tempVal;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = A;
   }

   tempVal = compute_valSet(nEvals, g, B);
   g.logg << ", f(B)=" << tempVal;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = B;
   }

   tempVal = compute_valSet(nEvals, g, C);
   g.logg << ", f(C)=" << tempVal << endL;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = C;
   }

   g.logg << INFO << "ATG: solVal=" << solVal;
   g.logg << INFO << ", queries=" << nEvals;
   g.logg << INFO << ", solSize=" << get_size_set(sol) << endL;

   size_t rounds = 0;

   if (reportRounds)
   {
      g.logg << INFO << "ATG: rounds=" << valueThisRound.size() << endL;
      rounds = valueThisRound.size();
      // for (size_t j = 0; j < valueThisRound.size(); ++j) {
      // allResults.add( to_string( j ), valueThisRound[ j ] );
      // }
   }

   reportResults(nEvals, solVal, rounds);
}

size_t TSAtg::get_size_set(vector<bool> &S)
{
   size_t sizeSol = 0;
   for (size_t i = 0; i < g.n; ++i)
      if (S[i])
         ++sizeSol;
   return sizeSol;
}
double TSAtg::USM(vector<node_id> A, double epsilon, double delta)
{
   int t = ceil(log(1 / delta) / log(1 + (4.0/3.0) * epsilon));
   std::random_device rd;
   std::mt19937 gen(rd());
   std::uniform_int_distribution<> dist(0, A.size() - 1);
   double max_f=0;
   #pragma omp parallel for
   for (int i = 0; i < t; i++)
   {
      vector<node_id> tmp;
      for (int j = 0; j < A.size(); j++)
      {
         if (dist(gen) == 0)
            tmp.push_back(A[j]);
      }
      double tmp_f=0;
      if(tmp.size()>0)
         tmp_f= compute_valSet(nEvals, g, tmp);
      #pragma omp critical
      {
         if(tmp_f > max_f)
            max_f = tmp_f;
      }
   }
   return max_f;
}
void TSAtg::run()
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> A(g.n, false);
   vector<bool> Aprime(g.n, false);
   vector<bool> B(g.n, false);
   vector<bool> Bprime(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<size_t> idsAprime;
   vector<size_t> idsBprime;
   vector<bool> C(g.n, false);
   vector<bool> sol(g.n, false);
   double solVal = 0;

   // Get max singleton
   double M = 0;
   // node_id a0;

   for (size_t x = 0; x < g.n; ++x)
   {
      if (marge(nEvals, g, x, A) > (M))
      {
         M = marge(nEvals, g, x, A);
      }
   }

   g.logg << INFO << "M = " << M;

   size_t m = log(1.0 / (c * k)) / log(1 - epsi);

   double tau_i = M / (1 - epsi);

   g.logg << INFO << ", number of iterations = " << m << endL;
   double delta=0.1;
   if (!fast)
   {
      delta = 1.0/(2*m);
   }
   g.logg << DEBUG << "delta = " << delta << endL;

   vector<double> valueThisRound;

   valueThisRound.push_back(0); // Finding topKgains was first adaptive round
   for (unsigned i = 0; i <= m; ++i)
   {
      g.logg << INFO << "--------------";
      g.logg << INFO << "1st for loop, iteration #" << i << endL;
      tau_i = tau_i * (1 - epsi);

      threshold_seq(nEvals, g, A, Aprime, idsA, idsAprime, emptyBoolVector, false,
                    k, tau_i, epsi, delta,
                    fast,
                    reportRounds,
                    valueThisRound);
      g.logg << INFO << "Size of A returned: " << idsA.size() << endL;
      if (idsA.size() == k)
         break;
   }

   tau_i = M / (1 - epsi);

   for (unsigned i = 0; i <= m; ++i)
   {
      g.logg << INFO << "--------------";
      g.logg << INFO << "2nd for loop, iteration #" << i << endL;
      
      tau_i = tau_i * (1 - epsi);

      threshold_seq(nEvals, g, B, Bprime, idsB, idsBprime, A, true,
                    k,
                    tau_i,
                    epsi, delta,
                    fast,
                    reportRounds,
                    valueThisRound);
      g.logg << INFO << "Size of B returned: " << idsB.size() << endL;
      if (idsB.size() == k)
         break;
   }

   random_set(g, C, idsA);

   double tempVal = compute_valSet(nEvals, g, Aprime);

   g.logg << INFO << "f(A')=" << tempVal;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = Aprime;
   }

   tempVal = compute_valSet(nEvals, g, Bprime);
   g.logg << ", f(B')=" << tempVal;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = Bprime;
   }

   tempVal = compute_valSet(nEvals, g, C);
   g.logg << ", f(A'')=" << tempVal << endL;
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = C;
   }

   g.logg << INFO << "ATG: solVal=" << solVal;
   g.logg << INFO << ", queries=" << nEvals;
   g.logg << INFO << ", solSize=" << get_size_set(sol) << endL;
   
   size_t rounds = 0;

   if (reportRounds)
   {
      g.logg << INFO << "ATG: rounds=" << valueThisRound.size() << endL;
      rounds = valueThisRound.size();
      // for (size_t j = 0; j < valueThisRound.size(); ++j) {
      // allResults.add( to_string( j ), valueThisRound[ j ] );
      // }
   }

   reportResults(nEvals, solVal, rounds);
}
int TSAtg::FindKPrime(int k,double epsi)
{
   int k_prime=0;
   for (int i = k; (1 - epsi) * i <= k; ++i)
   {
      k_prime = i;
   }
   return k_prime;
}
//run STA (this work)
void TSAtg::runBoostAdapt()//BoostAdapt
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> A(g.n, false);
   vector<bool> Aprime(g.n, false);
   vector<bool> B(g.n, false);
   vector<bool> Bprime(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<size_t> idsAprime;
   vector<size_t> idsBprime;
   vector<bool> sol(g.n, false);
   vector<bool> XY(g.n,false);
   epsi=0.1;
   nEvals=0;
   int adapt=0;
   double alpha=2.5;
   double a= 8 + 2*myalpha + 2/myalpha + (16/(1-4*epsi) + 2*(1 + 1/myalpha)*( (2*(2-epsi))/((1-epsi)*(1-2*epsi))))*epsi;
   double f_S0 = LinAdapt(epsi,1.0/3.0,adapt);
   double solVal = f_S0;
   cout<<"f_S0 = "<<f_S0<<endl;
   double M = a * f_S0;
   cout<<"M:"<<M<<endl;
   size_t m = ceil(log( (a*k) / (4*epsi)) / log(1/(1 - epsi))) + 1;
   cout<<"m:"<<m<<endl;
   double delta= 1.0/m;
   cout<<"delta:"<<delta<<endl;
   int k_prime=FindKPrime(k,epsi);
   cout<<"k_prime:"<<k_prime<<endl;
   double tau = k_prime * M / (4*k);
   vector<double> valueThisRound;
   for (unsigned i = 1; i <= m; ++i)
   {
      double tau_i = tau * pow((1 - epsi),i);
      cout<<"tau_i:"<<tau_i<<endl;
      if(i%2==1)
      {
         threshold_seq(nEvals, g, A, Aprime, idsA, idsAprime, XY, true,
                    k_prime, tau_i, epsi, delta,
                    fast,
                    reportRounds,
                    valueThisRound);
         #pragma omp parallel for
         for(int j=0;j<A.size();j++)
         {
            if(A[j]) XY[j]=true;
         }
      }
      else
      {
         threshold_seq(nEvals, g, B, Bprime, idsB, idsBprime, XY, true,
                    k_prime,
                    tau_i,
                    epsi, delta,
                    fast,
                    reportRounds,
                    valueThisRound);
         #pragma omp parallel for
         for(int j=0;j<B.size();j++)
         {
            if(B[j]) XY[j]=true;
         }
      }          
   }
   if(Aprime.size()<=k)
   {
      double tmp = compute_valSet(nEvals, g, Aprime);
      //cout<<"Aprime f:"<<tmp<<endl;
      if(tmp > solVal)
         solVal=tmp;
   }else
   {
      vector<bool> tmp(g.n,false);
      for(int j=0;j<k;j++)
      {
         node_id e=idsAprime[0];
         double gain=0;
         //adap++;
         #pragma omp parallel for
         for(int i=0;i<idsAprime.size();i++)
         {
            if(tmp[idsAprime[i]]) continue;
            double tmp_gain=marge(nEvals, g, idsAprime[i], tmp);
            if( tmp_gain > gain)
            {
               #pragma omp critical
               {
                  gain=tmp_gain;
                  e=idsAprime[i];
               }
            }
         }
         tmp[e]=true;
      }
      double tmp_f = compute_valSet(nEvals, g, tmp);
      //cout<<"Aprime f:"<<tmp_f<<endl;
      if(tmp_f > solVal)
         solVal=tmp_f;
   }
   if(Bprime.size()<=k)
   {
      double tmp = compute_valSet(nEvals, g, Bprime);
      //cout<<"Bprime f:"<<tmp<<endl;
      if(tmp > solVal)
         solVal=tmp;
   }
   else
   {
      vector<bool> tmp(g.n,false);
      for(int j=0;j<k;j++)
      {
         node_id e=idsBprime[0];
         double gain=0;
         //adapt++;
         #pragma omp parallel for
         for(int i=0;i<idsBprime.size();i++)
         {
            if(tmp[idsBprime[i]]) continue;
            double tmp_gain=marge(nEvals, g, idsBprime[i], tmp);
            if( tmp_gain > gain)
            {
               #pragma omp critical
               {
                  gain=tmp_gain;
                  e=idsBprime[i];
               }
            }
         }
         tmp[e]=true;
      }
      double tmp_f = compute_valSet(nEvals, g, tmp);
      //cout<<"Bprime f:"<<tmp_f<<endl;
      if(tmp_f > solVal)
         solVal=tmp_f;
   }
   size_t rounds =  valueThisRound.size() + adapt;
   reportResults(nEvals, solVal, rounds);
}
void TSAtg::runLinAst()
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   epsi=0.1;
   double solVal = 0;
   nEvals=0;
   int adapt=0;
   double a= 8 + 2*myalpha + 2/myalpha + (16/(1-4*epsi) + 2*(1 + 1/myalpha)*( (2*(2-epsi))/((1-epsi)*(1-2*epsi))))*epsi;
   double f_S0 = LinAdapt(epsi,0.3333333,adapt);
   //cout<<"f_S0 = "<<f_S0<<endl;
   double c= 4+ beta;
   size_t ell = ceil(log( 1/(a)) / log(1 - epsi)) + 1;
   
   double M = a * f_S0/(c*k);
   double tau_i = M;
   int rounds_thred;
   int size_sol=0;
   //cout<<"ell: "<<ell<<endl;
   #pragma omp parallel for
   for (unsigned i = 1; i <= ell; ++i)
   {
      vector<bool> A(g.n, false);
      vector<bool> Aprime(g.n, false);
      vector<bool> B(g.n, false);
      vector<bool> Bprime(g.n, false);
      vector<bool> C(g.n, false);
      vector<size_t> idsA;
      vector<size_t> idsB;
      vector<size_t> idsAprime;
      vector<size_t> idsBprime;
      
      vector<double_t> valueThisRound;

      valueThisRound.clear();
      valueThisRound.push_back(0);
      tau_i = tau_i * (1 - epsi);

      threshold_seq(nEvals, g, A, Aprime, idsA, idsAprime, emptyBoolVector, false,
                    k, tau_i, epsi, 0.5, fast, reportRounds, valueThisRound);

      threshold_seq(nEvals, g, B, Bprime, idsB, idsBprime, A, true,
                    k, tau_i, epsi, 0.5, fast, reportRounds, valueThisRound);

      random_set(g, C, idsA);

      double tempVal = compute_valSet(nEvals, g, Aprime);

      if (tempVal >= solVal)
      {
         #pragma omp critical
         {
            solVal = tempVal;
            rounds_thred=valueThisRound.size();
         }
      }

      tempVal = compute_valSet(nEvals, g, Bprime);
      if (tempVal >= solVal)
      {
         #pragma omp critical
         {
            solVal = tempVal;
            rounds_thred=valueThisRound.size();
         }
      }

      tempVal = compute_valSet(nEvals, g, C);
      if (tempVal >= solVal)
      {
         #pragma omp critical
         {
            solVal = tempVal;
            rounds_thred=valueThisRound.size();
         }
      }
   }
   size_t rounds =  rounds_thred + adapt;
   reportResults(nEvals, solVal, rounds);
}
//run BLA
void TSAtg::runLinAtg()
{
   if (alpha.size() == 0)
   {
      init_alpha(g);
   }
   vector<bool> A(g.n, false);
   vector<bool> Aprime(g.n, false);
   vector<bool> B(g.n, false);
   vector<bool> Bprime(g.n, false);
   vector<bool> C(g.n, false);
   vector<size_t> idsA;
   vector<size_t> idsB;
   vector<size_t> idsAprime;
   vector<size_t> idsBprime;
   vector<bool> sol(g.n, false);
   vector<bool> XY(g.n,false);
   epsi=0.1;
   nEvals=0;
   int adapt=0;
   double a = 8 + 2*myalpha + 2/myalpha + (16/(1-4*epsi) + 2*(1 + 1/myalpha)*( (2*(2-epsi))/((1-epsi)*(1-2*epsi))))*epsi;;
   double f_S0 = LinAdapt(epsi,1.0/3.0,adapt);
   //cout<<"f_S0 = "<<f_S0<<endl;
   size_t nEvals_LinAdapt=nEvals;
   double M = a * f_S0/k;
   double c = 8/epsi;
   double epsi2 = (1 - 1/exp(1))*epsi/8;
   //double epsi2 =epsi;
   int ell = ceil(log(a*c) / log(1/(1-epsi2))) + 1;
   double delta = 1.0/(3.0*ell);
   //cout<<"ell = "<<ell<<endl;
   epsi2 =epsi;
   double solVal = f_S0;
   size_t solRound=0;
   int size_sol=0;
   vector<double_t> valueThisRound;
   valueThisRound.push_back(0);
   for (unsigned i = 1; i <= ell; ++i)
   {
      double tau = M * pow(1 - epsi2,i-1);
      threshold_seq(nEvals, g, A, Aprime, idsA, idsAprime, XY, true,
                  k, tau, epsi2, delta,
                  fast,
                  reportRounds,
                  valueThisRound);
      for(int j=0;j<A.size();j++)
      {
         if(A[j]) XY[j]=true;
      }
      if (idsA.size() == k) break;
   }
   for (unsigned i = 1; i <= ell; ++i)
   {
      double tau = M * pow(1 - epsi2,i-1);
      threshold_seq(nEvals, g, B, Bprime, idsB, idsBprime, XY, true,
                    k,
                    tau,
                    epsi2, delta,
                    fast,
                    reportRounds,
                    valueThisRound);
      for(int j=0;j<B.size();j++)
      {
         if(B[j]) XY[j]=true;
      }
      if (idsB.size() == k) break;             
   }
   random_set(g, C, idsA);
   double tempVal = compute_valSet(nEvals, g, Aprime);
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = Aprime;
   }

   tempVal = compute_valSet(nEvals, g, A);
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = Bprime;
   }

   tempVal = compute_valSet(nEvals, g, Bprime);
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = Bprime;
   }

   tempVal = compute_valSet(nEvals, g, B);
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = Bprime;
   }

   tempVal = compute_valSet(nEvals, g, C);
   if (tempVal >= solVal)
   {
      solVal = tempVal;
      sol = C;
   }

   size_t rounds = 0;
   //g.logg << INFO << "ATG: rounds=" << valueThisRound.size() << endL;
   rounds = valueThisRound.size() + adapt;
   //g.logg << INFO << "AST: solVal=" << solVal;
   //g.logg << ", solSize = " << get_size_set(sol);
   //g.logg << ", queries=" << nEvals << endL;
   cout<<solVal<<","<<rounds<<","<<nEvals<<endl;
   reportResults(nEvals, solVal, rounds, nEvals_LinAdapt);
}
size_t Blits::sizeSet(vector<bool> &S)
{
   size_t ssize = 0;
   for (size_t i = 0; i < g.n; ++i)
   {
      if (S[i])
         ++ssize;
   }
   return ssize;
}

void Blits::sampleUX(vector<bool> &R, vector<bool> &X)
{
   // g.logg << "Starting sampleUX..." << endL;
   uniform_int_distribution<size_t> dist(0, g.n - 1);
   R.assign(g.n, false);
   for (size_t i = 0; i < k / r; ++i)
   {
      size_t pos;
      do
      {
         pos = dist(gen);
      } while (!X[pos] || R[pos]);
      R[pos] = true;
   }
   // g.logg << "Finished sampleUX." << endL;
}

double Blits::Delta(node_id a, vector<bool> &S, vector<bool> &X)
{
   double avg = 0.0;
   for (size_t i = 0; i < nSamps; ++i)
   {
      vector<bool> R;
      if (sizeSet(X) >= k / r)
         sampleUX(R, X);
      else
         R = X;
      R[a] = false;
      vector<bool> RcupS;
      setunion(RcupS, R, S);
      // RcupS[a] = false;
      avg += marge(nEvals, g, a, RcupS);
   }

   return avg / nSamps;
}

double Blits::exMarge(vector<bool> &S, vector<bool> &Xplus, vector<bool> &X)
{
   double sum = 0.0;
   for (size_t i = 0; i < nSamps; ++i)
   {
      vector<bool> R;
      sampleUX(R, X);
      vector<bool> base;
      vector<bool> larger;
      setintersection(base, R, Xplus);
      int64_t valBase = compute_valSet(nEvals, g, S);
      setunion(larger, S, base);
      int64_t valLarger = compute_valSet(nEvals, g, larger);
      sum += (valLarger - valBase);
   }

   return sum / nSamps;
}

bool Blits::sieve(vector<bool> &res, vector<bool> &A, size_t i, size_t &rounds)
{
   g.logg << "Starting sieve, iteration " << i << endL;
   vector<bool> X(g.n, true);
   res.assign(g.n, false);
   size_t sizeX = g.n;
   double base = 1.0 - 1 / static_cast<double>(r);
   double t = (0.5 - epsi / 4) * (pow(base, i - 1) * (1 - epsi / 2.0) * OPT - compute_valSet(nEvals, g, A));
   // if (t < 0)
   // return false;
   g.logg << "t = " << t << endL;
   size_t sizeXprior = 0;
   while (sizeX > k)
   {
      g.logg << "Size of X: " << sizeX << endL;
      vector<bool> Xplus(g.n, false);
      for (size_t a = 0; a < g.n; ++a)
      {
         if (Delta(a, A, X) >= 0.0)
            Xplus[a] = true;
      }
      ++rounds;

      if (sizeXprior == sizeX)
      {
         g.logg << WARN << "SizeX is not decreasing..." << endL;
         g.logg << INFO;

         vector<bool> R;
         sampleUX(R, X);
         setintersection(res, R, Xplus);
         return true;

         //	    return false;
      }

      double exMar = exMarge(A, Xplus, X);
      ++rounds;
      g.logg << INFO << "exMarge: " << exMar << endL;
      g.logg << INFO << "t/r: " << t / r << endL;
      if (exMar >= t / r)
      {
         vector<bool> R;
         sampleUX(R, X);
         setintersection(res, R, Xplus);
         return true;
      }

      Xplus.assign(g.n, false);
      sizeXprior = sizeX;
      sizeX = 0;
      for (size_t a = 0; a < g.n; ++a)
      {
         if (Delta(a, A, X) >= (1 + epsi / 4) * t / k)
         {
            Xplus[a] = true;
            ++sizeX;
         }
      }
      X = Xplus;
      ++rounds;
   }

   g.logg << "Size of X: " << sizeX << endL;

   vector<bool> Xplus(g.n, false);
   for (size_t a = 0; a < g.n; ++a)
   {
      if (Delta(a, A, X) >= 0.0)
         Xplus[a] = true;
   }
   vector<bool> R;
   ++rounds;
   if (sizeSet(X) >= k / r)
      sampleUX(R, X);
   else
      R = X;

   setintersection(res, R, Xplus);
   return true;
}

double Blits::leastBenefit(node_id u, vector<bool> &set)
{
   set[u] = false;
   double m = marge(nEvals, g, u, set);
   set[u] = true;
   return m;
}

void Blits::setunion(vector<bool> &res, vector<bool> &set1, vector<bool> &set2)
{
   res.assign(g.n, false);
   for (node_id u = 0; u < g.n; ++u)
   {
      if (set1[u] || set2[u])
      {
         res[u] = true;
      }
   }
}

void Blits::setintersection(vector<bool> &res, vector<bool> &set1, vector<bool> &set2)
{
   res.assign(g.n, false);
   for (node_id u = 0; u < g.n; ++u)
   {
      if (set1[u] && set2[u])
      {
         res[u] = true;
      }
   }
}

void Blits::run()
{
   if (r > k)
   {
      g.logg << ERROR << "r > k" << endL;
      g.logg << "Exiting Blits." << endL;
      g.logg << INFO;
      return;
   }

   vector<bool> S(g.n, false);

   vector<MyPair> margeGains;
   MyPair tmp;

   margeGains.clear();
   for (node_id u = 0; u < g.n; ++u)
   {
      tmp.gain = marge(nEvals, g, u, S);
      tmp.u = u;
      margeGains.push_back(tmp);
   }

   double topKgains = 0;
   std::sort(margeGains.begin(), margeGains.end(), revgainLT());
   for (size_t i = 0; i < k; ++i)
   {
      topKgains += margeGains[i].gain;
   }

   OPT = topKgains; // upper bound on OPT

   vector<vector<bool>> vSols;
   vector<size_t> roundsByGuess;
   size_t rounds;
   while (OPT > compute_valSet(nEvals, g, S))
   {
      g.logg << "Starting Blits, with OPT estimate: " << OPT << endL;
      rounds = 1; // Guess OPT in parallel
      S.assign(g.n, false);
      for (size_t i = 1; i <= r; ++i)
      {
         vector<bool> step;
         if (!sieve(step, S, i, rounds))
            break;
         vector<bool> Splus;
         setunion(Splus, S, step);
         S = Splus;
      }

      vSols.push_back(S);
      OPT = OPT * (1 - epsi);
      roundsByGuess.push_back(rounds);
   }

   vector<bool> Smax(g.n, false);
   double valS = 0;
   for (size_t i = 0; i < vSols.size(); ++i)
   {
      if (compute_valSet(nEvals, g, vSols[i]) > valS)
      {
         valS = compute_valSet(nEvals, g, vSols[i]);
         Smax = vSols[i];
         rounds = roundsByGuess[i];
      }
   }

   g.logg << "S: " << valS << endL;
   g.logg << "Evals: " << nEvals << endL;
   g.logg << "Rounds: " << rounds << endL;

   reportResults(nEvals, valS, rounds);
}

double Tg::leastBenefit(node_id u, vector<bool> &set)
{
   set[u] = false;
   double m = marge(nEvals, g, u, set);
   set[u] = true;
   return m;
}

void Tg::uncMax(vector<bool> &set, vector<double> &valRounds)
{
   vector<bool> all(set);
   vector<bool> none(g.n, false);

   for (size_t u = 0; u < g.n; ++u)
   {
      if (set[u])
      {
         if (reportRounds)
         {
            report_rounds(none, valRounds, g);
         }
         double margeA = marge(nEvals, g, u, none); // adding u to A
         vector<bool> allMinus(all);
         allMinus[u] = false;
         double margeB = static_cast<double>(compute_valSet(nEvals, g, allMinus)) - compute_valSet(nEvals, g, all);
         if (margeA >= margeB)
         {
            none[u] = true;
         }
         else
         {
            all[u] = false;
         }
      }
   }

   set = all;
}

void Tg::run()
{
   vector<bool> A(g.n, false);

   double maxGain;
   node_id maxIdx;
   MyPair tmp;

   vector<double> valRounds;

   for (size_t i = 0; i < k; ++i)
   {
      if (reportRounds)
      {
         report_rounds(A, valRounds, g);
      }
      maxGain = -1.0 * g.n * g.n;
      for (node_id u = 0; u < g.n; ++u)
      {

         if (marge(nEvals, g, u, A) > maxGain)
         {

            maxIdx = u;
            maxGain = marge(nEvals, g, u, A);
         }
      }

      A[maxIdx] = true;
   }

   vector<bool> B(g.n, false);
   for (size_t i = 0; i < k; ++i)
   {
      if (reportRounds)
      {
         report_rounds(A, valRounds, g);
      }
      maxGain = -1.0 * g.n * g.n;
      for (node_id u = 0; u < g.n; ++u)
      {
         if (!A[u])
         {
            if (marge(nEvals, g, u, B) > maxGain)
            {
               maxIdx = u;
               maxGain = marge(nEvals, g, u, B);
            }
         }
      }

      B[maxIdx] = true;
   }

   vector<bool> Amax = A;
   vector<bool> Bmax = B;

   uncMax(Amax, valRounds);
   uncMax(Bmax, valRounds);

   vector<double> vals;
   vals.push_back(compute_valSet(nEvals, g, A));
   vals.push_back(compute_valSet(nEvals, g, Amax));
   vals.push_back(compute_valSet(nEvals, g, B));
   vals.push_back(compute_valSet(nEvals, g, Bmax));
   size_t maxVal = 0;
   g.logg << "Vals: ";
   for (size_t i = 0; i < vals.size(); ++i)
   {
      g.logg << vals[i] << " ";
      if (vals[i] > maxVal)
      {
         maxVal = vals[i];
      }
   }

   g.logg << endL;
   g.logg << "Evals: " << nEvals << endL;

   reportResults(nEvals, maxVal, valRounds.size());

   if (reportRounds)
   {
      for (size_t j = 0; j < valRounds.size(); ++j)
      {
         // allResults.add( to_string( j ), valRounds[ j ] );
      }
   }
   
}

void Rand::run()
{
   vector<node_id> U(g.n);
   for (size_t i = 0; i < g.n; ++i)
   {
      U[i] = i;
   }

   random_shuffle(U.begin(), U.end());

   vector<bool> A(g.n, false);

   for (size_t i = 0; i < k; ++i)
   {
      A[U[i]] = true;
   };

   size_t tmp = 0;
   double val = compute_valSet(tmp, g, A);
   reportResults(0, val, 0);
};
vector<node_id> TSAtg::my_random_permutation(vector<node_id> v)
{
   std::random_device rd;
   std::mt19937 g(rd());
   std::shuffle(v.begin(), v.end(), g);
   return v;
}
double TSAtg::LinAdapt(double epsilon, double delta,int &adapt)
{
   std::chrono::steady_clock::time_point begin = std::chrono::steady_clock::now();
   vector<node_id> V;
   for(node_id i=0;i<g.n;i++)
   {
      V.push_back(i);
   }
   vector<node_id> Aprime;
   double f1 = TSAtg::LinBoundedSet(V,Aprime,epsilon,1.0/3.0,adapt);
   //cout<<"f1 = "<<f1<<endl;
   vector<node_id> Vprime;
   vector<bool> checkAprim(g.n,false);
   #pragma omp parallel for
   for(int i=0;i<Aprime.size();i++)
   {
      checkAprim[Aprime[i]]=true;
   }
   #pragma omp parallel for
   for(int i=0;i<g.n;i++)
   {
      if(checkAprim[i]==false)
      {
         #pragma omp critical
         {
            Vprime.push_back(i);
         }
      }
   }
   vector<node_id> Bprime;
   double f2 = TSAtg::LinBoundedSet(Vprime,Bprime,epsilon,1.0/3.0,adapt);
   //cout<<"f2 = "<<f2<<endl;
   double f3= USM(Aprime,epsilon,delta/(3.0*g.n));
   std::chrono::steady_clock::time_point end = std::chrono::steady_clock::now();
   size_t WallTimeMillis = std::chrono::duration_cast<std::chrono::milliseconds>(end - begin).count();
   double time = WallTimeMillis / 1000.0;
   cout<<"time LinAdapt = "<<time<<endl;
   return max(f1,max(f2,f3));
}
void TSAtg::updateCheck(vector<bool>& check, vector<node_id> A)
{
   for(int i=0;i<A.size();i++)
   {
      check[A[i]]=true;
   }
}
double TSAtg::LinBoundedSet(vector<node_id> V, vector<node_id> &Sprime, double epsilon, double delta, int &adapt)
{
   //std::chrono::steady_clock::time_point begin = std::chrono::steady_clock::now();
   vector<node_id> S;
   vector<bool> check_S(g.n, false);
   double f_S = FindEmax(V, S, check_S,adapt);
   //cout<<"f_S: "<<f_S<<endl;
   double beta = epsilon / (16 * log(8 / (1 - exp(-epsilon / 2))));
   int ell = ceil((4 + 4 / (beta * epsilon)) * log(V.size()/delta));
   vector<int> len_s;
   bool flag=true;
   vector<size_t> LAMBDA;
   vector<double> M;
   //std::chrono::steady_clock::time_point begin2 = std::chrono::steady_clock::now();
   for (int j = 1; j <= ell; j++)
   {
      adapt++;
      if(flag) V = updateV(V,f_S,check_S);
      //cout<<"V size: "<<V.size()<<endl;
      if (V.size() == 0) break;
      V = my_random_permutation(V);
      if(flag)
         LAMBDA = CreatLambda(V, epsilon);
      //cout<<"Lambda size: "<<LAMBDA.size()<<endl;
      //cout<<"Lambda: ";
      //for(int i=0;i<LAMBDA.size();i++)
      //{
        // cout<<LAMBDA[i]<<" ";
      //}
      //cout<<endl;
      M = FindM_lambda(V, S, f_S, LAMBDA);
      //cout<<"M: ";
      //for(int i=0;i<M.size();i++)
      //{
         //cout<<M[i]<<" ";
      //}
      //cout<<endl;
      vector<int> b(g.n, 0);
      vector<bool> B(LAMBDA.size(), false);
      B[0]=true;
      vector<int> B_count(LAMBDA.size(), 0);
      int lamda_sao = 0;
      //std::chrono::steady_clock::time_point begin3 = std::chrono::steady_clock::now();
      #pragma omp parallel for
      for (int t = 1; t < LAMBDA.size(); t++)
      {
         vector<node_id> T2 = getT(V, LAMBDA[t]);
         //cout<<"T2 size: "<<T2.size()<<endl;
         vector<node_id> T1 = getT(V, LAMBDA[t - 1]);
         //cout<<"T1 size: "<<T1.size()<<endl;
         vector<node_id> S_tmp=S;
         vector<bool> check_S_tmp=check_S;
         S_tmp.insert(S_tmp.end(), T1.begin(), T1.end());
         updateCheck(check_S_tmp,T1);
         vector<node_id> TT = getTT(T1, T2);
         //cout<<"TT size: "<<TT.size()<<endl;
         #pragma omp parallel for
         for (int j = 0; j < TT.size(); j++)
         {
            vector<node_id> S_tmp_tmp=S_tmp;
            vector<bool> check_S_tmp_tmp=check_S_tmp;
            vector<node_id> Tj1 = getT(V, T1.size() + j);
            //cout<<"Tj1 size: "<<Tj1.size()<<endl;
            S_tmp_tmp.insert(S_tmp_tmp.end(), Tj1.begin(), Tj1.end());
            updateCheck(check_S_tmp_tmp,Tj1);
            //cout<<"S_tmp_tmp size: "<<S_tmp_tmp.size()<<endl;
            double gain= marge(nEvals,g,TT[j],check_S_tmp_tmp);
            //cout<<"gain: "<<gain<<endl;
            if (gain >= myalpha*(1 - epsilon) * M[t - 1] / k)
            {
               #pragma omp critical
               {
                  b[TT[j]] = 1;
                  B_count[t]++;
               }
            }
            else if (gain < 0)
            {
               #pragma omp critical
               {
                  b[TT[j]] = -1;
               }
            }
         }
         if (B_count[t] >= (1 - epsilon) * TT.size())
         {
            #pragma omp critical
            {
               B[t] = true;
            }
         }
      }
      //std::chrono::steady_clock::time_point end3 = std::chrono::steady_clock::now();
      //size_t WallTimeMillis3 = std::chrono::duration_cast<std::chrono::milliseconds>(end3 - begin3).count();
      //double time3 = WallTimeMillis3 / 1000.0;
      //cout<<"Time parallel = "<<time3<<endl;
      //cout<<"B:";
      //for(int i=0;i<B.size();i++)
      //{
         //cout<<B[i]<<" ";
      //}
      //cout<<endl;
      int lamda = findLamda(B,k);
      //cout<<"lamda: "<<lamda<<endl;
      vector<node_id> T1;
      vector<node_id> T2 =getT(V,LAMBDA[lamda]);
      //cout<<"T2 size: "<<T2.size()<<endl;
      vector<node_id> Tj = getTTT(T1,T2,b);
      //cout<<"Tj size: "<<Tj.size()<<endl;
      len_s.push_back(Tj.size());
      flag=false;
      for (int i = 0; i < Tj.size(); i++)
      {
         S.push_back(Tj[i]);
         check_S[Tj[i]] = true;
         flag=true;
      }
      if(flag)
         f_S = compute_valSet(nEvals, g, S);
      if(S.size()>=k) break;
      //cout<<"f_S: "<<f_S<<endl;
   }
   //std::chrono::steady_clock::time_point end2 = std::chrono::steady_clock::now();
   //size_t WallTimeMillis2 = std::chrono::duration_cast<std::chrono::milliseconds>(end2 - begin2).count();
   //double time2 = WallTimeMillis2 / 1000.0;
   //cout<<"Time loop = "<<time2<<endl;
   //cout<<"f_S: "<<f_S<<endl;
   //cout<<"S size: "<<S.size()<<endl;
   int k_prime=0;
   if(S.size()<=k)
      k_prime=S.size();
   else
   {
      for(int i=len_s.size()-1;i>=0;i++)
      {
         if(k_prime + len_s[i]>k)
            break;
         else
            k_prime+=len_s[i];
      }
   }
   //cout<<"k_prime: "<<k_prime<<endl;
   for(int i=S.size()-1;i>=0;i--)
   {
      Sprime.push_back(S[i]);
      if(Sprime.size()>=k_prime) break;
   }
   //cout<<"Sprime size: "<<Sprime.size()<<endl;
   //std::chrono::steady_clock::time_point end = std::chrono::steady_clock::now();
   //size_t WallTimeMillis = std::chrono::duration_cast<std::chrono::milliseconds>(end - begin).count();
   //double time = WallTimeMillis / 1000.0;
   //cout<<"Time LinBoundedSet = "<<time<<endl;
   return compute_valSet(nEvals, g, Sprime);
}

int TSAtg::findLamda(vector<bool>& arr, int k)
{
   //if(arr.size()==0) return 1;
   //cout<<"B:";
   //for(int i=0;i<arr.size();i++)
   //{
      //cout<<arr[i]<<" ";
  // }
   //cout<<endl;
   int lamdaj1=-1;
   for (lamdaj1 = 0; lamdaj1 < arr.size(); lamdaj1++)
   {
      if (arr[lamdaj1] == false) break;
   }
   int arr_size=arr.size();
   if(lamdaj1>=arr_size) lamdaj1=arr_size-1;
   int start = -1;
   int len = 0;
   int last_index=-1;
   for (int i = 0; i < arr_size; i++)
   {
      if (arr[i])
      {
         if (start == -1)
            start = i; 
         len++;
      }
      else
      {
         if (len >= k) last_index=i; 
         start = -1;
         len = 0;
      }
  }
  if( last_index >= arr_size)
  {
      last_index=arr_size-1;
   }
  return max(lamdaj1,last_index);
}
vector<node_id> TSAtg::getTTT(vector<node_id> T1, vector<node_id> T2, vector<int> b)
{
   vector<node_id> T;
   for (int j = T1.size(); j < T2.size(); j++)
   {
      if (b[T2[j]] != -1)
         T.push_back(T2[j]);
   }
   return T;
}
vector<node_id> TSAtg::getT(vector<node_id> V, int position)
{
   vector<node_id> T;
   if (position <= 0) return T;
   for (int j = 0; j < position; j++)
   {
      T.push_back(V[j]);
   }
   return T;
}
vector<node_id> TSAtg::getTT(vector<node_id> T1, vector<node_id> T2)
{
   vector<node_id> T;
   for (int j = T1.size(); j < T2.size(); j++)
   {
      T.push_back(T2[j]);
   }
   return T;
}
double TSAtg::FindEmax(vector<node_id> V, vector<node_id> &S, vector<bool> &check_S, int& adapt)
{
   //std::chrono::steady_clock::time_point begin2 = std::chrono::steady_clock::now();
   double f_S = 0;
   for(int i=0;i<5;i++)
   {
      adapt++;
      node_id emax = -1;
      f_S=-1;
      #pragma omp parallel for
      for (int j=0;j<V.size();j++)
      {
         node_id e=V[j];
         if(check_S[e]) continue;       
         double f = marge(nEvals,g,e,check_S);
         if (f > f_S)
         {
            #pragma omp critical
            {
               f_S = f;
               emax = e;
            }
         }
      }
      check_S[emax] = true;
      S.push_back(emax);
   }
   //std::chrono::steady_clock::time_point end2 = std::chrono::steady_clock::now();
   //size_t WallTimeMillis2 = std::chrono::duration_cast<std::chrono::milliseconds>(end2 - begin2).count();
   //double time2 = WallTimeMillis2 / 1000.0;
   //cout<<"Time Emax = "<<time2<<endl;
   return compute_valSet(nEvals, g, S);
}
vector<double> TSAtg::FindM_lambda(vector<node_id> V, vector<node_id> S, double f_S, vector<size_t> LAMBDA)
{
   //std::chrono::steady_clock::time_point begin2 = std::chrono::steady_clock::now();
   double f_s= compute_valSet(nEvals, g, S);
   vector<double> M(LAMBDA.size(),0);
   #pragma omp parallel for
   for (int i = 0; i < LAMBDA.size(); i++)
   {
      /*double max_f = 0;
      #pragma omp parallel for
      for (int j = 0; j <= i; j++)
      {
         vector<node_id> SS = S;
         vector<node_id> T = getT(V, LAMBDA[j]);
         SS.insert(SS.end(), T.begin(), T.end());
         double f = compute_valSet(nEvals, g, SS);
         if (f > max_f)
            max_f = f;
      }*/
      vector<node_id> SS = S;
      vector<node_id> T = getT(V, LAMBDA[i]);
      SS.insert(SS.end(), T.begin(), T.end());
      double f = compute_valSet(nEvals, g, SS);
      M[i]=f;
   }
   //std::chrono::steady_clock::time_point end2 = std::chrono::steady_clock::now();
   //size_t WallTimeMillis2 = std::chrono::duration_cast<std::chrono::milliseconds>(end2 - begin2).count();
   //double time2 = WallTimeMillis2 / 1000.0;
   //cout<<"Time FindM_Lambda = "<<time2<<endl;
   return M;
}
vector<size_t> TSAtg::CreatLambda(vector<node_id> V,double epsilon)
{
   vector<size_t> LAMBDA;
   LAMBDA.push_back(0);
   int l = 1;
   while (pow((1 + epsilon), l) <= k)
   {
      LAMBDA.push_back(floor(pow(1 + epsilon, l)));
      l++;
   }
   l = 0;
   while (k + l * epsilon * k < V.size())
   {
      LAMBDA.push_back(floor(k + l * epsilon * k));
      l++;
   }
   LAMBDA.push_back(V.size());
   std::sort(LAMBDA.begin(), LAMBDA.end());
   LAMBDA.erase(std::unique(LAMBDA.begin(), LAMBDA.end()), LAMBDA.end());
   vector<size_t> LB;
   for(int i=0;i<LAMBDA.size();i++)
   {
      if(LAMBDA[i]<=V.size())
         LB.push_back(LAMBDA[i]);
   }
   return LB;
}
vector<node_id> TSAtg::updateV(vector<node_id> V, double f_S, vector<bool> check_S)
{
   //std::chrono::steady_clock::time_point begin = std::chrono::steady_clock::now();
   vector<node_id> Vn;
   #pragma omp parallel for
   for (int i=0; i<V.size();i++)
   {
      node_id e=V[i];
      if(check_S[e]==false && marge(nEvals,g,e,check_S) >= myalpha*f_S / k)
      {
         #pragma omp critical
         {
            Vn.push_back(e);
         }
      }
   }
   //std::chrono::steady_clock::time_point end = std::chrono::steady_clock::now();
   //size_t WallTimeMillis = std::chrono::duration_cast<std::chrono::milliseconds>(end - begin).count();
   //double time = WallTimeMillis / 1000.0;
   //cout<<"Time UpdateV = "<<time<<endl;
   return Vn;
}

void TSAtg::merge(vector<node_id> &A, vector<node_id> B)
{
   for (int j = 0; j < B.size(); j++)
   {
      A.push_back(B[j]);
   }
}
vector<node_id> TSAtg::getV(vector<bool> check_V)
{
   vector<node_id> V;
   for (node_id i = 0; i < check_V.size(); i++)
   {
      if (check_V[i] == true)
         V.push_back(i);
   }
   return V;
}
   vector<node_id> ParSKP::GetSEQ(vector<node_id> const A, vector<node_id> II, double c_A)
   {
      vector<node_id> I = II;
      std::random_device rd;
      std::mt19937 rng(rd());
      vector<node_id> V;
      double c_V = 0;
      while (true)
      {
         if (I.size() == 0)
            break;
         std::shuffle(I.begin(), I.end(), rng);
         vector<bool> check_I(I.size(), true);
         for (int i = 0; i < I.size(); i++)
         {
            if (c_A + c_V + 1.0 <= B)
            {
               V.push_back(I[i]);
               c_V += 1.0;
               check_I[i] = false;
            }
            else
               break;
         }
         int l = I.size();
         vector<node_id> I2;
         for (int i = l - 1; i >= 0; i--)
         {
            if (check_I[i] == true && (c_A + c_V + 1 <= B))
            {
               I2.push_back(I[i]);
            }
         }
         I = I2;
      }
      return V;
   }
   vector<node_id> ParSKP::RandBatch(double rho, vector<node_id> I, vector<node_id> const TT, int M, double p, vector<node_id> &Ui, vector<node_id> &Li, size_t &number_q, int &adaptive)
   {
      vector<node_id> T = TT;
      std::random_device rd;
      std::mt19937 rng(rd());
      vector<node_id> A, U, L;
      double c_A = T.size();
      double f_current_a = compute_valSet(number_q, g, A, T);
      vector<bool> check_U(g.n, false);
      vector<bool> check_A(g.n, false);
      vector<bool> check_L(g.n, false);
      int count = 0;

      adaptive++;
      #pragma omp parallel for
      for (int i = 0; i < I.size(); i++)
      {
         if (c_A + 1.0 <= B)
         {
            vector<node_id> tmp = A;
            tmp.push_back(I[i]);
            double f_tmp = compute_valSet(number_q, g, tmp, T);
            if ((f_tmp - f_current_a) / 1.0 >= rho)
               check_L[i] = true;
         }
      }
      for (int i = 0; i < I.size(); i++)
      {
         if (check_L[i] == true)
            L.push_back(I[i]);
      }
      vector<node_id> V_d;
      while (L.size() > 0 && count < M)
      {
         double c_L = L.size();
         V_d = GetSEQ(A, L, c_A);
         if (V_d.size() == 0)
            break;
         double f_current_d = f_current_a;
         int t1 = -1, t2 = -1;
         for (int i = 0; i < V_d.size(); i++)
         {
            vector<node_id> Gi = A;
            double c_Gi = c_A;
            vector<bool> check_Gi(g.n, false);
            for (int j = 0; j <= i; j++)
            {
               Gi.push_back(V_d[j]);
               check_Gi[V_d[j]] = true;
               c_Gi += 1.0;
            }
            double f_current_Gi = compute_valSet(number_q, g, Gi, T);
            double c_Eplus = 0, c_Esub = 0, d_Eplus = 0, d_Esub = 0;
            adaptive++;
            for (int j = 0; j < L.size(); j++)
            {
               if (check_Gi[L[i]] == false)
               {
                  Gi.push_back(L[j]);
                  double f_tmp = compute_valSet(nEvals, g, Gi, T);
                  Gi.pop_back();

                  if (f_tmp - f_current_Gi < 0)
                  {
                     c_Esub += 1.0;
                     d_Esub += std::abs(f_tmp - f_current_Gi);
                  }
                  else if ((f_tmp - f_current_Gi) >= rho && c_Gi + 1.0 <= B)
                  {
                     c_Eplus += 1.0;
                     d_Eplus += f_tmp - f_current_Gi;
                  }
               }
            }
            adaptive++;
            double d_D = 0;
            double f_current_dd = f_current_a;
            for (int j = 1; j <= i; j++)
            {
               vector<node_id> Vj;
               for (int k = 0; k <= j - 1; k++)
               {
                  Vj.push_back(V_d[k]);
               }
               vector<node_id> DD = A;
               DD.insert(DD.end(), Vj.begin(), Vj.end());
               double f_tmp_dd = compute_valSet(nEvals, g, DD, T);
               DD.push_back(V_d[j]);
               double f_tmp_dd2 = compute_valSet(nEvals, g, DD, T);
               if (f_tmp_dd2 - f_tmp_dd < 0)
                  d_D += std::abs(f_tmp_dd - f_current_dd);
            }
            if (c_Eplus > 0 && c_Eplus <= (1 - epsi) * c_L)
               t1 = i;
            if (d_Eplus > 0 && epsi * d_Eplus <= d_Esub + d_D)
               t2 = i;
            if (t1 != -1 || t2 != -1)
               break;
         }
         if (t1 == -1 && t2 == -1)
            break;
         for (int i = 0; i <= get_min(t1, t2); i++)
         {
            U.push_back(V_d[i]);
            check_U[V_d[i]] = true;
         }
         std::uniform_real_distribution<double> distribution(0.0, 1.0);
         double p2 = distribution(rng);
         if (p2 <= p)
         {
            for (int i = 0; i <= get_min(t1, t2); i++)
            {
               A.push_back(V_d[i]);
               check_A[V_d[i]] = true;
               c_A += 1.0;
            }
            f_current_a = compute_valSet(nEvals, g, A, T);
            if (t2 < t1)
               count++;
         }
         vector<node_id> temp_L;
         adaptive++;
         for (int i = 0; i < L.size(); i++)
         {
            if (check_U[L[i]] == false && c_A + 1.0 <= B)
            {
               A.push_back(L[i]);
               double f_tmp_a = compute_valSet(nEvals, g, A, T);
               A.pop_back();
               if ((f_tmp_a - f_current_a) >= rho)
                  temp_L.push_back(L[i]);
            }
         }
         L = temp_L;
      }
      Ui = U;
      Li = L;
      return A;
   }
   int ParSKP::get_min(int t1, int t2)
   {
      if (t1 == -1 && t2 == -1)
         return -1;
      if (t1 == -1)
         return t2;
      if (t2 == -1)
         return t1;
      return min(t1, t2);
   }
   double ParSKP::USM(vector<node_id> S)
   {
      vector<node_id> X;
      std::default_random_engine generator;
      std::uniform_real_distribution<float> distribution(0.0, 1.0);
      double c = 0;
      for (auto elem : S)
      {
         if (distribution(generator) <= 0.5)
         {
            if (c + 1.0 <= B)
            {
               X.push_back(elem);
               c += 1.0;
            }
         }
      }
      return compute_valSet(nEvals, g, X);
   }
   double ParSKP::Probe(double rho, vector<node_id> N1, vector<node_id> N2, size_t &number_q, int &adaptive)
   {
      double f_max = 0;
      vector<node_id> T;
      int M = static_cast<int>(std::ceil(1.0 / (epsi * epsi)));
      double p = 1;
      vector<node_id> I = N1;
      vector<bool> checkI(g.n, false);
      for (int i = 0; i < I.size(); i++)
      {
         checkI[I[i]] = true;
      }
      vector<node_id> Ui, Li;
      vector<node_id> temp;
      vector<node_id> A1 = RandBatch(rho, I, temp, M, p, Ui, Li, number_q, adaptive);
      double c_A1 = A1.size();
      for (int i = 0; i < A1.size(); i++)
      {
         checkI[A1[i]] = false;
      }
      I.clear();
      for (int i = 0; i < g.n; i++)
      {
         if (checkI[i] == true)
            I.push_back(i);
      }
      vector<node_id> Ui2, Li2;
      vector<node_id> A2 = RandBatch(rho, I, temp, M, p, Ui2, Li2, number_q, adaptive);
      double c_A2 = A2.size();
      vector<bool> checkA1A2(g.n, false);
      for (int j = 0; j < A1.size(); j++)
      {
         checkA1A2[A1[j]] = true;
      }
      for (int j = 0; j < A2.size(); j++)
      {
         checkA1A2[A2[j]] = true;
      }
      double max_f_A1_i = 0;
      adaptive++;
      for (int i = 0; i < N1.size(); i++)
      {
         if (checkA1A2[i] == false && c_A1 + 1.0 <= B)
         {
            vector<node_id> tmp = A1;
            tmp.push_back(N1[i]);
            double f = compute_valSet(number_q, g, tmp);
            if (f > max_f_A1_i)
               max_f_A1_i = f;
         }
      }
      adaptive++;
      double max_f_A2_i = 0;
      for (int i = 0; i < N1.size(); i++)
      {
         if (checkA1A2[i] == false && c_A2 + 1.0 <= B)
         {
            vector<node_id> tmp = A2;
            tmp.push_back(N1[i]);
            double f = compute_valSet(number_q, g, tmp);
            if (f > max_f_A2_i)
               max_f_A2_i = f;
         }
      }
      f_max = max(max_f_A2_i, max_f_A1_i);
      vector<node_id> A3 = A1;
      double c_A3 = c_A1;
      for (int i = 0; i < N2.size(); i++)
      {
         A3.push_back(N2[i]);
         c_A3 += 1.0;
      }
      if (c_A3 <= B)
      {
         double f = USM(A3);
         f_max = max(f_max, f);
      }
      return f_max;
   }
   double ParSKP::runParSKP2()
   {
      if (alpha.size() == 0)
      {
         init_alpha(g);
      }
      double p = 0.5;
      int adaptive = 0;
      nEvals = 0;
      vector<node_id> N1, N2;
      double f_emax1 = 0, f_emax2 = 0, f_max = 0, alpha = 1;
      int e_max1 = -1, e_max2 = -1;
      for (int i = 0; i < g.n; i++)
      {
         if (1.0> B / g.n)
            N1.push_back(i);
         else
            N2.push_back(i);
      }
      adaptive++;
      #pragma omp parallel for
      for (node_id i = 0; i < g.n; i++)
      {
         double tmp_f=0;
         vector<node_id> tmp;
         tmp.push_back(i);
         tmp_f = compute_valSet(nEvals, g, tmp);
         if (tmp_f > f_emax1)
         {
            #pragma omp critical
            {
               f_emax1 = tmp_f;
               e_max1 = i;
               f_emax2 = tmp_f;
               e_max2 = i;
            }
         }          
         
      }
      adaptive++;
      f_max = USM(N2);
      f_max = max(f_max, f_emax2);
      vector<node_id> I = N1;
      vector<node_id> T;
      int M = ceil(((log(epsi / g.n) / log(1 - epsi)) + 2) / (epsi * epsi));
      double pmax = f_emax1;
      int l = ceil((log((epsi * 1.0) / B) / log(1 - epsi))) + 1;
      for (int i = 1; i <= l; i++)
      {
         double pi = pmax * pow(1 - epsi, i - 1);
         vector<node_id> Ui, Li;
         vector<node_id> Ai = RandBatch(pi, I, T, M, p, Ui, Li, nEvals, adaptive);
         T.insert(T.end(), Ai.begin(), Ai.end());
         vector<node_id> Ii;
         for (const node_id &item : I)
         {
            if (std::find(Ui.begin(), Ui.end(), item) == Ui.end() && std::find(Li.begin(), Li.end(), item) == Li.end())
               Ii.push_back(item);
         }
         I = Ii;
      }
      double f_T = compute_valSet(nEvals, g, T);
      f_max = max(max(f_T, f_max), f_emax2);
      reportResults(nEvals, f_max, adaptive);
      return f_max;
   }