#ifndef ALGS_CPP
#define ALGS_CPP

#include "mygraph.h"
#include "logger.h"
#include "ThreadPool.h"
#include <algorithm>
#include <set>
#include <map>
#include <string>
#include <vector>
#include <fstream>
#include <thread>
#include <pthread.h>
#include <omp.h>

using namespace std;
using namespace mygraph;

extern resultsHandler allResults;
extern vector<double> alpha;
extern vector<bool> emptyBoolVector;
extern vector<bool> emptySetVector;
extern vector<double> emptyDoubleVector;
extern vector<size_t> emptySize_tVector;

enum Algs
{
   IG = 0,// InterlaceGreedy
   FRG = 1, // FastRandomGreedy
   ANM = 2, // AdaptiveNonomonotoneMax
   ENE = 3, // Ene et al. 2020
   TSAST = 4,// TSAst (Adaptive Simple Threshold) Kunhle
   TSATG = 5,// TSAtg (Adaptive Threshold Greedy) Kuhnle 
   LinAtg = 6, // ALG4 (this work)
   BoostAdapt = 7, // ALG4 (this work)
   ParSKP2 =8,// ParSKP2 (Cui et al., 2023)
   LinAst =9 
};

struct Args
{
   Algs alg;
   string graphFileName;
   string outputFileName = "";
   size_t k;
   tinyGraph g;
   double tElapsed;
   double wallTime;
   Logger logg;
   bool steal = true;
   double epsi = 0.1;
   double delta = 0.1;
   size_t N = 1;
   bool fast = false;
   bool reportRounds = true;
   size_t nSamps = 0;
   size_t nThreads = 40;
   bool printStdDev = true;
   bool a = true; // if true, run AST with M=max f(x)/(ck), otherwise, M=max f(x)/k
   string sAlg;
};

class MyPair
{
public:
   node_id u;
   double gain; //may be negative

   MyPair() {}
   MyPair(node_id a,
          int64_t g)
   {
      u = a;
      gain = g;
   }

   MyPair(const MyPair &rhs)
   {
      u = rhs.u;
      gain = rhs.gain;
   }

   void operator=(const MyPair &rhs)
   {
      u = rhs.u;
      gain = rhs.gain;
   }
};

struct gainLT
{
   bool operator()(const MyPair &p1, const MyPair &p2)
   {
      return p1.gain < p2.gain;
   }
};

extern gainLT gainLTobj;

struct revgainLT
{
   bool operator()(const MyPair &p1, const MyPair &p2)
   {
      return (p1.gain > p2.gain);
   }
};
extern revgainLT revgainLTobj;

void init_alpha(tinyGraph &g);

extern uniform_real_distribution<double> unidist;

double reduced_mean_dagum(size_t &nEvals,
                          tinyGraph &g,
                          vector<size_t> &idsA,
                          vector<size_t> &idsS,
                          size_t t,
                          double tau,
                          double epsi,
                          double delta,
                          bool fast = false);
#ifdef IMG
signed long marge(size_t &nEvals, tinyGraph &g, node_id u, vector<bool> &set);
size_t compute_valSet(size_t &nEvals, tinyGraph &g, vector<bool> &set);
double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &set_id);
double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &set_id, vector<node_id> &set_id2);
#endif
#ifdef REVMAX
signed long marge(size_t &nEvals, tinyGraph &g, node_id u, vector<bool> &set);
size_t compute_valSet(size_t &nEvals, tinyGraph &g, vector<bool> &set);
#endif
#ifdef MAXCUT
double compute_valSet(size_t &nEvals, tinyGraph &g, vector<bool> &set,
                      vector<bool> &cov = emptySetVector);
double compute_valSet(size_t &nEvals, tinyGraph &g, vector<node_id> &sset);
// size_t compute_valSet(size_t &nEvals, tinyGraph &g, vector<bool> &set);
double marge(size_t &nEvals, tinyGraph &g, node_id x, vector<bool> &set,
             vector<bool> &cov = emptySetVector);
#endif

void eval_multi_worker(vector<double> &x, tinyGraph &g, double &val, size_t nsamps, mt19937 &gen);

double eval_multilinear_base(size_t &nEvals, tinyGraph &g, vector<double> &x, size_t nsamps, size_t nThreads, vector<mt19937> &vgens);

void reportResults(size_t nEvals, double obj, size_t rounds = 0);

void filter(size_t &nEvals,
            tinyGraph &g,
            vector<bool> &A,
            vector<bool> &S,
            double tau,
            vector<size_t> &idsA);

void random_set(tinyGraph &g, vector<bool> &C, vector<size_t> &A);

void random_set(tinyGraph &g, vector<size_t> &C, vector<size_t> &A);

void unc_max(size_t &nEvals, tinyGraph &g, vector<size_t> &A, double epsi, double delta, vector<size_t> &result);

void sampleUt(vector<size_t> &R, vector<size_t> &A, size_t t);

unsigned samp_Dt(size_t &nEvals,
                 tinyGraph &g,
                 vector<size_t> &idsA,
                 vector<size_t> &idsS,
                 size_t t,
                 double tau);

double reduced_mean_chernoff(size_t &nEvals,
                             tinyGraph &g,
                             vector<size_t> &idsA,
                             vector<size_t> &idsS,
                             size_t t,
                             double tau,
                             double epsi,
                             double delta,
                             bool fast = false);

void report_rounds(vector<bool> &S,
                   vector<double> &valRounds,
                   tinyGraph &g);

void threshold_sample(size_t &nEvals,
                      tinyGraph &g,
                      vector<bool> &S, // sampled set, may not be empty
                      vector<size_t> &idsS,  // index of sampled set
                      vector<bool> &exclude, // exclude set
                      bool bexcl,            // if exclude set
                      size_t k,
                      double tau,
                      double epsi,
                      double delta,
                      bool fast,
                      bool reportRounds,
                      vector<double> &valRounds);

void threshold_seq_ama(size_t &nEvals,
                      tinyGraph &g,
                      vector<bool> &S, // sampled set
                      vector<size_t> &idsS,  // index of sampled set
                      vector<bool> &exclude, // exclude set
                      bool bexcl,            // if exclude set
                      size_t k,
                      double tau,
                      double epsi,
                      double delta,
                      bool fast,
                      bool reportRounds,
                      vector<double> &valRounds);

void threshold_seq_ama2(size_t &nEvals,
                      tinyGraph &g,
                      vector<bool> &S, // sampled set
                      vector<size_t> &idsS,  // index of sampled set
                      vector<bool> &exclude, // exclude set
                      bool bexcl,            // if exclude set
                      size_t k,
                      double tau,
                      double epsi,
                      double delta,
                      bool fast,
                      bool reportRounds,
                      vector<double> &valRounds);

void threshold_seq(size_t &nEvals,
                   tinyGraph &g,
                   vector<bool> &A,           // sampled set A, may not be empty
                   vector<bool> &Aprime,      // sampled set A', may not be empty
                   vector<size_t> &idsA,      // index of sampled set A
                   vector<size_t> &idsAprime, // index of sampled set A'
                   vector<bool> &exclude, // exclude set
                   bool bexcl,            // if exclude set
                   size_t k,
                   double tau,
                   double epsi,
                   double delta,
                   bool fast,
                   bool reportRounds,
                   vector<double> &valRounds);

class Ig
{
   size_t k;
   tinyGraph &g;
   bool steal;
   size_t nEvals = 0;

public:
   Ig(Args &args) : g(args.g)
   {
      k = args.k;
      steal = args.steal;
   }

   double leastBenefit(node_id u, vector<bool> &set);
   void run();
};

class Fig
{
   size_t k;
   tinyGraph &g;
   bool steal;
   size_t nEvals = 0;
   double epsi;
   double stopGain;
   bool reportRounds = false;

public:
   Fig(Args &args) : g(args.g)
   {
      k = args.k;
      steal = args.steal;
      epsi = args.epsi;
      reportRounds = args.reportRounds;
   }
   double leastBenefit(node_id u, vector<bool> &set, vector<double> &valRounds);
   bool swap(node_id u, node_id v, vector<bool> &set);
   size_t sizeSet(vector<bool> &S);
   void add(vector<bool> &S, vector<bool> &T, node_id &j, double &tau, vector<double> &valRounds);
   void run();
};

class Rg
{
   size_t k;
   tinyGraph &g;
   size_t nEvals = 0;
   bool reportRounds = false;

public:
   Rg(Args &args) : g(args.g)
   {
      k = args.k;
      reportRounds = args.reportRounds;
   }
   double leastBenefit(node_id u, vector<bool> &set);
   void run();
};

class Frg
{
   Args &myArgs;
   size_t k;
   tinyGraph &g;
   size_t nEvals = 0;
   double epsi;
   double w;
   double W;

public:
   Frg(Args &args) : myArgs(args), g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
   }
   double leastBenefit(node_id u, vector<bool> &set);
   void fillM(vector<node_id> &M, vector<bool> &S);
   void run();
   void randomSampling(double p, size_t s, vector<bool> &A);
   void sampleUnifSize(vector<bool> &R, size_t Size);
   void runRandom();
   void runSimple();
   void runImproved();
};

class Sg
{
   size_t k;
   tinyGraph &g;
   size_t nEvals = 0;
   bool reportRounds = false;

public:
   Sg(Args &args) : g(args.g)
   {
      k = args.k;
      reportRounds = args.reportRounds;
   }
   long leastBenefit(node_id u, vector<bool> &set);
   void run();
};

class Anm
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t r;
   double OPT; //guess for opt
   size_t nEvals = 0;
   bool reportRounds = false;
   bool fast = false;
   Anm(Args &args) : g(args.g)
   {
      k = args.k;
      //OPT = 130180;
      OPT = g.m;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      g.logg << "AdaptiveNonmonotoneMax initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL << INFO;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void run();
};

class Ast
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t nEvals = 0;
   bool fast = false;
   bool reportRounds = false;
   bool a = false;
   Ast(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      a = args.a;
      g.logg << "AST initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL << INFO;
      }
      else
      {
         delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void run();
};

class TSAst
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   // size_t r;
   // size_t nSamps = 30;
   size_t nEvals = 0;
   bool fast = false;
   bool reportRounds = false;
   bool a = false;

   TSAst(Args &args) : g(args.g)
   {
      k = args.k;
      //OPT = 130180;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      a = args.a;
      g.logg << "TSAST initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL << INFO;
      }
      else
      {
         delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void run();
};

class AmaAst
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t nEvals = 0;
   bool fast = false;
   bool reportRounds = false;
   bool a = false;
   AmaAst(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      a = args.a;
      g.logg << "AMAAST initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL << INFO;
      }
      else
      {
         delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void run();
};

class AmaAst2
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t nEvals = 0;
   bool fast = false;
   bool reportRounds = false;
   bool a = false;
   AmaAst2(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      a = args.a;
      g.logg << "AMAAST initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL << INFO;
      }
      else
      {
         delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void run();
};

class Ene
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t rounds = 0;
   double OPT; //guess for opt
   size_t nSamps = 10000;
   size_t nEvals = 0;
   bool fast = false;
   bool reportRounds = false;
   bool print = false;
   size_t nThreads;
   vector<mt19937> vgens;
   Ene(Args &args) : g(args.g)
   {
      k = args.k;
      //OPT = 130180;
      OPT = g.m;
      epsi = args.epsi;
      fast = args.fast;
      reportRounds = args.reportRounds;
      //delta = pow(epsi, 4) / (log( g.n) * log( 1.0 / epsi ) );
      //delta = pow(epsi, 4) / (log( g.n) * log( 1.0 / epsi ) ) * 10;
      delta = pow(epsi, 3);

      nSamps = args.nSamps;
      nThreads = args.nThreads;
      g.logg << "Ene initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      g.logg << "nSamps=" << nSamps << endL;

      if (nSamps == 0)
      {
         g.logg << "Computing multilinear extension exactly..." << endL;
      }

      nThreads = 1;

      std::random_device r;
      for (size_t i = 0; i < nThreads; ++i)
      {
         mt19937 gen(r());
         vgens.push_back(gen);
      }
   }
   double onenorm(vector<double> &x);
   double evalMultilinear(size_t &nEvals, vector<double> &x, size_t nsamps = 0);
   double evalMultilinearSample(size_t &nEvals, vector<double> &x, size_t nsamps = 0);
   double evalMultilinearOld(size_t &nEvals, vector<double> &x, size_t nsamps = 0);
   void gradientMultilinear(size_t &nEvals, vector<double> &gradient, vector<double> &x, size_t nsamps = 0);
   void gradientMultilinearSample(size_t &nEvals, vector<double> &gradient, vector<double> &x);
   void computeTeta(size_t &nEvals,
                    double eta,
                    double v,
                    vector<double> &z,
                    vector<node_id> &S,
                    vector<node_id> &Seta,
                    vector<node_id> &Teta,
                    size_t j,
                    vector<double> &zstart,
                    size_t idxthread,
                    vector<double> &vec_g);
   bool evalEta(size_t &nEvals, double eta, double epsi, double v,
                vector<double> &z, vector<node_id> &S, size_t j, vector<double> &zstart, size_t threadindex,
                vector<double> &vec_g);
   double findeta1(size_t &nEvals, double v, vector<double> &z, vector<double> &vec_g, vector<node_id> &S, size_t j, vector<double> &zstart, size_t threadindex);
   double findeta2(double epsi, size_t j, size_t k, vector<double> &z, vector<node_id> &S);
   void runM(size_t &rounds, size_t &nEvals, double epsi, double M, double &valout, size_t i);
   void run();
};

class Atg
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t nEvals = 0;
   bool fast = false;
   size_t c = 0;
   bool reportRounds = false;
   Atg(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      g.logg << "ATG initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL;

         c = 1 / epsi;
      }
      else
      {
         c = 8 / epsi;
         epsi = 0.63 * epsi / 8;
         delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void run();
};

class TSAtg
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   //double delta;
   size_t nEvals = 0;
   bool fast = false;
   size_t c = 0;
   bool reportRounds = false;
   double myalpha=4.0;
   double beta=4.0;
   double delta= 1.0/3.0;
   TSAtg(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      //delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      //g.logg << "TSATG initialized:" << endL;
      //g.logg << "epsi=" << epsi << endL;
      //g.logg << "delta=" << delta << endL;
      //g.logg << "k=" << k << endL;
      if (fast)
      {
         //g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL;

         c = 1 / epsi;
      }
      else
      {
         c = 8 / epsi;
         epsi = 0.63 * epsi / 8;
         // delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void updateCheck(vector<bool>& check, vector<node_id> A);
   vector<node_id> my_random_permutation(vector<node_id> V);
   double LinBoundedSet(vector<node_id> V, vector<node_id> &Sprime, double epsilon, double delta, int &adapt);
   double LinAdapt(double epsilon, double delta, int &adapt);
   vector<node_id> updateV(vector<node_id> V, double f_S, vector<bool> check_S);
   vector<node_id> updateV(vector<node_id> V,vector<node_id> A, double f_A, double tau);
   vector<size_t> CreatLambda(vector<node_id> V,double epsilon);
   vector<double> FindM_lambda(vector<node_id> V, vector<node_id> S, double f_S, vector<size_t> LAMBDA);
   double FindEmax(vector<node_id> V, vector<node_id>& S, vector<bool>& check_S, int& adapt);
   vector<node_id> getT(vector<node_id> V,int position);
   vector<node_id> getTT(vector<node_id> T1, vector<node_id> T2);
   vector<node_id> getTTT(vector<node_id> T1, vector<node_id> T2, vector<int> b);
   void merge(vector<node_id>& A, vector<node_id> B);
   vector<node_id> getV(vector<bool> check_V);
   int findLamda(vector<bool>& arr, int k);
   int FindKPrime(int k,double epsi);
   double USM(vector<node_id> A, double epsilon, double delta);
   void run();
   void runBoostAdapt();
   void runLinAtg();
   void runLinAst();
};

class AmaAtg
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t nEvals = 0;
   bool fast = false;
   size_t c = 0;
   bool reportRounds = false;
   AmaAtg(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      g.logg << "AMAATG initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL;

         c = 1 / epsi;
      }
      else
      {
         c = 8 / epsi;
         epsi = 0.63 * epsi / 8;
         delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void run();
};

class AmaAtg2
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t nEvals = 0;
   bool fast = false;
   size_t c = 0;
   bool reportRounds = false;
   AmaAtg2(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      g.logg << "AMAATG initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL;
         c = 1 / epsi;
      }
      else
      {
         c = 8 / epsi;
         epsi = 0.63 * epsi / 8;
         delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   void run();
};
class LBS
{
public:
   size_t k;
   tinyGraph &g;
   double epsi;
   double delta;
   size_t nEvals = 0;
   bool fast = false;
   size_t c = 0;
   bool reportRounds = false;
   LBS(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      delta = args.delta;
      fast = args.fast;
      reportRounds = args.reportRounds;
      g.logg << "LBS initialized:" << endL;
      g.logg << "epsi=" << epsi << endL;
      g.logg << "delta=" << delta << endL;
      g.logg << "k=" << k << endL;
      if (fast)
      {
         g.logg << WARN << "Fast mode enabled. Theoretical guarantees will not hold!" << endL;
         c = 1 / epsi;
      }
      else
      {
         c = 8 / epsi;
         epsi = 0.63 * epsi / 8;
         delta = delta / 2;
      }
   }
   size_t get_size_set(vector<bool> &S);
   vector<node_id> my_random_permutation(vector<node_id> V);
   vector<node_id> ThreshSeq(vector<node_id> V,int k, double epsi, double delta, double tau, vector<node_id>& AA, int &numDaptive);
   double LinBoundedSet(vector<node_id> V);
   double LinAdapt(vector<node_id> V);
   vector<node_id> updateV(vector<node_id> V, vector<node_id> S, double f_S);
   vector<node_id> updateV(vector<node_id> V,vector<node_id> A, double f_A, double tau);
   vector<size_t> CreatLambda(vector<node_id> V);
   vector<double> FindM_lambda(vector<node_id> V, vector<node_id> S, double f_S, vector<size_t> LAMBDA);
   double FindEmax(vector<node_id> V, vector<node_id>& S);
   vector<node_id> getT(vector<node_id> V,int position);
   vector<node_id> getTT(vector<node_id> T1, vector<node_id> T2);
   vector<node_id> getTTT(vector<node_id> T1, vector<node_id> T2, vector<int> b);
   void merge(vector<node_id>& A, vector<node_id> B);
   vector<node_id> getV(vector<bool> check_V);
   int findLamda(vector<bool>& arr, int k);
   void run(double ell, double alpha);
};
class Blits
{
   size_t k;
   tinyGraph &g;
   double epsi;
   size_t r;
   double OPT; //guess for opt
   size_t nSamps = 100;
   size_t nEvals = 0;

public:
   Blits(Args &args) : g(args.g)
   {
      k = args.k;
      //OPT = 130180;
      OPT = g.m;
      epsi = args.epsi;
      r = 10; //20/ epsi * log( g.n ) / log ( 1 + epsi /2 );
      g.logg << "Blits initialized, r = " << r << endL;
   }
   size_t sizeSet(vector<bool> &S);
   void sampleUX(vector<bool> &R, vector<bool> &X);
   double Delta(node_id a, vector<bool> &S, vector<bool> &X);
   double exMarge(vector<bool> &S, vector<bool> &Xplus, vector<bool> &X);
   bool sieve(vector<bool> &res, vector<bool> &A, size_t i, size_t &rounds);
   double leastBenefit(node_id u, vector<bool> &set);
   void setunion(vector<bool> &res, vector<bool> &set1, vector<bool> &set2);
   void setintersection(vector<bool> &res, vector<bool> &set1, vector<bool> &set2);
   void run();
};

class Tg
{
   size_t k;
   tinyGraph &g;
   size_t nEvals = 0;
   bool reportRounds = false;

public:
   Tg(Args &args) : g(args.g)
   {
      k = args.k;
      reportRounds = args.reportRounds;
   }
   double leastBenefit(node_id u, vector<bool> &set);
   void uncMax(vector<bool> &set, vector<double> &valRounds);
   void run();
};

class Rand
{
public:
   size_t k;
   tinyGraph &g;

   Rand(Args &args) : g(args.g)
   {
      k = args.k;
   }
   void run();
};

class ParSKP
{
   random_device rd;
   mt19937 gen;
   double epsi, T;
   size_t k, nEvals = 0;
   tinyGraph &g;
   double B, skp1 = 0;
   int skp1_adaptive = 0;
   int skp1_total_adaptive = 0;
   size_t skp1_query = 0;
   int run_skp1;

public:
   ParSKP(Args &args) : g(args.g)
   {
      k = args.k;
      epsi = args.epsi;
      B = args.k;
   }
   vector<node_id> GetSEQ(vector<node_id> const A, vector<node_id> II, double c_A);
   vector<node_id> RandBatch(double rho, vector<node_id> I, vector<node_id> const TT, int M, double p, vector<node_id> &Ui, vector<node_id> &Li, size_t &number_q, int &adaptive);
   int get_min(int t1, int t2);
   double USM(vector<node_id> S);
   double Probe(double rho, vector<node_id> N1, vector<node_id> N2, size_t &number_q, int &adaptive);
   double runParSKP2();
};
#endif