#include "mygraph.h"
#include "algs.h"

#include <iostream>
#include <string.h>
#include <unistd.h>
#include <chrono>
#include <getopt.h>
#include <vector>
#include <pthread.h>

using namespace mygraph;
using namespace std;

void print_help()
{
   cout << "Options: " << endl;
   cout << "-G <graph filename in binary format>" << endl
        << "-k <cardinality constraint>" << endl
        << "-a <Alg >" << endl
        << "-n [Samples to approx. multilinear extension, 0 for exact on maxcut]" << endl
        << "-o <outputFileName>" << endl
        << "-N <repetitions>" << endl
        << "-e <epsilon (default 0.1)>" << endl
        << "-d <delta (default 0.1)>" << endl
        << "-v [verbose]" << endl
        << "-f [fast mode (for algs. using ThresholdSample)]" << endl
        << "-r [report round information]" << endl       
        << "-s [Don't print std-deviation]" << endl;
}
Algs string_to_alg(string sin)
{
   transform(sin.begin(), sin.end(), sin.begin(), ::toupper);
   if (sin == "LINATG") return LinAtg;
   if (sin == "IG") return IG;
   if (sin == "FRG") return FRG;
   if (sin == "ENE") return ENE;
   if (sin == "ANM") return ANM;
   if (sin == "TSAST") return TSAST;
   if (sin == "TSATG") return TSATG;
   if (sin == "PARSKP2") return ParSKP2;
   if (sin == "BOOSTADAPT") return BoostAdapt;
   if (sin == "LINAST") return LinAst;
   return LinAtg;
}
void parseArgs(int argc, char **argv, Args &arg)
{
   int c;
   extern char *optarg;

   if (argc == 1)
   {
      print_help();
      exit(2);
   }

   string sarg;

   static struct option longopts[] = {
       {"TSAST", 0, NULL, TSAST},
       {"TSATG", 0, NULL, TSATG},
       {0, 0, 0, 0}};

   while ((c = getopt_long(argc, argv, ":G:k:a:o:e:d:vfrqsn:", longopts, NULL)) != -1)
   {
      switch (c)
      {
      case 's':
         arg.printStdDev = false;
         break;
      case 'f':
         arg.fast = true;
         break;
      case 'v':
         arg.g.logg.set_level(TRACE);
         break;
      case 'e':
         sarg.assign(optarg);
         arg.epsi = stod(sarg);
         break;
      case 'n':
         sarg.assign(optarg);
         arg.nSamps = stoi(sarg);
         break;
      case 'd':
         sarg.assign(optarg);
         arg.delta = stod(sarg);
         break;
      case 'o':
         sarg.assign(optarg);
         arg.outputFileName = sarg;
         break;
      case 'G':
         arg.graphFileName.assign(optarg);
         break;
      case 'k':
         sarg.assign(optarg);
         arg.k = stoi(sarg);
         break;
      case 'N':
         sarg.assign(optarg);
         arg.N = stoi(sarg);
         break;
      case 'l':
         arg.steal = false;
         break;
      case 'r':
         arg.reportRounds = true;
         break;
      case 'q':
         arg.logg.enabled = false;
         arg.g.logg.enabled = false;
         break;
      case 'a':
         sarg.assign(optarg);
         arg.sAlg = sarg;
         arg.alg = string_to_alg(sarg);
         break;
      case '?':
         print_help();
         exit(2);
         break;
      }
   }
}

void readGraph(Args &args)
{
   args.logg << INFO << "Reading graph from file: " << args.graphFileName << "..." << endL;
   args.g.read_bin(args.graphFileName);
   args.logg << INFO << "Nodes: " << args.g.n << ", edges: " << args.g.m << endL;
}

void runAlg(Args &args)
{
   size_t N = args.N;
   allResults.init("obj");
   allResults.init("nEvals");
   allResults.init("k");
   allResults.add("k", args.k);
   allResults.add("epsi", args.epsi);
   allResults.add("delta", args.delta);
   allResults.add("N", args.N);

   for (size_t i = 0; i < N; ++i)
   {
      args.g.logg << "-----------------------Run Alg: Repetition = " << i << "-----------------------" << endL;
      clock_t t_start = clock();
      std::chrono::steady_clock::time_point begin = std::chrono::steady_clock::now();
      switch (args.alg)
      {
         case IG:
         {
            args.logg(INFO, "Starting InterlaceGreedy...");
            Ig ig(args);
            ig.run();
         }
         break;
         case FRG:
         {
            args.logg(INFO, "Starting FastRandomizedGreedy...");
            Frg frg(args);
            frg.run();
         }
         break;         
         case ENE:
         {
            args.logg(INFO, "Starting Ene et al. (2019)...");
            Ene ene(args);
            ene.run();
         }
         break;         
         case ANM:
         {
            args.logg(INFO, "Starting ANM...");
            Anm anm(args);
            anm.run();
         }
         break;     
         case TSAST:
         {
            args.logg(INFO, "Starting AST (Kunhle)");
            TSAst tsAST(args);
            tsAST.run();
         }
         break;
         case TSATG:
         {
            args.logg(INFO, "Starting ATG (Kunhle)");
            TSAtg tsATG(args);
            tsATG.run();
         }
         break;        
         case BoostAdapt:
         {
            //args.logg(INFO, "Starting BoostAdapt (this work)");
            TSAtg tsATG1(args);
            tsATG1.runBoostAdapt();//BoostAdapt
         }
         break;
         case LinAst:
         {
            //args.logg(INFO, "Starting LinAst");
            TSAtg tsATG1(args);
            tsATG1.runLinAst();
         }
         break;
         case LinAtg:
         {
            //args.logg(INFO, "Starting LinAtg (this work)");
            TSAtg tsATG2(args);
            tsATG2.runLinAtg();
         }
         break;
         case ParSKP2:
         {
            args.logg(INFO, "Starting ParSKP2");
            ParSKP parSKP(args);
            parSKP.runParSKP2();
         }
         break;
      }

      args.tElapsed = elapsedTime(t_start);
      std::chrono::steady_clock::time_point end = std::chrono::steady_clock::now();
      size_t WallTimeMillis = std::chrono::duration_cast<std::chrono::milliseconds>(end - begin).count();
      args.wallTime = WallTimeMillis / 1000.0;
      allResults.add("time", args.wallTime);
   }
}

void outputResults(Args &args, ostream &of)
{
   of << args.sAlg << ",";
   if (args.a)
   {
      of << "true,";
   }
   else
   {
      of << "false,";
   }
   allResults.print("epsi", of, false);
   allResults.print("delta", of, false);
   allResults.print("N", of, false);
   allResults.print("k", of, false);

   allResults.print("obj", of, true);
   allResults.print("nEvals", of, true);
   allResults.print("rounds", of, true);
   allResults.print("nEvals_Adapt", of, true);
   allResults.print("time", of, true);
   of << endl;

   args.g.logg << INFO << "CPU time elapsed (s): " << args.tElapsed << endL;
   args.g.logg << INFO << "Wall time elapsed (s): " << args.wallTime << endL;
}

int main(int argc, char **argv)
{
   Args args;
   parseArgs(argc, argv, args);
   readGraph(args);
   runAlg(args);
   if (args.outputFileName != "")
   {
      ofstream of(args.outputFileName.c_str(), ofstream::out | ofstream::app);
      outputResults(args, of);
      of.close();
   }
   else
   {
      cout<<"Lỗi file"<<endl;
      allResults.print("obj", cout, args.printStdDev);
      allResults.print("nEvals", cout, args.printStdDev);
      allResults.print("rounds", cout, args.printStdDev);
   }
}
