#ifndef MYGRAPH_CPP
#define MYGRAPH_CPP
#include <list>
#include <vector>
#include <string>
#include <iostream>
#include <fstream>
#include <sstream>
#include <algorithm>
#include <map>
#include <unordered_set>
#include <thread>
#include <iomanip>
#include <cmath>
#include <utility>
#include <queue>
#include <set>
#include <thread>
#include <mutex>
#include <random>
#include "binheap.h"
#include "logger.h"
using namespace std;

double elapsedTime(clock_t &t_start);

template <typename TContainer, typename TArgs, typename TWorker>
void parallelWork(TContainer &v, TArgs &args,
                  TWorker &worker, size_t nThreads)
{
    auto itBegin = v.begin();
    auto itEnd = itBegin;

    if (nThreads > v.size())
        nThreads = v.size();

    thread *wThreads = new thread[nThreads];
    mutex mtx;
    for (size_t i = 0; i < nThreads; ++i)
    {
        if (i == nThreads - 1)
        {
            itEnd = v.end();
        }
        else
        {
            itEnd = itBegin;
            for (size_t j = 0; j < (v.size()) / nThreads; ++j)
                ++itEnd;
        }

        wThreads[i] = thread(&TWorker::run,
                             itBegin,
                             itEnd,
                             ref(args),
                             ref(mtx));

        itBegin = itEnd;
    }

    for (size_t i = 0; i < nThreads; ++i)
    {
        wThreads[i].join();
    }

    delete[] wThreads;
}

template <typename T>
void print_vector(vector<T> &v, ostream &os = cout)
{
    for (size_t i = 0; i < v.size(); ++i)
    {
        os << v[i] << ' ';
    }
    os << endl;
}

template <typename T, typename Compare>
std::vector<std::size_t> sort_permutation(
    const std::vector<T> &vec,
    Compare &compare)
{
    std::vector<std::size_t> p(vec.size());
    std::iota(p.begin(), p.end(), 0);
    std::sort(p.begin(), p.end(),
              [&](std::size_t i, std::size_t j)
              { return compare(vec[i], vec[j]); });
    return p;
}

template <typename T>
void apply_permutation_in_place(
    std::vector<T> &vec,
    const std::vector<std::size_t> &p)
{
    std::vector<bool> done(vec.size());
    for (std::size_t i = 0; i < vec.size(); ++i)
    {
        if (done[i])
        {
            continue;
        }
        done[i] = true;
        std::size_t prev_j = i;
        std::size_t j = p[i];
        while (i != j)
        {
            std::swap(vec[prev_j], vec[j]);
            done[j] = true;
            prev_j = j;
            j = p[j];
        }
    }
}

template <typename T>
bool vector_ctns(std::vector<T> &vec, T item)
{
    for (std::size_t i = 0; i < vec.size(); ++i)
    {
        if (vec[i] == item)
        {
            return true;
        }
    }

    return false;
}

namespace mygraph
{

    extern random_device rd;
    extern mt19937 gen;
    extern double tmpDouble;

    //Compact, undirected graph structure

    extern uint32_t bitMask;
    extern uint32_t bitS;
    extern uint32_t bitW;

    typedef uint32_t node_id;

    bool mycompare(const node_id &a, const node_id &b);

    /*
    * Edge classes
    */
    class tinyEdge
    {
    public:
        //last 30 bits are target node_id
        //first bit is inS, second bit is inW
        uint32_t target;
        unsigned char weight;
        tinyEdge();
        tinyEdge(node_id nid, unsigned char w = 1);
        tinyEdge(const tinyEdge &rhs);
        node_id getId() const;
        bool inS();
        bool inW();
        void setS();
        void setW();
        void unsetS();
        void unsetW();
    };

    /*
    * Only works if inS and inW are 0
    * Faster than operator<
    */
    bool tinyEdgeCompare(const tinyEdge &a, const tinyEdge &b);

    /*
    * Works regardless of status of front bits
    */
    bool operator<(const tinyEdge &a, const tinyEdge &b);

    /*
    * Full edge class 'fEdge'
    */
    class fEdge
    {
    public:
        node_id x;
        node_id y;

        mutable double w;
        mutable bool inS = false;

        fEdge() {}
        fEdge(node_id xx, node_id yy) : x(xx), y(yy), w(1.0) {}
        fEdge(node_id xx, node_id yy, double ww) : x(xx), y(yy), w(ww) {}
        fEdge(const fEdge &rhs);
        bool operator()(const fEdge &f1, const fEdge &f2);
    };

    //Ignores edge weight, undirected
    bool operator==(const fEdge &f1, const fEdge &f2);

    //For printing, ignores weights
    ostream &operator<<(ostream &os, const fEdge &f2);

    /*
    * (Bounded) Path enumeration
    *
    */
    class NodePath
    {
    public:
        vector<node_id> nodes;
        vector<fEdge> fedges;
        // vector<set<fEdge, fEdgeLT>::iterator> edges;
        vector<set<fEdge>::iterator> edges;

        uint32_t length;
        NodePath();
        NodePath(const NodePath &rhs);
        void addNode(const node_id &x, uint32_t w);
        bool ctns(const node_id &x);
        void print(ostream &os);
    };

    //Node class
    class tinyNode
    {
    public:
        bool inA = false;
        bool inB = false;
        bool inD = false;
        bool inE = false;
        bool inC = false;
        vector<tinyEdge> neis;

        tinyNode() {}

        tinyNode(const tinyNode &rhs);
        vector<tinyEdge>::iterator incident(node_id &out);
    };

    class tinyGraph
    {
    public:
        vector<tinyNode> adjList;
        unsigned n;
        unsigned m;
        Logger logg;

        double preprocessTime;
        tinyGraph();
        tinyGraph(const tinyGraph &h);
        void assign(const tinyGraph &h);
        void init_empty_graph();
        size_t getDegree(node_id v);
        size_t getWeightedDegree(node_id v);
        size_t getDegreeMinusA(node_id v);
        size_t getDegreeMinusB(node_id v);
        size_t getDegreeMinusSet(node_id v, vector<bool> &set);
        size_t getWeightedDegreeMinusSet(node_id v, vector<bool> &set);
        void clearAB();
        /*
      * Replaces graph with undirected, unweighted ER graph with n nodes
      * edge prob p
      */
        void erdos_renyi_undirected(node_id n, double p);
        /*
      * Replaces graph with undirected, BA graph
      */
        void barabasi_albert(node_id n, size_t m0, size_t m);
        /*
       * Adds undirected edge immediately into graph
       */
        void add_edge_immediate(unsigned from, unsigned to, unsigned char wht = 1);
        /*
       * Adds half of undirected edge, keeping adj. list sorted by target id
       * Checks to make sure graph remains simple, does not add edge
       * otherwise
       */
        bool add_edge_half(node_id from, node_id to, vector<tinyEdge>::iterator &edgeAdded,
                           unsigned char wht = 1);
        bool add_edge(fEdge &e);
        bool add_edge(node_id &from, node_id &to, unsigned char w = 1);
        unsigned char getEdgeWeight(node_id from, node_id to);
        unsigned char remove_edge_half(node_id from, node_id to);
        void read_bin(string fname);
        void write_bin(string fname);
        void write_edge_list(string fname);
        /*
       * Reads undirected graph from edge list
       * Returns pre-processing time for graph
       */
        double read_edge_list(string fname, bool checkSimple = true);
        /*
       * Reads undirected graph from edge list
       * Returns pre-processing time for graph
       */
        double read_directed_edge_list(string fname, bool checkSimple = true);
        vector<tinyEdge>::iterator findEdgeInList(node_id source, node_id target);
        void print(ostream &os);
        /*
       * Removed weight of edge removed
       * 0 if no edge removed
       */
        unsigned char remove_edge(node_id s, node_id t);
        /*
       * Removed weight of edge removed
       * 0 if no edge removed
       */
        unsigned char remove_edge(fEdge &e);
        unsigned countS();
    };

    void enumBoundedPaths(vector<NodePath> &res, node_id s, node_id t, tinyGraph &g, uint32_t T);

    //Graph structure solely for removing isolated nodes
    //Removes all isolated nodes, renumbers vertices from 0, n-1
    //unweighted, undirected graphs only
    class simplifyNode
    {
    public:
        node_id id;
        vector<simplifyNode *> neis;
        vector<unsigned> nei_weights;
        //      vector< simplifyNode* > _neis;
    };

    class simplifyGraph
    {
    public:
        node_id n;
        Logger logg;
        vector<simplifyNode *> adjList;

        simplifyGraph()
        {
            n = 0;
        }
        void add_edge(node_id from, node_id to, unsigned wht = 1);
        void remove_isolates();
        void renumber_vertices();
        void write_bin(string fname);
        void read_edge_list(string fname);
        void read_unweighted_edge_list(string fname);
    };

    class resultsHandler
    {
    public:
        map<string, vector<double>> data;
        void init(string name);
        void add(string name, double val);
        bool print(string name, ostream &os, bool printStdDev = true, double &Oldmean = tmpDouble);
        void print(ostream &os, bool printStdDev = true);
    };

    class algResult
    {
    public:
        string algName;
        string graphName;
        unsigned n;
        unsigned m;
        double p;
        double preprocess;
    };

    /******
    *
    * Graph algorithms
    *
    */

    static uint32_t uint32Max = ~(static_cast<uint32_t>(0));

    /********************************************************
    * Dijkstra's alg.
    * Distances are constrained to be integers
    * s is source, t is target
    */
    class Dijkstra
    {
    public:
        uint32_t d_infinity = uint32Max; //~( static_cast< uint32_t >(0) );
        node_id d_undefined = d_infinity;
        vector<uint32_t> h;       //estimate of distance d(u,t) for each u
        vector<uint32_t> h_saved; //estimate of distance d(u,t) for each u
        vector<node_id> prev;
        vector<node_id> forw;
        vector<uint32_t> dist; //Will store d(s, u) for each u (only interested in dist[t])
        vector<vector<node_id>> all_prev;

        node_id s;
        node_id t;
        tinyGraph &g;
        node_id n;

        bool h_valid = false;
        clock_t t_start;
        double t_elapsed;
        uint32_t dst;
        double t_dij;
        Dijkstra(node_id ss, node_id tt, tinyGraph &gg);
        Dijkstra(const Dijkstra &rhs);
        void print_hPath();
        void print_APath();
        void emptyQ(MinHeap &Q, uint32_t B = uint32Max);
        bool update_dij_after_addition(node_id u, node_id v, unsigned char wht = static_cast<unsigned char>(1),
                                       uint32_t B = uint32Max);
        /* Updates the dijkstra distances from
       * t, stored in h[u] = dist(u,t)
       *
       * Only computes distances within bound B
       */
        uint32_t update_dij_after_deletion(node_id u, node_id v, uint32_t B = uint32Max);
        void save_h();
        void restore_h();
        /*
       * This function assumes graph is undirected
       * otherwise, swapping the pair like this doesn't
       * work as intended
       */
        uint32_t compute_h(uint32_t B = uint32Max);
        /* Performs Dijkstra's alg from source s
       * to compute dist(u) = dist(s, u) for each u 
       * 
       * Quits when distance exceeds B
       * Distances are constrained to be integers
       *
       * Returns distance of shortest path between s,t
       */
        uint32_t compute_dij(uint32_t B = uint32Max);
        /* Performs Dijkstra's alg from source s
       * to compute dist(u) = dist(s, u) for each u 
       * 
       * Quits when distance exceeds B
       * Distances are constrained to be integers
       *
       * Returns distance of shortest path between s,t
       *
       * Differs from above functiion by storing all
       * shortest paths by storing a vector of all
       * the previous nodes in all_prev
       */
        uint32_t compute_dij_all_paths(uint32_t B = uint32Max);
        /* Astar update after distance of some edges
       * in g has increased
       * Distances are constrained to be integers
       * Proceeds from s to t
       *
       * Returns distance of shortest path between s,t
       * path from s to t can be reconstructed using prev
       */
        uint32_t update_Astar();
    };
};

#endif