import torch
from torch import nn
from torch.nn import init
import torch.nn.functional as F 




def init_weights(net, init_type='normal', init_gain=0.02):
    """Initialize network weights.

    Parameters:
        net (network)   -- network to be initialized
        init_type (str) -- the name of an initialization method: normal | xavier | kaiming | orthogonal
        init_gain (float)    -- scaling factor for normal, xavier and orthogonal.

    We use 'normal' in the original pix2pix and CycleGAN paper. But xavier and kaiming might
    work better for some applications. Feel free to try yourself.
    """

    def init_func(m):  # define the initialization function
        classname = m.__class__.__name__
        if hasattr(m, 'weight') and (classname.find('Conv') != -1 or classname.find('Linear') != -1):
            if init_type == 'normal':
                init.normal_(m.weight.data, 0.0, init_gain)
            elif init_type == 'xavier':
                init.xavier_normal_(m.weight.data, gain=init_gain)
            elif init_type == 'kaiming_uniform':
                init.kaiming_uniform(m.weight.data, a=0, mode='fan_in')
                init.kaiming_uniform(m.bias.data, a=0, mode='fan_in')
            elif init_type == 'orthogonal':
                init.orthogonal_(m.weight.data, gain=init_gain)
                init.orthogonal_(m.bias.data, gain=init_gain)
            else:
                raise NotImplementedError('initialization method [%s] is not implemented' % init_type)
            if hasattr(m, 'bias') and m.bias is not None:
                init.constant_(m.bias.data, 0.0)
        elif classname.find(
                'BatchNorm2d') != -1:  # BatchNorm Layer's weight is not a matrix; only normal distribution applies.
            init.normal_(m.weight.data, 1.0, init_gain)
            init.constant_(m.bias.data, 0.0)

    print('initialize network with %s' % init_type)
    net.apply(init_func)



# Batch Norm version
class MLP(nn.Module):
    def __init__(self, input_dim: int, hidden_layer: int,num_item: int, dropout: float = 0.0, n_task: int = 2,**kwargs):
        super(MLP, self).__init__(**kwargs)
        self.input_size = input_dim
        self.num_item = num_item
        self.n_task = n_task

        fcs = []
        last_size = self.input_size
        for _ in range(hidden_layer):
            size = last_size // 2
            linear = nn.Linear(last_size, size)
            # print(kwargs, kwargs.keys())
            bn = nn.BatchNorm1d(size)  #JK
            linear.bias.data.fill_(0.0)
            fcs.append(linear)
            use_bn = True #JK
            if use_bn:    #JK
                fcs.append(bn) #JK
            last_size = size
            fcs.append(nn.ReLU())
            if dropout > 0.0:
                fcs.append(nn.Dropout(dropout))
        self.shared_layers= nn.Sequential(*fcs)

        self.towers = nn.ModuleList([])
        for i in range(n_task):
            linear = nn.Sequential(
                nn.Linear(last_size, num_item),
                # nn.BatchNorm1d(num_item),
                # nn.ReLU(), 
                # nn.Linear(num_item, num_item)
                )
            self.towers.append(linear)


    def forward(self, x):
        features = self.shared_layers(x)
        outs = []
        for tower in self.towers:
            out = tower(features)
            outs.append(out)
        outs = torch.stack(outs, dim=1) # B x Num task x Num Item
        return outs







