# Copyright 2022 DeepMind Technologies Limited.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Library of stored bots for MeltingPot scenarios."""

import dataclasses
import functools
import os
from typing import AbstractSet, Callable, Iterable, Literal, Mapping, Optional, Sequence

import immutabledict

from meltingpot.python.utils.puppeteers import alternator
from meltingpot.python.utils.puppeteers import clean_up
from meltingpot.python.utils.puppeteers import coins
from meltingpot.python.utils.puppeteers import coordination_in_the_matrix
from meltingpot.python.utils.puppeteers import fixed_goal
from meltingpot.python.utils.puppeteers import gift_refinements
from meltingpot.python.utils.puppeteers import in_the_matrix
from meltingpot.python.utils.puppeteers import puppeteer
from meltingpot.python.utils.puppeteers import running_with_scissors_in_the_matrix


def _find_models_root() -> str:
  import re  # pylint: disable=g-import-not-at-top
  return re.sub('meltingpot/python/.*', 'meltingpot/assets/saved_models',
                __file__)


MODELS_ROOT = _find_models_root()

# pylint: disable=line-too-long

# Ordered puppet goals must match the order used in bot training.
_PUPPET_GOALS = immutabledict.immutabledict(
    # keep-sorted start numeric=yes block=yes
    bach_or_stravinsky_in_the_matrix__arena=puppeteer.puppet_goals([
        'COLLECT_BACH',
        'COLLECT_STRAVINSKY',
        'INTERACT_PLAYING_BACH',
        'INTERACT_PLAYING_STRAVINSKY',
    ]),
    bach_or_stravinsky_in_the_matrix__repeated=puppeteer.puppet_goals([
        'COLLECT_BACH',
        'COLLECT_STRAVINSKY',
        'INTERACT_PLAYING_BACH',
        'INTERACT_PLAYING_STRAVINSKY',
    ]),
    chicken_in_the_matrix__arena=puppeteer.puppet_goals([
        'COLLECT_DOVE',
        'COLLECT_HAWK',
        'INTERACT_PLAYING_DOVE',
        'INTERACT_PLAYING_HAWK',
    ]),
    chicken_in_the_matrix__repeated=puppeteer.puppet_goals([
        'COLLECT_DOVE',
        'COLLECT_HAWK',
        'INTERACT_PLAYING_DOVE',
        'INTERACT_PLAYING_HAWK',
    ]),
    clean_up=puppeteer.puppet_goals([
        'EAT',
        'CLEAN',
    ]),
    coins=puppeteer.puppet_goals([
        'COOPERATE',
        'DEFECT',
        'SPITE',
    ]),
    coop_mining=puppeteer.puppet_goals([
        'EXTRACT_IRON',
        'MINE_GOLD',
        'EXTRACT_GOLD',
        'EXTRACT_ALL',
    ]),
    externality_mushrooms__dense=puppeteer.puppet_goals([
        'COLLECT_MUSHROOM_HIHE',
        'COLLECT_MUSHROOM_FIZE',
        'COLLECT_MUSHROOM_ZIFE',
        'COLLECT_MUSHROOM_NINE',
        'DESTROY_MUSHROOM_HIHE',
        'DESTROY_MUSHROOM_FIZE',
        'DESTROY_MUSHROOM_ZIFE',
    ]),
    gift_refinements=puppeteer.puppet_goals([
        'COLLECT_TOKENS',
        'GIFT',
        'CONSUME_SIMPLE_TOKENS',
        'CONSUME_TOKENS',
        'FORAGE',
    ]),
    prisoners_dilemma_in_the_matrix__arena=puppeteer.puppet_goals([
        'COLLECT_COOPERATE',
        'COLLECT_DEFECT',
        'INTERACT_COOPERATE',
        'INTERACT_DEFECT',
    ]),
    prisoners_dilemma_in_the_matrix__repeated=puppeteer.puppet_goals([
        'COLLECT_COOPERATE',
        'COLLECT_DEFECT',
        'INTERACT_COOPERATE',
        'INTERACT_DEFECT',
    ]),
    pure_coordination_in_the_matrix__arena=puppeteer.puppet_goals([
        'COLLECT_RED',
        'COLLECT_GREEN',
        'COLLECT_BLUE',
        'INTERACT_PLAYING_RED',
        'INTERACT_PLAYING_GREEN',
        'INTERACT_PLAYING_BLUE',
        'COLLECT_RED_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_GREEN_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_BLUE_IGNORING_OTHER_CONSIDERATIONS',
    ]),
    pure_coordination_in_the_matrix__repeated=puppeteer.puppet_goals([
        'COLLECT_RED',
        'COLLECT_GREEN',
        'COLLECT_BLUE',
        'INTERACT_PLAYING_RED',
        'INTERACT_PLAYING_GREEN',
        'INTERACT_PLAYING_BLUE',
        'COLLECT_RED_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_GREEN_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_BLUE_IGNORING_OTHER_CONSIDERATIONS',
    ]),
    rationalizable_coordination_in_the_matrix__arena=puppeteer.puppet_goals([
        'COLLECT_YELLOW',
        'COLLECT_VIOLET',
        'COLLECT_CYAN',
        'INTERACT_PLAYING_YELLOW',
        'INTERACT_PLAYING_VIOLET',
        'INTERACT_PLAYING_CYAN',
        'COLLECT_YELLOW_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_VIOLET_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_CYAN_IGNORING_OTHER_CONSIDERATIONS',
    ]),
    rationalizable_coordination_in_the_matrix__repeated=puppeteer.puppet_goals([
        'COLLECT_YELLOW',
        'COLLECT_VIOLET',
        'COLLECT_CYAN',
        'INTERACT_PLAYING_YELLOW',
        'INTERACT_PLAYING_VIOLET',
        'INTERACT_PLAYING_CYAN',
        'COLLECT_YELLOW_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_VIOLET_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_CYAN_IGNORING_OTHER_CONSIDERATIONS',
    ]),
    running_with_scissors_in_the_matrix__arena=puppeteer.puppet_goals([
        'COLLECT_ROCK',
        'COLLECT_PAPER',
        'COLLECT_SCISSORS',
        'INTERACT_PLAYING_ROCK',
        'INTERACT_PLAYING_PAPER',
        'INTERACT_PLAYING_SCISSORS',
        'COLLECT_ROCK_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_PAPER_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_SCISSORS_IGNORING_OTHER_CONSIDERATIONS',
    ]),
    running_with_scissors_in_the_matrix__one_shot=puppeteer.puppet_goals([
        'COLLECT_ROCK',
        'COLLECT_PAPER',
        'COLLECT_SCISSORS',
        'INTERACT_PLAYING_ROCK',
        'INTERACT_PLAYING_PAPER',
        'INTERACT_PLAYING_SCISSORS',
        'COLLECT_ROCK_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_PAPER_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_SCISSORS_IGNORING_OTHER_CONSIDERATIONS',
    ]),
    running_with_scissors_in_the_matrix__repeated=puppeteer.puppet_goals([
        'COLLECT_ROCK',
        'COLLECT_PAPER',
        'COLLECT_SCISSORS',
        'INTERACT_PLAYING_ROCK',
        'INTERACT_PLAYING_PAPER',
        'INTERACT_PLAYING_SCISSORS',
        'COLLECT_ROCK_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_PAPER_IGNORING_OTHER_CONSIDERATIONS',
        'COLLECT_SCISSORS_IGNORING_OTHER_CONSIDERATIONS',
    ]),
    stag_hunt_in_the_matrix__arena=puppeteer.puppet_goals([
        'COLLECT_STAG',
        'COLLECT_HARE',
        'INTERACT_PLAYING_STAG',
        'INTERACT_PLAYING_HARE',
    ]),
    stag_hunt_in_the_matrix__repeated=puppeteer.puppet_goals([
        'COLLECT_STAG',
        'COLLECT_HARE',
        'INTERACT_PLAYING_STAG',
        'INTERACT_PLAYING_HARE',
    ]),
    # keep-sorted end
)

_RESOURCES = immutabledict.immutabledict(
    # keep-sorted start numeric=yes block=yes
    bach_or_stravinsky_in_the_matrix__arena=immutabledict.immutabledict({
        'BACH': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['bach_or_stravinsky_in_the_matrix__arena']['COLLECT_BACH'],
            interact_goal=_PUPPET_GOALS['bach_or_stravinsky_in_the_matrix__arena']['INTERACT_PLAYING_BACH'],
        ),
        'STRAVINSKY': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['bach_or_stravinsky_in_the_matrix__arena']['COLLECT_STRAVINSKY'],
            interact_goal=_PUPPET_GOALS['bach_or_stravinsky_in_the_matrix__arena']['INTERACT_PLAYING_STRAVINSKY'],
        ),
    }),
    bach_or_stravinsky_in_the_matrix__repeated=immutabledict.immutabledict({
        'BACH': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['bach_or_stravinsky_in_the_matrix__repeated']['COLLECT_BACH'],
            interact_goal=_PUPPET_GOALS['bach_or_stravinsky_in_the_matrix__repeated']['INTERACT_PLAYING_BACH'],
        ),
        'STRAVINSKY': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['bach_or_stravinsky_in_the_matrix__repeated']['COLLECT_STRAVINSKY'],
            interact_goal=_PUPPET_GOALS['bach_or_stravinsky_in_the_matrix__repeated']['INTERACT_PLAYING_STRAVINSKY'],
        ),
    }),
    chicken_in_the_matrix__arena=immutabledict.immutabledict({
        'DOVE': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['chicken_in_the_matrix__arena']['COLLECT_DOVE'],
            interact_goal=_PUPPET_GOALS['chicken_in_the_matrix__arena']['INTERACT_PLAYING_DOVE'],
        ),
        'HAWK': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['chicken_in_the_matrix__arena']['COLLECT_HAWK'],
            interact_goal=_PUPPET_GOALS['chicken_in_the_matrix__arena']['INTERACT_PLAYING_HAWK'],
        ),
    }),
    chicken_in_the_matrix__repeated=immutabledict.immutabledict({
        'DOVE': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['chicken_in_the_matrix__repeated']['COLLECT_DOVE'],
            interact_goal=_PUPPET_GOALS['chicken_in_the_matrix__repeated']['INTERACT_PLAYING_DOVE'],
        ),
        'HAWK': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['chicken_in_the_matrix__repeated']['COLLECT_HAWK'],
            interact_goal=_PUPPET_GOALS['chicken_in_the_matrix__repeated']['INTERACT_PLAYING_HAWK'],
        ),
    }),
    prisoners_dilemma_in_the_matrix__arena=immutabledict.immutabledict({
        'COOPERATE': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['prisoners_dilemma_in_the_matrix__arena']['COLLECT_COOPERATE'],
            interact_goal=_PUPPET_GOALS['prisoners_dilemma_in_the_matrix__arena']['INTERACT_COOPERATE'],
        ),
        'DEFECT': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['prisoners_dilemma_in_the_matrix__arena']['COLLECT_DEFECT'],
            interact_goal=_PUPPET_GOALS['prisoners_dilemma_in_the_matrix__arena']['INTERACT_DEFECT'],
        ),
    }),
    prisoners_dilemma_in_the_matrix__repeated=immutabledict.immutabledict({
        'COOPERATE': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['prisoners_dilemma_in_the_matrix__repeated']['COLLECT_COOPERATE'],
            interact_goal=_PUPPET_GOALS['prisoners_dilemma_in_the_matrix__repeated']['INTERACT_COOPERATE'],
        ),
        'DEFECT': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['prisoners_dilemma_in_the_matrix__repeated']['COLLECT_DEFECT'],
            interact_goal=_PUPPET_GOALS['prisoners_dilemma_in_the_matrix__repeated']['INTERACT_DEFECT'],
        ),
    }),
    pure_coordination_in_the_matrix__arena=immutabledict.immutabledict({
        'RED': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__arena']['COLLECT_RED'],
            interact_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__arena']['INTERACT_PLAYING_RED'],
        ),
        'GREEN': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__arena']['COLLECT_GREEN'],
            interact_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__arena']['INTERACT_PLAYING_GREEN'],
        ),
        'BLUE': in_the_matrix.Resource(
            index=2,
            collect_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__arena']['COLLECT_BLUE'],
            interact_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__arena']['INTERACT_PLAYING_BLUE'],
        ),
    }),
    pure_coordination_in_the_matrix__repeated=immutabledict.immutabledict({
        'RED': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__repeated']['COLLECT_RED'],
            interact_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__repeated']['INTERACT_PLAYING_RED'],
        ),
        'GREEN': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__repeated']['COLLECT_GREEN'],
            interact_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__repeated']['INTERACT_PLAYING_GREEN'],
        ),
        'BLUE': in_the_matrix.Resource(
            index=2,
            collect_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__repeated']['COLLECT_BLUE'],
            interact_goal=_PUPPET_GOALS['pure_coordination_in_the_matrix__repeated']['INTERACT_PLAYING_BLUE'],
        ),
    }),
    rationalizable_coordination_in_the_matrix__arena=immutabledict.immutabledict({
        'YELLOW': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__arena']['COLLECT_YELLOW'],
            interact_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__arena']['INTERACT_PLAYING_YELLOW'],
        ),
        'VIOLET': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__arena']['COLLECT_VIOLET'],
            interact_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__arena']['INTERACT_PLAYING_VIOLET'],
        ),
        'CYAN': in_the_matrix.Resource(
            index=2,
            collect_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__arena']['COLLECT_CYAN'],
            interact_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__arena']['INTERACT_PLAYING_CYAN'],
        ),
    }),
    rationalizable_coordination_in_the_matrix__repeated=immutabledict.immutabledict({
        'YELLOW': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__repeated']['COLLECT_YELLOW'],
            interact_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__repeated']['INTERACT_PLAYING_YELLOW'],
        ),
        'VIOLET': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__repeated']['COLLECT_VIOLET'],
            interact_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__repeated']['INTERACT_PLAYING_VIOLET'],
        ),
        'CYAN': in_the_matrix.Resource(
            index=2,
            collect_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__repeated']['COLLECT_CYAN'],
            interact_goal=_PUPPET_GOALS['rationalizable_coordination_in_the_matrix__repeated']['INTERACT_PLAYING_CYAN'],
        ),
    }),
    running_with_scissors_in_the_matrix__arena=immutabledict.immutabledict({
        'ROCK': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__arena']['COLLECT_ROCK'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__arena']['INTERACT_PLAYING_ROCK'],
        ),
        'PAPER': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__arena']['COLLECT_PAPER'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__arena']['INTERACT_PLAYING_PAPER'],
        ),
        'SCISSORS': in_the_matrix.Resource(
            index=2,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__arena']['COLLECT_SCISSORS'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__arena']['INTERACT_PLAYING_SCISSORS'],
        ),
    }),
    running_with_scissors_in_the_matrix__one_shot=immutabledict.immutabledict({
        'ROCK': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__one_shot']['COLLECT_ROCK'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__one_shot']['INTERACT_PLAYING_ROCK'],
        ),
        'PAPER': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__one_shot']['COLLECT_PAPER'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__one_shot']['INTERACT_PLAYING_PAPER'],
        ),
        'SCISSORS': in_the_matrix.Resource(
            index=2,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__one_shot']['COLLECT_SCISSORS'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__one_shot']['INTERACT_PLAYING_SCISSORS'],
        ),
    }),
    running_with_scissors_in_the_matrix__repeated=immutabledict.immutabledict({
        'ROCK': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__repeated']['COLLECT_ROCK'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__repeated']['INTERACT_PLAYING_ROCK'],
        ),
        'PAPER': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__repeated']['COLLECT_PAPER'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__repeated']['INTERACT_PLAYING_PAPER'],
        ),
        'SCISSORS': in_the_matrix.Resource(
            index=2,
            collect_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__repeated']['COLLECT_SCISSORS'],
            interact_goal=_PUPPET_GOALS['running_with_scissors_in_the_matrix__repeated']['INTERACT_PLAYING_SCISSORS'],
        ),
    }),
    stag_hunt_in_the_matrix__arena=immutabledict.immutabledict({
        'STAG': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['stag_hunt_in_the_matrix__arena']['COLLECT_STAG'],
            interact_goal=_PUPPET_GOALS['stag_hunt_in_the_matrix__arena']['INTERACT_PLAYING_STAG'],
        ),
        'HARE': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['stag_hunt_in_the_matrix__arena']['COLLECT_HARE'],
            interact_goal=_PUPPET_GOALS['stag_hunt_in_the_matrix__arena']['INTERACT_PLAYING_HARE'],
        ),
    }),
    stag_hunt_in_the_matrix__repeated=immutabledict.immutabledict({
        'STAG': in_the_matrix.Resource(
            index=0,
            collect_goal=_PUPPET_GOALS['stag_hunt_in_the_matrix__repeated']['COLLECT_STAG'],
            interact_goal=_PUPPET_GOALS['stag_hunt_in_the_matrix__repeated']['INTERACT_PLAYING_STAG'],
        ),
        'HARE': in_the_matrix.Resource(
            index=1,
            collect_goal=_PUPPET_GOALS['stag_hunt_in_the_matrix__repeated']['COLLECT_HARE'],
            interact_goal=_PUPPET_GOALS['stag_hunt_in_the_matrix__repeated']['INTERACT_PLAYING_HARE'],
        ),
    }),
    # keep-sorted end
)


@dataclasses.dataclass(frozen=True)
class BotConfig:
  """Bot config.

  Attributes:
    substrate: the substrate the bot was trained for.
    roles: the roles the bot was trained for.
    model_path: the path to the bot's saved model.
    model_version: whether the bot is a "1.0" bot or a new "1.1" bot.
    puppeteer_builder: returns the puppeteer used to control the bot.
  """
  substrate: str
  roles: AbstractSet[str]
  model_path: str
  puppeteer_builder: Optional[Callable[[], puppeteer.Puppeteer]]

  def __post_init__(self):
    object.__setattr__(self, 'roles', frozenset(self.roles))


def saved_model(*,
                substrate: str,
                roles: Iterable[str] = ('default',),
                model: str,
                models_root: str = MODELS_ROOT) -> BotConfig:
  """Returns the config for a saved model bot.

  Args:
    substrate: the substrate on which the bot was trained.
    roles: the roles the bot was trained for.
    model: the name of the model.
    models_root: The path to the directory containing the saved_models.
  """
  model_path = os.path.join(models_root, substrate, model)
  return BotConfig(
      substrate=substrate,
      roles=frozenset(roles),
      model_path=model_path,
      puppeteer_builder=None)


def puppet(*,
           substrate: str,
           roles: Iterable[str] = ('default',),
           model: str,
           puppeteer_builder: Callable[[], puppeteer.Puppeteer],
           models_root: str = MODELS_ROOT) -> BotConfig:
  """Returns the config for a puppet bot.

  Args:
    substrate: the substrate on which the bot was trained.
    roles: the roles the bot was trained for.
    model: the name of the model.
    puppeteer_builder: returns the puppeteer used to control the bot.
    models_root: the path to the directory containing the saved_models.
  """
  puppet_path = os.path.join(models_root, substrate, model)
  return BotConfig(
      substrate=substrate,
      roles=frozenset(roles),
      model_path=puppet_path,
      puppeteer_builder=puppeteer_builder)


BOT_CONFIGS: Mapping[str, BotConfig] = immutabledict.immutabledict(
    # keep-sorted start numeric=yes block=yes
    allelopathic_harvest__open__bot_that_supports_green_0=saved_model(
        substrate='allelopathic_harvest__open',
        model='bot_that_loves_green_0',
        roles=('default', 'player_who_likes_red', 'player_who_likes_green',),
    ),
    allelopathic_harvest__open__bot_that_supports_green_1=saved_model(
        substrate='allelopathic_harvest__open',
        model='bot_that_loves_green_1',
        roles=('default', 'player_who_likes_red', 'player_who_likes_green',),
    ),
    allelopathic_harvest__open__bot_that_supports_green_2=saved_model(
        substrate='allelopathic_harvest__open',
        model='bot_that_loves_green_2',
        roles=('default', 'player_who_likes_red', 'player_who_likes_green',),
    ),
    allelopathic_harvest__open__bot_that_supports_green_3=saved_model(
        substrate='allelopathic_harvest__open',
        model='bot_that_loves_green_3',
        roles=('default', 'player_who_likes_red', 'player_who_likes_green',),
    ),
    allelopathic_harvest__open__bot_that_supports_red_0=saved_model(
        substrate='allelopathic_harvest__open',
        model='bot_that_loves_red_0',
        roles=('default', 'player_who_likes_red', 'player_who_likes_green',),
    ),
    allelopathic_harvest__open__bot_that_supports_red_1=saved_model(
        substrate='allelopathic_harvest__open',
        model='bot_that_loves_red_1',
        roles=('default', 'player_who_likes_red', 'player_who_likes_green',),
    ),
    allelopathic_harvest__open__bot_that_supports_red_2=saved_model(
        substrate='allelopathic_harvest__open',
        model='bot_that_loves_red_2',
        roles=('default', 'player_who_likes_red', 'player_who_likes_green',),
    ),
    allelopathic_harvest__open__bot_that_supports_red_3=saved_model(
        substrate='allelopathic_harvest__open',
        model='bot_that_loves_red_3',
        roles=('default', 'player_who_likes_red', 'player_who_likes_green',),
    ),
    bach_or_stravinsky_in_the_matrix__arena__bach_picker_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__arena',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['bach_or_stravinsky_in_the_matrix__arena']['BACH'],
            margin=3,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__arena__stravinsky_picker_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__arena',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['bach_or_stravinsky_in_the_matrix__arena']['STRAVINSKY'],
            margin=3,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__arena__turn_taking_initial_bach_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__arena',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.AlternatingSpecialist,
            targets=[
                _RESOURCES['bach_or_stravinsky_in_the_matrix__arena']['BACH'],
                _RESOURCES['bach_or_stravinsky_in_the_matrix__arena']['STRAVINSKY'],
            ],
            interactions_per_target=2,
            margin=2,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__arena__turn_taking_initial_stravinsky_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__arena',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.AlternatingSpecialist,
            targets=[
                _RESOURCES['bach_or_stravinsky_in_the_matrix__arena']['STRAVINSKY'],
                _RESOURCES['bach_or_stravinsky_in_the_matrix__arena']['BACH'],
            ],
            interactions_per_target=2,
            margin=2,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__bach_picker_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__bach_tft_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
            defect_resource=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
            tremble_probability=0,
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__bach_tft_tremble_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
            defect_resource=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
            tremble_probability=0.25,
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__stravinsky_picker_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__stravinsky_tft_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
            defect_resource=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
            tremble_probability=0,
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__stravinsky_tft_tremble_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
            defect_resource=_RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
            tremble_probability=0.25,
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__turn_taking_initial_bach_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.AlternatingSpecialist,
            targets=[
                _RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
                _RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
            ],
            interactions_per_target=1,
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__turn_taking_initial_bach_1=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.AlternatingSpecialist,
            targets=[
                _RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
                _RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
            ],
            interactions_per_target=3,
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__turn_taking_initial_stravinsky_0=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.AlternatingSpecialist,
            targets=[
                _RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
                _RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
            ],
            interactions_per_target=1,
            margin=5,
        ),
    ),
    bach_or_stravinsky_in_the_matrix__repeated__turn_taking_initial_stravinsky_1=puppet(
        substrate='bach_or_stravinsky_in_the_matrix__repeated',
        model='puppet_0',
        roles=('default', 'bach_fan', 'stravinsky_fan',),
        puppeteer_builder=functools.partial(
            in_the_matrix.AlternatingSpecialist,
            targets=[
                _RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['STRAVINSKY'],
                _RESOURCES['bach_or_stravinsky_in_the_matrix__repeated']['BACH'],
            ],
            interactions_per_target=3,
            margin=5,
        ),
    ),
    boat_race__eight_races__cooperator_0=saved_model(
        substrate='boat_race__eight_races',
        model='cooperator_0',
        roles=('default', 'target'),
    ),
    boat_race__eight_races__defector_0=saved_model(
        substrate='boat_race__eight_races',
        model='defector_0',
        roles=('default',),
    ),
    chemistry__three_metabolic_cycles__blue_0=saved_model(
        substrate='chemistry__three_metabolic_cycles',
        model='blue_0',
    ),
    chemistry__three_metabolic_cycles__green_0=saved_model(
        substrate='chemistry__three_metabolic_cycles',
        model='green_0',
    ),
    chemistry__three_metabolic_cycles__yellow_0=saved_model(
        substrate='chemistry__three_metabolic_cycles',
        model='yellow_0',
    ),
    chemistry__three_metabolic_cycles_with_plentiful_distractors__blue_0=saved_model(
        substrate='chemistry__three_metabolic_cycles_with_plentiful_distractors',
        model='blue_0',
    ),
    chemistry__three_metabolic_cycles_with_plentiful_distractors__green_0=saved_model(
        substrate='chemistry__three_metabolic_cycles_with_plentiful_distractors',
        model='green_0',
    ),
    chemistry__three_metabolic_cycles_with_plentiful_distractors__yellow_0=saved_model(
        substrate='chemistry__three_metabolic_cycles_with_plentiful_distractors',
        model='yellow_0',
    ),
    chemistry__two_metabolic_cycles__blue_0=saved_model(
        substrate='chemistry__two_metabolic_cycles',
        model='blue_0',
    ),
    chemistry__two_metabolic_cycles__green_0=saved_model(
        substrate='chemistry__two_metabolic_cycles',
        model='green_0',
    ),
    chemistry__two_metabolic_cycles_with_distractors__blue_0=saved_model(
        substrate='chemistry__two_metabolic_cycles_with_distractors',
        model='blue_0',
    ),
    chemistry__two_metabolic_cycles_with_distractors__green_0=saved_model(
        substrate='chemistry__two_metabolic_cycles_with_distractors',
        model='green_0',
    ),
    chicken_in_the_matrix__arena__puppet_dove_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['chicken_in_the_matrix__arena']['DOVE'],
            margin=1,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_dove_margin_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['chicken_in_the_matrix__arena']['DOVE'],
            margin=5,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_grim_one_strike_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__arena']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__arena']['HAWK'],
            threshold=1,
            margin=1,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_grim_one_strike_margin_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__arena']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__arena']['HAWK'],
            threshold=1,
            margin=5,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_grim_three_strikes_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__arena']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__arena']['HAWK'],
            threshold=3,
            margin=1,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_grim_three_strikes_margin_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__arena']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__arena']['HAWK'],
            threshold=3,
            margin=5,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_grim_two_strikes_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__arena']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__arena']['HAWK'],
            threshold=2,
            margin=1,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_grim_two_strikes_margin_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__arena']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__arena']['HAWK'],
            threshold=2,
            margin=5,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_hawk_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['chicken_in_the_matrix__arena']['HAWK'],
            margin=1,
        ),
    ),
    chicken_in_the_matrix__arena__puppet_hawk_margin_0=puppet(
        substrate='chicken_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['chicken_in_the_matrix__arena']['HAWK'],
            margin=5,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_corrigible_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Corrigible,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            threshold=3,
            margin=5,
            tremble_probability=0,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_corrigible_tremble_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Corrigible,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            threshold=3,
            margin=5,
            tremble_probability=0.15,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_dove_margin_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            margin=5,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_dove_margin_1=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            margin=7,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_flip_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            initial_target=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            final_target=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            threshold=3,
            initial_margin=1,
            final_margin=5,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_grim_one_strike_margin_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            threshold=1,
            margin=5,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_grim_one_strike_margin_1=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            threshold=1,
            margin=7,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_grim_two_strikes_margin_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            threshold=2,
            margin=5,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_grim_two_strikes_margin_1=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            threshold=2,
            margin=7,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_hawk_margin_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            margin=5,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_hawk_margin_1=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            margin=7,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_tft_margin_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            tremble_probability=0,
            margin=5,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_tft_margin_1=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            tremble_probability=0,
            margin=7,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_tft_tremble_margin_0=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            tremble_probability=0.15,
            margin=5,
        ),
    ),
    chicken_in_the_matrix__repeated__puppet_tft_tremble_margin_1=puppet(
        substrate='chicken_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['chicken_in_the_matrix__repeated']['DOVE'],
            defect_resource=_RESOURCES['chicken_in_the_matrix__repeated']['HAWK'],
            tremble_probability=0.15,
            margin=7,
        ),
    ),
    clean_up__cleaner_0=saved_model(
        substrate='clean_up',
        model='cleaner_0',
    ),
    clean_up__cleaner_1=saved_model(
        substrate='clean_up',
        model='cleaner_1',
    ),
    clean_up__consumer_0=saved_model(
        substrate='clean_up',
        model='consumer_0',
    ),
    clean_up__consumer_1=saved_model(
        substrate='clean_up',
        model='consumer_1',
    ),
    clean_up__puppet_alternator_first_cleans_0=puppet(
        substrate='clean_up',
        model='puppet_0',
        roles=('default',),
        puppeteer_builder=functools.partial(
            alternator.Alternator,
            goals=[
                _PUPPET_GOALS['clean_up']['CLEAN'],
                _PUPPET_GOALS['clean_up']['EAT'],
            ],
            steps_per_goal=200,
        )),
    clean_up__puppet_alternator_first_eats_0=puppet(
        substrate='clean_up',
        model='puppet_0',
        roles=('default',),
        puppeteer_builder=functools.partial(
            alternator.Alternator,
            goals=[
                _PUPPET_GOALS['clean_up']['EAT'],
                _PUPPET_GOALS['clean_up']['CLEAN'],
            ],
            steps_per_goal=200,
        ),
    ),
    clean_up__puppet_high_threshold_reciprocator_0=puppet(
        substrate='clean_up',
        model='puppet_0',
        roles=('default',),
        puppeteer_builder=functools.partial(
            clean_up.ConditionalCleaner,
            clean_goal=_PUPPET_GOALS['clean_up']['CLEAN'],
            eat_goal=_PUPPET_GOALS['clean_up']['EAT'],
            coplayer_cleaning_signal='NUM_OTHERS_WHO_CLEANED_THIS_STEP',
            threshold=3,
            recency_window=5,
            reciprocation_period=75,
            niceness_period=0,
        ),
    ),
    clean_up__puppet_low_threshold_reciprocator_0=puppet(
        substrate='clean_up',
        model='puppet_0',
        roles=('default',),
        puppeteer_builder=functools.partial(
            clean_up.ConditionalCleaner,
            clean_goal=_PUPPET_GOALS['clean_up']['CLEAN'],
            eat_goal=_PUPPET_GOALS['clean_up']['EAT'],
            coplayer_cleaning_signal='NUM_OTHERS_WHO_CLEANED_THIS_STEP',
            threshold=2,
            recency_window=5,
            reciprocation_period=75,
            niceness_period=0,
        ),
    ),
    clean_up__puppet_nice_low_threshold_reciprocator_0=puppet(
        substrate='clean_up',
        model='puppet_0',
        roles=('default',),
        puppeteer_builder=functools.partial(
            clean_up.ConditionalCleaner,
            clean_goal=_PUPPET_GOALS['clean_up']['CLEAN'],
            eat_goal=_PUPPET_GOALS['clean_up']['EAT'],
            coplayer_cleaning_signal='NUM_OTHERS_WHO_CLEANED_THIS_STEP',
            threshold=2,
            recency_window=5,
            reciprocation_period=75,
            niceness_period=200,
        ),
    ),
    coins__puppet_cooperator_0=puppet(
        substrate='coins',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            fixed_goal.FixedGoal,
            goal=_PUPPET_GOALS['coins']['COOPERATE'],
        ),
    ),
    coins__puppet_defector_0=puppet(
        substrate='coins',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            fixed_goal.FixedGoal,
            goal=_PUPPET_GOALS['coins']['DEFECT'],
        ),
    ),
    coins__puppet_one_strike_reciprocator_0=puppet(
        substrate='coins',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            coins.Reciprocator,
            cooperate_goal=_PUPPET_GOALS['coins']['COOPERATE'],
            defect_goal=_PUPPET_GOALS['coins']['DEFECT'],
            spite_goal=_PUPPET_GOALS['coins']['SPITE'],
            partner_defection_signal='MISMATCHED_COIN_COLLECTED_BY_PARTNER',
            recency_window=100,
            threshold=1,
            frames_to_punish=100,
            spiteful_punishment_window=0,
        ),
    ),
    coins__puppet_one_strike_strong_reciprocator_0=puppet(
        substrate='coins',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            coins.Reciprocator,
            cooperate_goal=_PUPPET_GOALS['coins']['COOPERATE'],
            defect_goal=_PUPPET_GOALS['coins']['DEFECT'],
            spite_goal=_PUPPET_GOALS['coins']['SPITE'],
            partner_defection_signal='MISMATCHED_COIN_COLLECTED_BY_PARTNER',
            recency_window=100,
            threshold=1,
            frames_to_punish=100,
            spiteful_punishment_window=50,
        ),
    ),
    coins__puppet_three_strikes_reciprocator_0=puppet(
        substrate='coins',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            coins.Reciprocator,
            cooperate_goal=_PUPPET_GOALS['coins']['COOPERATE'],
            defect_goal=_PUPPET_GOALS['coins']['DEFECT'],
            spite_goal=_PUPPET_GOALS['coins']['SPITE'],
            partner_defection_signal='MISMATCHED_COIN_COLLECTED_BY_PARTNER',
            recency_window=150,
            threshold=3,
            frames_to_punish=150,
            spiteful_punishment_window=0,
        ),
    ),
    coins__puppet_three_strikes_strong_reciprocator_0=puppet(
        substrate='coins',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            coins.Reciprocator,
            cooperate_goal=_PUPPET_GOALS['coins']['COOPERATE'],
            defect_goal=_PUPPET_GOALS['coins']['DEFECT'],
            spite_goal=_PUPPET_GOALS['coins']['SPITE'],
            partner_defection_signal='MISMATCHED_COIN_COLLECTED_BY_PARTNER',
            recency_window=150,
            threshold=3,
            frames_to_punish=150,
            spiteful_punishment_window=75,
        ),
    ),
    collaborative_cooking__asymmetric__apprentice_0=saved_model(
        substrate='collaborative_cooking__asymmetric',
        model='apprentice_0',
    ),
    collaborative_cooking__asymmetric__apprentice_1=saved_model(
        substrate='collaborative_cooking__asymmetric',
        model='apprentice_1',
    ),
    collaborative_cooking__asymmetric__chef_0=saved_model(
        substrate='collaborative_cooking__asymmetric',
        model='chef_0',
    ),
    collaborative_cooking__asymmetric__chef_1=saved_model(
        substrate='collaborative_cooking__asymmetric',
        model='chef_1',
    ),
    collaborative_cooking__circuit__apprentice_0=saved_model(
        substrate='collaborative_cooking__circuit',
        model='apprentice_0',
    ),
    collaborative_cooking__circuit__apprentice_1=saved_model(
        substrate='collaborative_cooking__circuit',
        model='apprentice_1',
    ),
    collaborative_cooking__circuit__chef_0=saved_model(
        substrate='collaborative_cooking__circuit',
        model='chef_0',
    ),
    collaborative_cooking__circuit__chef_1=saved_model(
        substrate='collaborative_cooking__circuit',
        model='chef_1',
    ),
    collaborative_cooking__cramped__apprentice_0=saved_model(
        substrate='collaborative_cooking__cramped',
        model='apprentice_0',
    ),
    collaborative_cooking__cramped__apprentice_1=saved_model(
        substrate='collaborative_cooking__cramped',
        model='apprentice_1',
    ),
    collaborative_cooking__cramped__chef_0=saved_model(
        substrate='collaborative_cooking__cramped',
        model='chef_0',
    ),
    collaborative_cooking__cramped__chef_1=saved_model(
        substrate='collaborative_cooking__cramped',
        model='chef_1',
    ),
    collaborative_cooking__crowded__independent_chef_0=saved_model(
        substrate='collaborative_cooking__crowded',
        model='independent_chef_0',
    ),
    collaborative_cooking__crowded__robust_chef_0=saved_model(
        substrate='collaborative_cooking__crowded',
        model='robust_chef_0',
    ),
    collaborative_cooking__figure_eight__independent_chef_0=saved_model(
        substrate='collaborative_cooking__figure_eight',
        model='independent_chef_0',
    ),
    collaborative_cooking__figure_eight__robust_chef_0=saved_model(
        substrate='collaborative_cooking__figure_eight',
        model='robust_chef_0',
    ),
    collaborative_cooking__forced__apprentice_0=saved_model(
        substrate='collaborative_cooking__forced',
        model='apprentice_0',
    ),
    collaborative_cooking__forced__apprentice_1=saved_model(
        substrate='collaborative_cooking__forced',
        model='apprentice_1',
    ),
    collaborative_cooking__forced__chef_0=saved_model(
        substrate='collaborative_cooking__forced',
        model='chef_0',
    ),
    collaborative_cooking__forced__chef_1=saved_model(
        substrate='collaborative_cooking__forced',
        model='chef_1',
    ),
    collaborative_cooking__ring__apprentice_0=saved_model(
        substrate='collaborative_cooking__ring',
        model='apprentice_0',
    ),
    collaborative_cooking__ring__apprentice_1=saved_model(
        substrate='collaborative_cooking__ring',
        model='apprentice_1',
    ),
    collaborative_cooking__ring__chef_0=saved_model(
        substrate='collaborative_cooking__ring',
        model='chef_0',
    ),
    collaborative_cooking__ring__chef_1=saved_model(
        substrate='collaborative_cooking__ring',
        model='chef_1',
    ),
    commons_harvest__closed__free_0=saved_model(
        substrate='commons_harvest__closed',
        model='free_0',
    ),
    commons_harvest__closed__free_1=saved_model(
        substrate='commons_harvest__closed',
        model='free_1',
    ),
    commons_harvest__closed__free_2=saved_model(
        substrate='commons_harvest__closed',
        model='free_2',
    ),
    commons_harvest__closed__free_3=saved_model(
        substrate='commons_harvest__closed',
        model='free_3',
    ),
    commons_harvest__closed__pacifist_0=saved_model(
        substrate='commons_harvest__closed',
        model='pacifist_0',
    ),
    commons_harvest__closed__pacifist_1=saved_model(
        substrate='commons_harvest__closed',
        model='pacifist_1',
    ),
    commons_harvest__closed__pacifist_2=saved_model(
        substrate='commons_harvest__closed',
        model='pacifist_2',
    ),
    commons_harvest__open__free_0=saved_model(
        substrate='commons_harvest__open',
        model='free_0',
    ),
    commons_harvest__open__free_1=saved_model(
        substrate='commons_harvest__open',
        model='free_1',
    ),
    commons_harvest__open__pacifist_0=saved_model(
        substrate='commons_harvest__open',
        model='pacifist_0',
    ),
    commons_harvest__open__pacifist_1=saved_model(
        substrate='commons_harvest__open',
        model='pacifist_1',
    ),
    commons_harvest__partnership__free_0=saved_model(
        substrate='commons_harvest__partnership',
        model='free_0',
    ),
    commons_harvest__partnership__free_1=saved_model(
        substrate='commons_harvest__partnership',
        model='free_1',
    ),
    commons_harvest__partnership__free_2=saved_model(
        substrate='commons_harvest__partnership',
        model='free_2',
    ),
    commons_harvest__partnership__good_partner_0=saved_model(
        substrate='commons_harvest__partnership',
        model='good_partner_0',
    ),
    commons_harvest__partnership__good_partner_1=saved_model(
        substrate='commons_harvest__partnership',
        model='good_partner_1',
    ),
    commons_harvest__partnership__good_partner_2=saved_model(
        substrate='commons_harvest__partnership',
        model='good_partner_2',
    ),
    commons_harvest__partnership__pacifist_0=saved_model(
        substrate='commons_harvest__partnership',
        model='pacifist_0',
    ),
    commons_harvest__partnership__pacifist_1=saved_model(
        substrate='commons_harvest__partnership',
        model='pacifist_1',
    ),
    commons_harvest__partnership__pacifist_2=saved_model(
        substrate='commons_harvest__partnership',
        model='pacifist_2',
    ),
    commons_harvest__partnership__sustainable_fighter_0=saved_model(
        substrate='commons_harvest__partnership',
        model='sustainable_fighter_0',
    ),
    commons_harvest__partnership__sustainable_fighter_1=saved_model(
        substrate='commons_harvest__partnership',
        model='sustainable_fighter_1',
    ),
    coop_mining__cooperator_0=puppet(
        substrate='coop_mining',
        model='puppet_0',
        roles=('default', 'target'),
        puppeteer_builder=functools.partial(
            fixed_goal.FixedGoal,
            _PUPPET_GOALS['coop_mining']['EXTRACT_GOLD'],
        ),
    ),
    coop_mining__defector_0=puppet(
        substrate='coop_mining',
        model='puppet_0',
        roles=('default',),
        puppeteer_builder=functools.partial(
            fixed_goal.FixedGoal,
            _PUPPET_GOALS['coop_mining']['EXTRACT_IRON'],
        ),
    ),
    coop_mining__mixed_0=puppet(
        substrate='coop_mining',
        model='puppet_0',
        roles=('default', 'target'),
        puppeteer_builder=functools.partial(
            alternator.Alternator,
            goals=[
                _PUPPET_GOALS['coop_mining']['EXTRACT_IRON'],
                _PUPPET_GOALS['coop_mining']['EXTRACT_GOLD'],
            ],
            steps_per_goal=100,
        ),
    ),
    daycare__foraging_child_0=saved_model(
        substrate='daycare',
        model='foraging_child_0',
        roles=('child',),
    ),
    daycare__foraging_parent_0=saved_model(
        substrate='daycare',
        model='foraging_parent_0',
        roles=('parent',),
    ),
    daycare__helping_parent_0=saved_model(
        substrate='daycare',
        model='helping_parent_0',
        roles=('parent',),
    ),
    daycare__pointing_child_0=saved_model(
        substrate='daycare',
        model='pointing_child_0',
        roles=('child',),
    ),
    externality_mushrooms__dense__puppet_fize_0=puppet(
        substrate='externality_mushrooms__dense',
        model='puppet_0',
        roles=('default',),
        puppeteer_builder=functools.partial(
            fixed_goal.FixedGoal, (_PUPPET_GOALS['externality_mushrooms__dense']
                                   ['COLLECT_MUSHROOM_FIZE'])),
    ),
    externality_mushrooms__dense__puppet_hihe_0=puppet(
        substrate='externality_mushrooms__dense',
        model='puppet_0',
        roles=('default',),
        puppeteer_builder=functools.partial(
            fixed_goal.FixedGoal, (_PUPPET_GOALS['externality_mushrooms__dense']
                                   ['COLLECT_MUSHROOM_HIHE'])),
    ),
    factory_commons__either_or__sustainable_0=saved_model(
        substrate='factory_commons__either_or',
        model='sustainable_0',
        roles=('default',),
    ),
    factory_commons__either_or__sustainable_1=saved_model(
        substrate='factory_commons__either_or',
        model='sustainable_1',
        roles=('default',),
    ),
    factory_commons__either_or__sustainable_2=saved_model(
        substrate='factory_commons__either_or',
        model='sustainable_2',
        roles=('default',),
    ),
    factory_commons__either_or__unsustainable_0=saved_model(
        substrate='factory_commons__either_or',
        model='unsustainable_0',
        roles=('default',),
    ),
    factory_commons__either_or__unsustainable_1=saved_model(
        substrate='factory_commons__either_or',
        model='unsustainable_1',
        roles=('default',),
    ),
    factory_commons__either_or__unsustainable_2=saved_model(
        substrate='factory_commons__either_or',
        model='unsustainable_2',
        roles=('default',),
    ),
    fruit_market__concentric_rivers__apple_farmer_0=saved_model(
        substrate='fruit_market__concentric_rivers',
        model='apple_farmer_0',
        roles=('apple_farmer',),
    ),
    fruit_market__concentric_rivers__apple_farmer_1=saved_model(
        substrate='fruit_market__concentric_rivers',
        model='apple_farmer_1',
        roles=('apple_farmer',),
    ),
    fruit_market__concentric_rivers__apple_farmer_2=saved_model(
        substrate='fruit_market__concentric_rivers',
        model='apple_farmer_2',
        roles=('apple_farmer',),
    ),
    fruit_market__concentric_rivers__banana_farmer_0=saved_model(
        substrate='fruit_market__concentric_rivers',
        model='banana_farmer_0',
        roles=('banana_farmer',),
    ),
    fruit_market__concentric_rivers__banana_farmer_1=saved_model(
        substrate='fruit_market__concentric_rivers',
        model='banana_farmer_1',
        roles=('banana_farmer',),
    ),
    fruit_market__concentric_rivers__banana_farmer_2=saved_model(
        substrate='fruit_market__concentric_rivers',
        model='banana_farmer_2',
        roles=('banana_farmer',),
    ),
    gift_refinements__cooperator_0=puppet(
        substrate='gift_refinements',
        roles=('default', 'target'),
        model='puppet_0',
        puppeteer_builder=functools.partial(
            gift_refinements.GiftRefinementsCooperator,
            collect_goal=_PUPPET_GOALS['gift_refinements']['COLLECT_TOKENS'],
            consume_goal=_PUPPET_GOALS['gift_refinements']['CONSUME_TOKENS'],
            gift_goal=_PUPPET_GOALS['gift_refinements']['GIFT'],
        ),
    ),
    gift_refinements__defector_0=puppet(
        substrate='gift_refinements',
        roles=('default', 'target'),
        model='puppet_0',
        puppeteer_builder=functools.partial(
            fixed_goal.FixedGoal,
            goal=_PUPPET_GOALS['gift_refinements']['FORAGE'],
            ),
    ),
    gift_refinements__extreme_cooperator_0=puppet(
        substrate='gift_refinements',
        roles=('default', 'target'),
        model='puppet_0',
        puppeteer_builder=functools.partial(
            gift_refinements.GiftRefinementsExtremeCooperator,
            collect_goal=_PUPPET_GOALS['gift_refinements']['COLLECT_TOKENS'],
            consume_goal=_PUPPET_GOALS['gift_refinements']['CONSUME_TOKENS'],
            gift_goal=_PUPPET_GOALS['gift_refinements']['GIFT'],
        ),
    ),
    hidden_agenda__collector_crew_0=saved_model(
        substrate='hidden_agenda',
        model='collector_crew_0',
        roles=('crewmate',),
    ),
    hidden_agenda__collector_crew_1=saved_model(
        substrate='hidden_agenda',
        model='collector_crew_1',
        roles=('crewmate',),
    ),
    hidden_agenda__hunter_impostor_0=saved_model(
        substrate='hidden_agenda',
        model='hunter_impostor_0',
        roles=('impostor',),
    ),
    paintball__capture_the_flag__shaped_bot_0=saved_model(
        substrate='paintball__capture_the_flag',
        model='shaped_0',
        roles=('default',),
    ),
    paintball__capture_the_flag__shaped_bot_1=saved_model(
        substrate='paintball__capture_the_flag',
        model='shaped_1',
        roles=('default',),
    ),
    paintball__capture_the_flag__shaped_bot_2=saved_model(
        substrate='paintball__capture_the_flag',
        model='shaped_2',
        roles=('default',),
    ),
    paintball__capture_the_flag__shaped_bot_3=saved_model(
        substrate='paintball__capture_the_flag',
        model='shaped_3',
        roles=('default',),
    ),
    paintball__king_of_the_hill__free_0=saved_model(
        substrate='paintball__king_of_the_hill',
        model='free_bot_0',
        roles=('default',),
    ),
    paintball__king_of_the_hill__free_1=saved_model(
        substrate='paintball__king_of_the_hill',
        model='free_bot_1',
        roles=('default',),
    ),
    paintball__king_of_the_hill__free_2=saved_model(
        substrate='paintball__king_of_the_hill',
        model='free_bot_2',
        roles=('default',),
    ),
    paintball__king_of_the_hill__spawn_camper_0=saved_model(
        substrate='paintball__king_of_the_hill',
        model='spawn_camper_0',
        roles=('default',),
    ),
    paintball__king_of_the_hill__spawn_camper_1=saved_model(
        substrate='paintball__king_of_the_hill',
        model='spawn_camper_1',
        roles=('default',),
    ),
    paintball__king_of_the_hill__spawn_camper_2=saved_model(
        substrate='paintball__king_of_the_hill',
        model='spawn_camper_2',
        roles=('default',),
    ),
    paintball__king_of_the_hill__spawn_camper_3=saved_model(
        substrate='paintball__king_of_the_hill',
        model='spawn_camper_3',
        roles=('default',),
    ),
    predator_prey__alley_hunt__predator_0=saved_model(
        substrate='predator_prey__alley_hunt',
        model='basic_predator_0',
        roles=('predator',),
    ),
    predator_prey__alley_hunt__predator_1=saved_model(
        substrate='predator_prey__alley_hunt',
        model='basic_predator_1',
        roles=('predator',),
    ),
    predator_prey__alley_hunt__predator_2=saved_model(
        substrate='predator_prey__alley_hunt',
        model='basic_predator_2',
        roles=('predator',),
    ),
    predator_prey__alley_hunt__prey_0=saved_model(
        substrate='predator_prey__alley_hunt',
        model='basic_prey_0',
        roles=('prey',),
    ),
    predator_prey__alley_hunt__prey_1=saved_model(
        substrate='predator_prey__alley_hunt',
        model='basic_prey_1',
        roles=('prey',),
    ),
    predator_prey__alley_hunt__prey_2=saved_model(
        substrate='predator_prey__alley_hunt',
        model='basic_prey_2',
        roles=('prey',),
    ),
    predator_prey__open__basic_predator_0=saved_model(
        substrate='predator_prey__open',
        model='basic_predator_0',
        roles=('predator',),
    ),
    predator_prey__open__basic_predator_1=saved_model(
        substrate='predator_prey__open',
        model='basic_predator_1',
        roles=('predator',),
    ),
    predator_prey__open__basic_prey_0=saved_model(
        substrate='predator_prey__open',
        model='basic_prey_0',
        roles=('prey',),
    ),
    predator_prey__open__basic_prey_1=saved_model(
        substrate='predator_prey__open',
        model='basic_prey_1',
        roles=('prey',),
    ),
    predator_prey__open__basic_prey_2=saved_model(
        substrate='predator_prey__open',
        model='basic_prey_2',
        roles=('prey',),
    ),
    predator_prey__open__smart_prey_0=saved_model(
        substrate='predator_prey__open',
        model='smart_prey_0',
        roles=('prey',),
    ),
    predator_prey__open__smart_prey_1=saved_model(
        substrate='predator_prey__open',
        model='smart_prey_1',
        roles=('prey',),
    ),
    predator_prey__open__smart_prey_2=saved_model(
        substrate='predator_prey__open',
        model='smart_prey_2',
        roles=('prey',),
    ),
    predator_prey__orchard__acorn_specialist_prey_0=saved_model(
        substrate='predator_prey__orchard',
        model='acorn_specialist_prey_0',
        roles=('prey',),
    ),
    predator_prey__orchard__acorn_specialist_prey_1=saved_model(
        substrate='predator_prey__orchard',
        model='acorn_specialist_prey_1',
        roles=('prey',),
    ),
    predator_prey__orchard__acorn_specialist_prey_2=saved_model(
        substrate='predator_prey__orchard',
        model='acorn_specialist_prey_2',
        roles=('prey',),
    ),
    predator_prey__orchard__acorn_specialist_prey_3=saved_model(
        substrate='predator_prey__orchard',
        model='acorn_specialist_prey_3',
        roles=('prey',),
    ),
    predator_prey__orchard__acorn_specialist_prey_4=saved_model(
        substrate='predator_prey__orchard',
        model='acorn_specialist_prey_4',
        roles=('prey',),
    ),
    predator_prey__orchard__basic_predator_0=saved_model(
        substrate='predator_prey__orchard',
        model='basic_predator_0',
        roles=('predator',),
    ),
    predator_prey__orchard__basic_predator_1=saved_model(
        substrate='predator_prey__orchard',
        model='basic_predator_1',
        roles=('predator',),
    ),
    predator_prey__orchard__basic_predator_2=saved_model(
        substrate='predator_prey__orchard',
        model='basic_predator_2',
        roles=('predator',),
    ),
    predator_prey__orchard__basic_prey_0=saved_model(
        substrate='predator_prey__orchard',
        model='basic_prey_0',
        roles=('prey',),
    ),
    predator_prey__orchard__basic_prey_1=saved_model(
        substrate='predator_prey__orchard',
        model='basic_prey_1',
        roles=('prey',),
    ),
    predator_prey__orchard__basic_prey_2=saved_model(
        substrate='predator_prey__orchard',
        model='basic_prey_2',
        roles=('prey',),
    ),
    predator_prey__orchard__basic_prey_3=saved_model(
        substrate='predator_prey__orchard',
        model='basic_prey_3',
        roles=('prey',),
    ),
    predator_prey__orchard__basic_prey_4=saved_model(
        substrate='predator_prey__orchard',
        model='basic_prey_4',
        roles=('prey',),
    ),
    predator_prey__orchard__basic_prey_5=saved_model(
        substrate='predator_prey__orchard',
        model='basic_prey_5',
        roles=('prey',),
    ),
    predator_prey__random_forest__basic_predator_0=saved_model(
        substrate='predator_prey__random_forest',
        model='basic_predator_0',
        roles=('predator',),
    ),
    predator_prey__random_forest__basic_predator_1=saved_model(
        substrate='predator_prey__random_forest',
        model='basic_predator_1',
        roles=('predator',),
    ),
    predator_prey__random_forest__basic_predator_2=saved_model(
        substrate='predator_prey__random_forest',
        model='basic_predator_2',
        roles=('predator',),
    ),
    predator_prey__random_forest__basic_prey_0=saved_model(
        substrate='predator_prey__random_forest',
        model='basic_prey_0',
        roles=('prey',),
    ),
    predator_prey__random_forest__basic_prey_1=saved_model(
        substrate='predator_prey__random_forest',
        model='basic_prey_1',
        roles=('prey',),
    ),
    predator_prey__random_forest__basic_prey_2=saved_model(
        substrate='predator_prey__random_forest',
        model='basic_prey_2',
        roles=('prey',),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_cooperator_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['COOPERATE'],
            margin=1,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_cooperator_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['COOPERATE'],
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_defector_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['DEFECT'],
            margin=1,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_defector_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['DEFECT'],
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_grim_one_strike_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['DEFECT'],
            threshold=1,
            margin=1,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_grim_one_strike_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['DEFECT'],
            threshold=1,
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_grim_three_strikes_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['DEFECT'],
            threshold=3,
            margin=1,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_grim_three_strikes_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['DEFECT'],
            threshold=3,
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_grim_two_strikes_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['DEFECT'],
            threshold=2,
            margin=1,
        ),
    ),
    prisoners_dilemma_in_the_matrix__arena__puppet_grim_two_strikes_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__arena']['DEFECT'],
            threshold=2,
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_cooperator_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_cooperator_margin_1=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            margin=7,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_corrigible_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Corrigible,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            threshold=3,
            margin=5,
                        tremble_probability=0,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_corrigible_tremble_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Corrigible,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            threshold=3,
            margin=5,
            tremble_probability=0.15,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_defector_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_defector_margin_1=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            margin=7,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_flip_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            initial_target=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            final_target=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            threshold=3,
            initial_margin=1,
            final_margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_grim_one_strike_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            threshold=1,
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_grim_one_strike_margin_1=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            threshold=1,
            margin=7,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_grim_two_strikes_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            threshold=2,
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_grim_two_strikes_margin_1=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            threshold=2,
            margin=7,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_tft_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            tremble_probability=0,
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_tft_margin_1=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            tremble_probability=0,
            margin=7,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_tft_tremble_margin_0=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            tremble_probability=0.15,
            margin=5,
        ),
    ),
    prisoners_dilemma_in_the_matrix__repeated__puppet_tft_tremble_margin_1=puppet(
        substrate='prisoners_dilemma_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['COOPERATE'],
            defect_resource=_RESOURCES['prisoners_dilemma_in_the_matrix__repeated']['DEFECT'],
            tremble_probability=0.15,
            margin=7,
        ),
    ),
    pure_coordination_in_the_matrix__arena__flip_a2b_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['RED'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['GREEN'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__flip_a2c_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['RED'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['BLUE'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__flip_b2a_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['GREEN'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['RED'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__flip_b2c_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['GREEN'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['BLUE'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__flip_c2a_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['BLUE'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['RED'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__flip_c2b_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['BLUE'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__arena']['GREEN'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__pure_a_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__arena']['RED'],
            margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__pure_b_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__arena']['GREEN'],
            margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__pure_c_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__arena']['BLUE'],
            margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__pure_greedy_a_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__arena']['RED'],
            margin=6,
        ),
    ),
    pure_coordination_in_the_matrix__arena__pure_greedy_b_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__arena']['GREEN'],
            margin=6,
        ),
    ),
    pure_coordination_in_the_matrix__arena__pure_greedy_c_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__arena']['BLUE'],
            margin=6,
        ),
    ),
    pure_coordination_in_the_matrix__arena__resp2prev_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            coordination_in_the_matrix.CoordinateWithPrevious,
            resources=(
                _RESOURCES['pure_coordination_in_the_matrix__arena']['RED'],
                _RESOURCES['pure_coordination_in_the_matrix__arena']['GREEN'],
                _RESOURCES['pure_coordination_in_the_matrix__arena']['BLUE'],
            ),
            margin=1,
        ),
    ),
    pure_coordination_in_the_matrix__arena__resp2prev_greedy_0=puppet(
        substrate='pure_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            coordination_in_the_matrix.CoordinateWithPrevious,
            resources=(
                _RESOURCES['pure_coordination_in_the_matrix__arena']['RED'],
                _RESOURCES['pure_coordination_in_the_matrix__arena']['GREEN'],
                _RESOURCES['pure_coordination_in_the_matrix__arena']['BLUE'],
            ),
            margin=6,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_a2b_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_a2b_1=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_a2c_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_a2c_1=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_b2a_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_b2a_1=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_b2c_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_b2c_1=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_c2a_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_c2a_1=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_c2b_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__flip_c2b_1=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            final_target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__pure_a_margin_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
            margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__pure_b_margin_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
            margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__pure_c_margin_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            margin=5,
        ),
    ),
    pure_coordination_in_the_matrix__repeated__resp2prev_margin_0=puppet(
        substrate='pure_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            coordination_in_the_matrix.CoordinateWithPrevious,
            resources=(
                _RESOURCES['pure_coordination_in_the_matrix__repeated']['RED'],
                _RESOURCES['pure_coordination_in_the_matrix__repeated']['GREEN'],
                _RESOURCES['pure_coordination_in_the_matrix__repeated']['BLUE'],
            ),
            margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__flip_a2b_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['YELLOW'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['VIOLET'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__flip_a2c_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['YELLOW'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['CYAN'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__flip_b2a_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['VIOLET'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['YELLOW'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__flip_b2c_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['VIOLET'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['CYAN'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__flip_c2a_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['CYAN'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['YELLOW'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__flip_c2b_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=5,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['CYAN'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['VIOLET'],
            initial_margin=1,
            final_margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__pure_a_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['YELLOW'],
            margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__pure_b_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['VIOLET'],
            margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__pure_c_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['CYAN'],
            margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__pure_greedy_a_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['YELLOW'],
            margin=6,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__pure_greedy_b_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['VIOLET'],
            margin=6,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__pure_greedy_c_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__arena']['CYAN'],
            margin=6,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__resp2prev_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            coordination_in_the_matrix.CoordinateWithPrevious,
            resources=(
                _RESOURCES['rationalizable_coordination_in_the_matrix__arena']['YELLOW'],
                _RESOURCES['rationalizable_coordination_in_the_matrix__arena']['VIOLET'],
                _RESOURCES['rationalizable_coordination_in_the_matrix__arena']['CYAN'],
            ),
            margin=1,
        ),
    ),
    rationalizable_coordination_in_the_matrix__arena__resp2prev_greedy_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__arena',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            coordination_in_the_matrix.CoordinateWithPrevious,
            resources=(
                _RESOURCES['rationalizable_coordination_in_the_matrix__arena']['YELLOW'],
                _RESOURCES['rationalizable_coordination_in_the_matrix__arena']['VIOLET'],
                _RESOURCES['rationalizable_coordination_in_the_matrix__arena']['CYAN'],
            ),
            margin=6,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_a2b_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_a2b_1=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_a2c_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_a2c_1=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_b2a_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_b2a_1=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_b2c_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_b2c_1=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_c2a_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_c2a_1=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_c2b_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=4,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__flip_c2b_1=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=12,
            initial_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            final_target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__pure_a_margin_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
            margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__pure_b_margin_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
            margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__pure_c_margin_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            margin=5,
        ),
    ),
    rationalizable_coordination_in_the_matrix__repeated__resp2prev_margin_0=puppet(
        substrate='rationalizable_coordination_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            coordination_in_the_matrix.CoordinateWithPrevious,
            resources=(
                _RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['YELLOW'],
                _RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['VIOLET'],
                _RESOURCES['rationalizable_coordination_in_the_matrix__repeated']['CYAN'],
            ),
            margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__flip_p2r_0=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=3,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['PAPER'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['SCISSORS'],
            initial_margin=1,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__flip_r2s_0=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=3,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['ROCK'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['SCISSORS'],
            initial_margin=1,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__flip_s2p_0=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=3,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['SCISSORS'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['PAPER'],
            initial_margin=1,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__free_0=saved_model(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='free_0',
    ),
    running_with_scissors_in_the_matrix__arena__paper_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['PAPER'],
            margin=3,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__paper_margin_1=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['PAPER'],
            margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__rock_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['ROCK'],
            margin=3,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__rock_margin_1=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['ROCK'],
            margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__scissors_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['SCISSORS'],
            margin=3,
        ),
    ),
    running_with_scissors_in_the_matrix__arena__scissors_margin_1=puppet(
        substrate='running_with_scissors_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__arena']['SCISSORS'],
            margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__one_shot__paper_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__one_shot',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__one_shot']['PAPER'],
            margin=3,
        ),
    ),
    running_with_scissors_in_the_matrix__one_shot__rock_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__one_shot',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__one_shot']['ROCK'],
            margin=3,
        ),
    ),
    running_with_scissors_in_the_matrix__one_shot__scissors_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__one_shot',
        model='puppet_0',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__one_shot']['SCISSORS'],
            margin=3,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__flip_p2r_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=3,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['PAPER'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['ROCK'],
            initial_margin=1,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__flip_p2r_1=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=2,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['PAPER'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['ROCK'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__flip_r2s_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=3,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['ROCK'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['SCISSORS'],
            initial_margin=1,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__flip_r2s_1=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=2,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['ROCK'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['SCISSORS'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__flip_s2p_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=3,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['SCISSORS'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['PAPER'],
            initial_margin=1,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__flip_s2p_1=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            threshold=2,
            initial_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['SCISSORS'],
            final_target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['PAPER'],
            initial_margin=5,
            final_margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__free_0=saved_model(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='free_0',
    ),
    running_with_scissors_in_the_matrix__repeated__paper_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['PAPER'],
            margin=1,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__paper_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['PAPER'],
            margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__resp2prev_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            running_with_scissors_in_the_matrix.CounterPrevious,
            rock_resource=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['ROCK'],
            paper_resource=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['PAPER'],
            scissors_resource=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['SCISSORS'],
            margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__rock_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['ROCK'],
            margin=1,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__rock_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['ROCK'],
            margin=5,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__scissors_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['SCISSORS'],
            margin=1,
        ),
    ),
    running_with_scissors_in_the_matrix__repeated__scissors_margin_0=puppet(
        substrate='running_with_scissors_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['running_with_scissors_in_the_matrix__repeated']['SCISSORS'],
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_grim_one_strike_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['HARE'],
            threshold=1,
            margin=1,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_grim_one_strike_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['HARE'],
            threshold=1,
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_grim_three_strikes_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['HARE'],
            threshold=3,
            margin=1,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_grim_three_strikes_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['HARE'],
            threshold=3,
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_grim_two_strikes_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['HARE'],
            threshold=2,
            margin=1,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_grim_two_strikes_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__arena']['HARE'],
            threshold=2,
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_hare_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['stag_hunt_in_the_matrix__arena']['HARE'],
            margin=1,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_hare_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['stag_hunt_in_the_matrix__arena']['HARE'],
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_stag_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['stag_hunt_in_the_matrix__arena']['STAG'],
            margin=1,
        ),
    ),
    stag_hunt_in_the_matrix__arena__puppet_stag_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__arena',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['stag_hunt_in_the_matrix__arena']['STAG'],
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_corrigible_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Corrigible,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            threshold=3,
            margin=5,
            tremble_probability=0,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_corrigible_tremble_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Corrigible,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            threshold=3,
            margin=5,
            tremble_probability=0.15,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_flip_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.ScheduledFlip,
            initial_target=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            final_target=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            threshold=3,
            initial_margin=1,
            final_margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_grim_one_strike_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            threshold=1,
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_grim_one_strike_margin_1=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            threshold=1,
            margin=7,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_grim_two_strikes_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            threshold=2,
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_grim_two_strikes_margin_1=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.GrimTrigger,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            threshold=2,
            margin=7,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_hare_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_hare_margin_1=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            margin=7,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_stag_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_stag_margin_1=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.Specialist,
            target=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            margin=7,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_tft_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            tremble_probability=0,
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_tft_margin_1=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            tremble_probability=0,
            margin=7,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_tft_tremble_margin_0=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            tremble_probability=0.15,
            margin=5,
        ),
    ),
    stag_hunt_in_the_matrix__repeated__puppet_tft_tremble_margin_1=puppet(
        substrate='stag_hunt_in_the_matrix__repeated',
        model='puppet_1',
        puppeteer_builder=functools.partial(
            in_the_matrix.TitForTat,
            cooperate_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['STAG'],
            defect_resource=_RESOURCES['stag_hunt_in_the_matrix__repeated']['HARE'],
            tremble_probability=0.15,
            margin=7,
        ),
    ),
    territory__inside_out__aggressor_0=saved_model(
        substrate='territory__inside_out',
        model='aggressor_0',
    ),
    territory__inside_out__aggressor_1=saved_model(
        substrate='territory__inside_out',
        model='aggressor_1',
    ),
    territory__inside_out__aggressor_2=saved_model(
        substrate='territory__inside_out',
        model='aggressor_2',
    ),
    territory__inside_out__aggressor_3=saved_model(
        substrate='territory__inside_out',
        model='aggressor_3',
    ),
    territory__inside_out__aggressor_with_extra_training_0=saved_model(
        substrate='territory__inside_out',
        model='aggressor_with_extra_training_0',
    ),
    territory__inside_out__somewhat_tolerant_bot_0=saved_model(
        substrate='territory__inside_out',
        model='somewhat_tolerant_bot_0',
    ),
    territory__inside_out__somewhat_tolerant_bot_1=saved_model(
        substrate='territory__inside_out',
        model='somewhat_tolerant_bot_1',
    ),
    territory__open__aggressor_0=saved_model(
        substrate='territory__open',
        model='aggressor_0',
    ),
    territory__open__aggressor_1=saved_model(
        substrate='territory__open',
        model='aggressor_1',
    ),
    territory__open__aggressor_2=saved_model(
        substrate='territory__open',
        model='aggressor_2',
    ),
    territory__open__aggressor_3=saved_model(
        substrate='territory__open',
        model='aggressor_3',
    ),
    territory__open__aggressor_with_extra_training_0=saved_model(
        substrate='territory__open',
        model='aggressor_with_extra_training_0',
    ),
    territory__rooms__aggressor_0=saved_model(
        substrate='territory__rooms',
        model='aggressor_0',
    ),
    territory__rooms__aggressor_1=saved_model(
        substrate='territory__rooms',
        model='aggressor_1',
    ),
    territory__rooms__aggressor_2=saved_model(
        substrate='territory__rooms',
        model='aggressor_2',
    ),
    territory__rooms__aggressor_3=saved_model(
        substrate='territory__rooms',
        model='aggressor_3',
    ),
    territory__rooms__aggressor_with_extra_training_0=saved_model(
        substrate='territory__rooms',
        model='aggressor_with_extra_training_0',
    ),
    # keep-sorted end
)
