# Shamelessly stolen from Microsoft Autogen team: thanks to them for this great resource!
# https://github.com/microsoft/autogen/blob/gaia_multiagent_v01_march_1st/autogen/browser_utils.py
import copy
import re

from smolagents.models import MessageRole, Model

def parse_final_answer(response: str) -> str | None:
    """
    Parses the final answer from a model's response string using a robust,
    multi-stage approach.

    It handles different formats the model might output, including:
    - Code blocks like `final_answer("FINAL ANSWER: ...")`
    - Simple quoted strings like `"FINAL ANSWER: ..."`
    - Plain text like `FINAL ANSWER: ...`

    Args:
        response: The raw string output from the language model.

    Returns:
        The cleaned final answer as a string, or None if it cannot be found.
    """
    if not isinstance(response, str):
        return None

    # --- Stage 1: Isolate the most probable "answer block" ---
    # This is based on your excellent suggestion. We look for a high-confidence
    # container for the answer first.

    # Pattern 1: Looks for `final_answer("FINAL ANSWER: ...")`
    match = re.search(r'final_answer\s*\(\s*["\'](.*?)["\']\s*\)', response, re.DOTALL)
    if match:
        # If found, our new, smaller search space is the content of the quotes.
        search_block = match.group(1)
    else:
        # Otherwise, the whole response is our search space.
        search_block = response

    # --- Stage 2: Extract the answer from the block ---
    # Now we run the simple, reliable split logic on our (potentially much smaller)
    # search_block.

    if "FINAL ANSWER:" in search_block:
        # Get everything after the *last* "FINAL ANSWER:"
        final_part = search_block.split("FINAL ANSWER:")[-1]
        
        # Clean up the extracted part
        # 1. Remove leading/trailing whitespace
        # 2. If the result is *still* enclosed in quotes, remove them.
        cleaned_answer = final_part.strip()
        if cleaned_answer.startswith(('"', "'")) and cleaned_answer.endswith(('"', "'")):
            cleaned_answer = cleaned_answer[1:-1]
        
        return cleaned_answer.strip()

    # If "FINAL ANSWER:" was not found in the block, return None.
    return None

def prepare_response(original_task: str, inner_messages, reformulation_model: Model) -> str:
    messages = [
        {
            "role": MessageRole.SYSTEM,
            "content": [
                {
                    "type": "text",
                    "text": f"""Earlier you were asked the following:

{original_task}

Your team then worked diligently to address that request. Read below a transcript of that conversation:""",
                }
            ],
        }
    ]

    # The first message just repeats the question, so remove it
    # if len(inner_messages) > 1:
    #    del inner_messages[0]

    # copy them to this context
    try:
        for message in inner_messages:
            if not message.content:
                continue
            message = copy.deepcopy(message)
            message.role = MessageRole.USER
            messages.append(message)
    except Exception:
        messages += [{"role": MessageRole.ASSISTANT, "content": str(inner_messages)}]

    # ask for the final answer
    messages.append(
        {
            "role": MessageRole.USER,
            "content": [
                {
                    "type": "text",
                    "text": f"""
Read the above conversation and output a FINAL ANSWER to the question. The question is repeated here for convenience:

{original_task}

To output the final solution, follow these rules:

- Output ONLY valid Python function definition(s) that solve the task.
- Keep the EXACT function name and signature as specified (do not add or remove parameters).
- Do NOT include any explanations, comments, print statements, tests, or example usage.
- Do NOT read/write files, do NOT access network, do NOT import third-party libraries.
- If imports are allowed, use ONLY the ones listed. Otherwise, do not include any imports.
- The output must contain ONLY the function definition(s), with no surrounding text.
- Do NOT wrap the code in markdown fences or backticks.

EXAMPLES:

Example 1:
Input task: "Write a function that adds two numbers"
Expected output:
def add(a, b):
    return a + b

Example 2: 
Input task: "Write a function that checks if a number is even"
Expected output:
def is_even(n: int) -> bool:
    return n % 2 == 0

Example 3:
Input task: "Write a function that calculates square perimeter"
Expected output:
def square_perimeter(side_length):
    return 4 * side_length

IMPORTANT: Your response should contain ONLY the function definition(s), exactly like the examples above. No markdown formatting, no explanations, no extra text.
""",
                }
            ],
        }
    )

    response = reformulation_model(messages).content
    
    print("> Reformulation model response: ", response) # for debugging

    # final_answer = response.split("FINAL ANSWER: ")[-1].strip()   # old version, less robust
    final_answer = response
    print("> Reformulated answer: ", final_answer)

    #     if "unable to determine" in final_answer.lower():
    #         messages.append({"role": MessageRole.ASSISTANT, "content": response })
    #         messages.append({"role": MessageRole.USER, "content": [{"type": "text", "text": """
    # I understand that a definitive answer could not be determined. Please make a well-informed EDUCATED GUESS based on the conversation.

    # To output the educated guess, use the following template: EDUCATED GUESS: [YOUR EDUCATED GUESS]
    # Your EDUCATED GUESS should be a number OR as few words as possible OR a comma separated list of numbers and/or strings. DO NOT OUTPUT 'I don't know', 'Unable to determine', etc.
    # ADDITIONALLY, your EDUCATED GUESS MUST adhere to any formatting instructions specified in the original question (e.g., alphabetization, sequencing, units, rounding, decimal places, etc.)
    # If you are asked for a number, express it numerically (i.e., with digits rather than words), don't use commas, and don't include units such as $ or percent signs unless specified otherwise.
    # If you are asked for a string, don't use articles or abbreviations (e.g. cit for cities), unless specified otherwise. Don't output any final sentence punctuation such as '.', '!', or '?'.
    # If you are asked for a comma separated list, apply the above rules depending on whether the elements are numbers or strings.
    # """.strip()}]})

    #         response = model(messages).content
    #         print("\n>>>Making an educated guess.\n", response)
    #         final_answer = response.split("EDUCATED GUESS: ")[-1].strip()

    # 新增一个返回 token usage 的功能
    return final_answer, reformulation_model(messages).token_usage, messages
