import numpy as np

from ribs.emitters._emitter_base import EmitterBase
from maze_domain import mutate

class discreteEmitter(EmitterBase):
    """Emits solutions by doing discrete mutation operators on the previous solutions

    If the archive is empty and ``self._initial_solutions`` is set, a call to
    :meth:`ask` will return ``self._initial_solutions``. If
    ``self._initial_solutions`` is not set, we mutation ``self.x0``
      with standard deviation ``self.sigma``. Otherwise,
    each solution is drawn from a distribution centered at a randomly chosen
    elite with standard deviation ``self.sigma``.

    This is the classic variation operator presented in `Mouret 2015
    <https://arxiv.org/pdf/1504.04909.pdf>`_.

    Args:
        archive (ribs.archives.ArchiveBase): An archive to use when creating and
            inserting solutions. For instance, this can be
            :class:`ribs.archives.GridArchive`.
        sigma (float or array-like): Standard deviation of the Gaussian
            distribution. Note we assume the Gaussian is diagonal, so if this
            argument is an array, it must be 1D.
        x0 (array-like): Center of the Gaussian distribution from which to
            sample solutions when the archive is empty. Must be 1-dimensional.
            This argument is ignored if ``initial_solutions`` is set.
        initial_solutions (array-like): An (n, solution_dim) array of solutions
            to be used when the archive is empty. If this argument is None, then
            solutions will be sampled from a Gaussian distribution centered at
            ``x0`` with standard deviation ``sigma``.
        bounds (None or array-like): Bounds of the solution space. Solutions are
            clipped to these bounds. Pass None to indicate there are no bounds.
            Alternatively, pass an array-like to specify the bounds for each
            dim. Each element in this array-like can be None to indicate no
            bound, or a tuple of ``(lower_bound, upper_bound)``, where
            ``lower_bound`` or ``upper_bound`` may be None to indicate no bound.
        batch_size (int): Number of solutions to return in :meth:`ask`.
        seed (int): Value to seed the random number generator. Set to None to
            avoid a fixed seed.
    Raises:
        ValueError: There is an error in x0 or initial_solutions.
        ValueError: There is an error in the bounds configuration.
    """

    def __init__(self,
                 archive,
                 *,
                 mut_rate = 0.4,
                 x0=None,
                 domain="maze",
                 bounds=[(0, 8)] * 66,
                 initial_solutions=None,
                 batch_size=64,
                 seed=None):
        self._rng = np.random.default_rng(seed)
        self._batch_size = batch_size

        self._mut_rate = mut_rate

        self.domain = domain

        self._x0 = x0
        self._initial_solutions = None

        EmitterBase.__init__(
            self,
            archive,
            solution_dim=archive.solution_dim,
            bounds=bounds,
        )

    @property
    def x0(self):
        """numpy.ndarray: Center of the Gaussian distribution from which to
        sample solutions when the archive is empty (if initial_solutions is not
        set)."""
        return self._x0

    @property
    def initial_solutions(self):
        """numpy.ndarray: The initial solutions which are returned when the
        archive is empty (if x0 is not set)."""
        return self._initial_solutions

    @property
    def sigma(self):
        """float or numpy.ndarray: Standard deviation of the (diagonal) Gaussian
        distribution when the archive is not empty."""
        return self._sigma

    @property
    def batch_size(self):
        """int: Number of solutions to return in :meth:`ask`."""
        return self._batch_size

    def mutate_knights(self, sols_):
        """mutates batch size solutions"""
        r = np.random.rand(self._batch_size)
        sols = np.copy(sols_)
        for i in range(min(sols.shape[0], self._batch_size)):
            if (r[i] < 0.4):
                j = np.random.randint(2, self.solution_dim)
                sols[i][j] = (sols[i][j] + 1) % 8
            elif (r[i] < 0.8):
                j = np.random.randint(2, self.solution_dim)
                sols[i][j] = np.random.randint(0, 8)
            elif (r[i]< 0.95):
                x = np.random.randint(0, 8)
                y = np.random.randint(0, 8)

                sols[i][0] = x
                sols[i][1] = y
            else:
                #reset sols to be random
                sols[i] = np.random.randint(0, 8, size= self.solution_dim)
        return sols

    def mutate_maze(self, sols_):
        """mutates batch size solutions"""
        new_sols = []
        for i in sols_:
            new_sols.append(mutate(i))
        return np.array(new_sols)

    def ask(self):
        if self.domain == "maze":
            return self.ask_maze()
        elif self.domain == "knights":
            return self.ask_knights()
        else:
            raise ValueError("domain must be either 'maze' or 'knights'")

    def ask_knights(self):
        """Creates solutions by adding Gaussian noise to elites in the archive.

        If the archive is empty and ``self._initial_solutions`` is set, we
        return ``self._initial_solutions``. If ``self._initial_solutions`` is
        not set, we draw from Gaussian distribution centered at ``self.x0``
        with standard deviation ``self.sigma``. Otherwise, each solution is
        drawn from a distribution centered at a randomly chosen elite with
        standard deviation ``self.sigma``.

        Returns:
            If the archive is not empty, ``(batch_size, solution_dim)`` array
            -- contains ``batch_size`` new solutions to evaluate. If the
            archive is empty, we return ``self._initial_solutions``, which
            might not have ``batch_size`` solutions.
        """
        if self.archive.empty:
            if self._initial_solutions is not None:
                return np.clip(self._initial_solutions, 0, bounds[0][1]-1)
            parents = np.expand_dims(self.x0, axis=0)
        else:
            parents = self.archive.sample_elites(
                self._batch_size).solution_batch

        return np.clip(self.mutate_knights(parents), self.lower_bounds, self.upper_bounds).astype(int)

    def ask_maze(self):
        """Creates solutions by adding Gaussian noise to elites in the archive.

        If the archive is empty and ``self._initial_solutions`` is set, we
        return ``self._initial_solutions``. If ``self._initial_solutions`` is
        not set, we draw from Gaussian distribution centered at ``self.x0``
        with standard deviation ``self.sigma``. Otherwise, each solution is
        drawn from a distribution centered at a randomly chosen elite with
        standard deviation ``self.sigma``.

        Returns:
            If the archive is not empty, ``(batch_size, solution_dim)`` array
            -- contains ``batch_size`` new solutions to evaluate. If the
            archive is empty, we return ``self._initial_solutions``, which
            might not have ``batch_size`` solutions.
        """
        if self.archive.empty:
            if self._initial_solutions is not None:
                return np.clip(self._initial_solutions, 0, bounds[0][1]-1)
            parents = np.expand_dims(self.x0, axis=0)
        else:
            parents = self.archive.sample_elites(
                self._batch_size).solution_batch

        return np.clip(self.mutate_maze(parents), self.lower_bounds, self.upper_bounds)