import { useState, useEffect } from 'react'
import './App.css'

function App() {
  const [grid, setGrid] = useState(Array(4).fill().map(() => Array(4).fill(0)))
  const [gameOver, setGameOver] = useState(false)
  const [won, setWon] = useState(false)
  const [score, setScore] = useState(0)

  useEffect(() => {
    startNewGame()
  }, [])

  const startNewGame = () => {
    const newGrid = Array(4).fill().map(() => Array(4).fill(0))
    addRandomTile(newGrid)
    addRandomTile(newGrid)
    setGrid(newGrid)
    setGameOver(false)
    setWon(false)
    setScore(0)
  }

  const addRandomTile = (grid) => {
    const emptyCells = []
    for (let i = 0; i < 4; i++) {
      for (let j = 0; j < 4; j++) {
        if (grid[i][j] === 0) {
          emptyCells.push({ i, j })
        }
      }
    }
    if (emptyCells.length > 0) {
      const { i, j } = emptyCells[Math.floor(Math.random() * emptyCells.length)]
      grid[i][j] = Math.random() < 0.9 ? 2 : 4
    }
  }

  const moveTiles = (direction) => {
    if (gameOver) return

    const newGrid = JSON.parse(JSON.stringify(grid))
    let moved = false
    let newScore = score

    const rotateGrid = (grid) => {
      return grid[0].map((_, i) => grid.map(row => row[i]).reverse())
    }

    const moveLeft = (grid) => {
      for (let i = 0; i < 4; i++) {
        let row = grid[i].filter(val => val !== 0)
        for (let j = 0; j < row.length - 1; j++) {
          if (row[j] === row[j + 1]) {
            row[j] *= 2
            newScore += row[j]
            if (row[j] === 2048) setWon(true)
            row[j + 1] = 0
            moved = true
          }
        }
        row = row.filter(val => val !== 0)
        while (row.length < 4) row.push(0)
        if (JSON.stringify(grid[i]) !== JSON.stringify(row)) moved = true
        grid[i] = row
      }
      return grid
    }

    switch (direction) {
      case 'left':
        moveLeft(newGrid)
        break
      case 'right':
        newGrid.forEach(row => row.reverse())
        moveLeft(newGrid)
        newGrid.forEach(row => row.reverse())
        break
      case 'up':
        let rotatedUp = rotateGrid(newGrid)
        rotatedUp.forEach(row => row.reverse())
        moveLeft(rotatedUp)
        rotatedUp.forEach(row => row.reverse())
        newGrid.splice(0, 4, ...rotateGrid(rotateGrid(rotateGrid(rotatedUp))))
        break
      case 'down':
        let rotatedDown = rotateGrid(newGrid)
        moveLeft(rotatedDown)
        newGrid.splice(0, 4, ...rotateGrid(rotateGrid(rotateGrid(rotatedDown))))
        break
      default:
        break
    }

    if (moved) {
      addRandomTile(newGrid)
      setScore(newScore)
      setGrid(newGrid)
      if (isGameOver(newGrid)) {
        setGameOver(true)
      }
    }
  }

  const isGameOver = (grid) => {
    for (let i = 0; i < 4; i++) {
      for (let j = 0; j < 4; j++) {
        if (grid[i][j] === 0) return false
        if (j < 3 && grid[i][j] === grid[i][j + 1]) return false
        if (i < 3 && grid[i][j] === grid[i + 1][j]) return false
      }
    }
    return true
  }

  const reviveGame = () => {
    setGameOver(false)
    const newGrid = JSON.parse(JSON.stringify(grid))
    addRandomTile(newGrid)
    setGrid(newGrid)
  }

  useEffect(() => {
    const handleKeyDown = (e) => {
      if (['ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight', 'w', 'a', 's', 'd'].includes(e.key)) {
        e.preventDefault()
        switch (e.key) {
          case 'ArrowUp':
          case 'w':
            moveTiles('up')
            break
          case 'ArrowDown':
          case 's':
            moveTiles('down')
            break
          case 'ArrowLeft':
          case 'a':
            moveTiles('left')
            break
          case 'ArrowRight':
          case 'd':
            moveTiles('right')
            break
        }
      }
    }

    window.addEventListener('keydown', handleKeyDown)
    return () => window.removeEventListener('keydown', handleKeyDown)
  }, [grid, gameOver])

  const getTileColor = (value) => {
    const colors = {
      0: '#f0e6d2',
      2: '#800000',
      4: '#900000',
      8: '#a00000',
      16: '#b00000',
      32: '#c00000',
      64: '#d00000',
      128: '#e00000',
      256: '#f00000',
      512: '#ff0000',
      1024: '#ff1000',
      2048: '#ff2000'
    }
    return colors[value] || '#ff3000'
  }

  return (
    <div className="app">
      <h1>2048 Game</h1>
      <div className="game-header">
        <div className="score">Score: {score}</div>
        <button onClick={startNewGame}>New Game</button>
      </div>
      <div className="game-container">
        {grid.map((row, i) => (
          <div key={i} className="grid-row">
            {row.map((cell, j) => (
              <div
                key={`${i}-${j}`}
                className="grid-cell"
                style={{
                  backgroundColor: getTileColor(cell),
                  color: cell > 4 ? '#ffffff' : '#000000'
                }}
              >
                {cell !== 0 ? cell : ''}
              </div>
            ))}
          </div>
        ))}
      </div>
      {gameOver && (
        <div className="game-over">
          <h2>Game Over!</h2>
          <button onClick={reviveGame}>Revive</button>
          <button onClick={startNewGame}>New Game</button>
        </div>
      )}
      {won && (
        <div className="game-won">
          <h2>You Won!</h2>
          <button onClick={() => setWon(false)}>Continue Playing</button>
        </div>
      )}
    </div>
  )
}

export default App
