const express = require('express')
const sqlite3 = require('sqlite3').verbose()
const cors = require('cors')
const app = express()
const port = 3001

app.use(cors())
app.use(express.json())

// Initialize database
const db = new sqlite3.Database('./hospital.db', (err) => {
  if (err) {
    console.error(err.message)
  }
  console.log('Connected to the hospital database.')
})

// Create tables
db.serialize(() => {
  db.run(`CREATE TABLE IF NOT EXISTS patients (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    visit_date TEXT,
    diagnosis TEXT,
    comments TEXT
  )`)

  db.run(`CREATE TABLE IF NOT EXISTS inventory (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    quantity INTEGER,
    price REAL,
    category TEXT
  )`)

  db.run(`CREATE TABLE IF NOT EXISTS financial_records (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    date TEXT DEFAULT CURRENT_TIMESTAMP,
    description TEXT,
    amount REAL,
    type TEXT CHECK(type IN ('income', 'expense'))
  )`)

  db.run(`CREATE TABLE IF NOT EXISTS employees (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    username TEXT UNIQUE NOT NULL,
    password TEXT NOT NULL,
    role TEXT NOT NULL
  )`)

  db.run(`CREATE TABLE IF NOT EXISTS daily_reports (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    employee_id INTEGER,
    report_date TEXT DEFAULT CURRENT_TIMESTAMP,
    content TEXT,
    FOREIGN KEY(employee_id) REFERENCES employees(id)
  )`)
})

// Patients API
app.get('/api/patients', (req, res) => {
  db.all('SELECT * FROM patients', [], (err, rows) => {
    if (err) {
      res.status(400).json({ error: err.message })
      return
    }
    res.json(rows)
  })
})

app.post('/api/patients', (req, res) => {
  const { name, visit_date, diagnosis, comments } = req.body
  db.run(
    'INSERT INTO patients (name, visit_date, diagnosis, comments) VALUES (?, ?, ?, ?)',
    [name, visit_date, diagnosis, comments],
    function (err) {
      if (err) {
        res.status(400).json({ error: err.message })
        return
      }
      res.json({ id: this.lastID })
    }
  )
})

// Pharmacy API
app.get('/api/inventory', (req, res) => {
  db.all('SELECT * FROM inventory', [], (err, rows) => {
    if (err) {
      res.status(400).json({ error: err.message })
      return
    }
    res.json(rows)
  })
})

app.post('/api/inventory', (req, res) => {
  const { name, quantity, price, category } = req.body
  db.run(
    'INSERT INTO inventory (name, quantity, price, category) VALUES (?, ?, ?, ?)',
    [name, quantity, price, category],
    function (err) {
      if (err) {
        res.status(400).json({ error: err.message })
        return
      }
      res.json({ id: this.lastID })
    }
  )
})

// Financial API
app.get('/api/financial', (req, res) => {
  db.all('SELECT * FROM financial_records', [], (err, rows) => {
    if (err) {
      res.status(400).json({ error: err.message })
      return
    }
    res.json(rows)
  })
})

app.post('/api/financial', (req, res) => {
  const { description, amount, type } = req.body
  db.run(
    'INSERT INTO financial_records (description, amount, type) VALUES (?, ?, ?)',
    [description, amount, type],
    function (err) {
      if (err) {
        res.status(400).json({ error: err.message })
        return
      }
      res.json({ id: this.lastID })
    }
  )
})

// Reports API
app.get('/api/reports', (req, res) => {
  db.all('SELECT * FROM daily_reports', [], (err, rows) => {
    if (err) {
      res.status(400).json({ error: err.message })
      return
    }
    res.json(rows)
  })
})

// Error handling
app.use((err, req, res, next) => {
  console.error(err.stack)
  res.status(500).send('Something broke!')
})

app.listen(port, () => {
  console.log(`Server running at http://localhost:${port}`)
})
