const express = require('express');
const sqlite3 = require('sqlite3').verbose();
const cors = require('cors');
const bodyParser = require('body-parser');
const app = express();
const PORT = 3001;

app.use(cors());
app.use(bodyParser.json());

const db = new sqlite3.Database('./attendance.db');

// Initialize database
db.serialize(() => {
  db.run(`CREATE TABLE IF NOT EXISTS employees (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    email TEXT UNIQUE NOT NULL,
    password TEXT NOT NULL,
    role TEXT NOT NULL DEFAULT 'employee'
  )`);

  db.run(`CREATE TABLE IF NOT EXISTS attendance (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    employee_id INTEGER NOT NULL,
    date TEXT NOT NULL,
    clock_in TEXT,
    clock_out TEXT,
    status TEXT DEFAULT 'present',
    FOREIGN KEY (employee_id) REFERENCES employees (id)
  )`);
});

// Auth middleware
const authenticate = (req, res, next) => {
  const token = req.headers.authorization;
  if (!token) return res.status(401).send('Unauthorized');
  
  db.get('SELECT * FROM employees WHERE id = ?', [token], (err, user) => {
    if (err || !user) return res.status(401).send('Unauthorized');
    req.user = user;
    next();
  });
};

// Routes
app.post('/api/login', (req, res) => {
  const { email, password } = req.body;
  db.get('SELECT * FROM employees WHERE email = ? AND password = ?', [email, password], (err, user) => {
    if (err || !user) return res.status(401).send('Invalid credentials');
    res.json({ id: user.id, name: user.name, role: user.role });
  });
});

app.post('/api/clock-in', authenticate, (req, res) => {
  const today = new Date().toISOString().split('T')[0];
  const now = new Date().toTimeString().split(' ')[0];
  
  db.run(
    'INSERT INTO attendance (employee_id, date, clock_in) VALUES (?, ?, ?)',
    [req.user.id, today, now],
    function(err) {
      if (err) return res.status(500).send(err.message);
      res.json({ message: 'Clocked in successfully' });
    }
  );
});

app.post('/api/clock-out', authenticate, (req, res) => {
  const today = new Date().toISOString().split('T')[0];
  const now = new Date().toTimeString().split(' ')[0];
  
  db.run(
    'UPDATE attendance SET clock_out = ? WHERE employee_id = ? AND date = ? AND clock_out IS NULL',
    [now, req.user.id, today],
    function(err) {
      if (err) return res.status(500).send(err.message);
      res.json({ message: 'Clocked out successfully' });
    }
  );
});

app.get('/api/my-attendance', authenticate, (req, res) => {
  db.all(
    'SELECT date, clock_in, clock_out, status FROM attendance WHERE employee_id = ? ORDER BY date DESC',
    [req.user.id],
    (err, records) => {
      if (err) return res.status(500).send(err.message);
      res.json(records);
    }
  );
});

app.get('/api/all-attendance', authenticate, (req, res) => {
  if (req.user.role !== 'admin') return res.status(403).send('Forbidden');
  
  db.all(
    `SELECT e.name, a.date, a.clock_in, a.clock_out, a.status 
     FROM attendance a JOIN employees e ON a.employee_id = e.id 
     ORDER BY a.date DESC`,
    (err, records) => {
      if (err) return res.status(500).send(err.message);
      res.json(records);
    }
  );
});

app.listen(PORT, () => {
  console.log(`Server running on http://localhost:${PORT}`);
});
