import { useState, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'

export default function AdminDashboard({ user, onLogout }) {
  const [attendance, setAttendance] = useState([])
  const [stats, setStats] = useState(null)
  const [loading, setLoading] = useState(true)
  const navigate = useNavigate()

  useEffect(() => {
    fetchAllAttendance()
  }, [])

  const fetchAllAttendance = async () => {
    try {
      const response = await fetch('http://localhost:3001/api/all-attendance', {
        headers: { 'Authorization': user.id }
      })
      const data = await response.json()
      setAttendance(data)
      calculateStats(data)
      setLoading(false)
    } catch (err) {
      console.error(err)
    }
  }

  const calculateStats = (data) => {
    const stats = {
      totalRecords: data.length,
      presentCount: data.filter(r => r.status === 'present').length,
      lateCount: data.filter(r => r.clock_in && r.clock_in > '09:00:00').length,
      averageHours: null
    }
    
    const validRecords = data.filter(r => r.clock_in && r.clock_out)
    if (validRecords.length > 0) {
      const totalHours = validRecords.reduce((sum, record) => {
        const inTime = new Date(`2000-01-01T${record.clock_in}`)
        const outTime = new Date(`2000-01-01T${record.clock_out}`)
        return sum + (outTime - inTime) / (1000 * 60 * 60)
      }, 0)
      stats.averageHours = (totalHours / validRecords.length).toFixed(2)
    }
    
    setStats(stats)
  }

  return (
    <div className="admin-dashboard">
      <header>
        <h2>Admin Dashboard</h2>
        <button onClick={onLogout}>Logout</button>
      </header>
      
      {loading ? (
        <p>Loading...</p>
      ) : (
        <>
          <div className="stats">
            <h3>Statistics</h3>
            <div className="stat-cards">
              <div className="stat-card">
                <h4>Total Records</h4>
                <p>{stats.totalRecords}</p>
              </div>
              <div className="stat-card">
                <h4>Present Days</h4>
                <p>{stats.presentCount}</p>
              </div>
              <div className="stat-card">
                <h4>Late Arrivals</h4>
                <p>{stats.lateCount}</p>
              </div>
              <div className="stat-card">
                <h4>Avg. Hours/Day</h4>
                <p>{stats.averageHours || 'N/A'}</p>
              </div>
            </div>
          </div>
          
          <div className="attendance-list">
            <h3>All Attendance Records</h3>
            <table>
              <thead>
                <tr>
                  <th>Employee</th>
                  <th>Date</th>
                  <th>Clock In</th>
                  <th>Clock Out</th>
                  <th>Status</th>
                </tr>
              </thead>
              <tbody>
                {attendance.map((record, index) => (
                  <tr key={index}>
                    <td>{record.name}</td>
                    <td>{record.date}</td>
                    <td>{record.clock_in || '-'}</td>
                    <td>{record.clock_out || '-'}</td>
                    <td>{record.status}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </>
      )}
    </div>
  )
}
