import React, { useState, useEffect } from 'react'
import { Line } from 'react-chartjs-2'
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  Title,
  Tooltip,
  Legend,
} from 'chart.js'

ChartJS.register(
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  Title,
  Tooltip,
  Legend
)

const App = () => {
  const [realTimeData, setRealTimeData] = useState({
    generation: 0,
    consumption: 0,
    battery: 0
  })
  const [historicalData, setHistoricalData] = useState([])
  const [timeRange, setTimeRange] = useState('day')

  // Mock real-time data updates
  useEffect(() => {
    const interval = setInterval(() => {
      setRealTimeData({
        generation: Math.random() * 10,
        consumption: Math.random() * 5,
        battery: Math.random() * 100
      })
    }, 3000)
    return () => clearInterval(interval)
  }, [])

  // Mock historical data
  useEffect(() => {
    const generateHistoricalData = () => {
      const data = []
      const points = timeRange === 'day' ? 24 : timeRange === 'week' ? 7 : 30
      
      for (let i = 0; i < points; i++) {
        data.push({
          time: timeRange === 'day' ? `${i}:00` : `Day ${i + 1}`,
          generation: Math.random() * 10,
          consumption: Math.random() * 5,
          battery: Math.random() * 100
        })
      }
      return data
    }

    setHistoricalData(generateHistoricalData())
  }, [timeRange])

  const chartData = {
    labels: historicalData.map(data => data.time),
    datasets: [
      {
        label: 'Power Generation (kW)',
        data: historicalData.map(data => data.generation),
        borderColor: 'darkolivegreen',
        backgroundColor: 'rgba(85, 107, 47, 0.5)',
      },
      {
        label: 'Power Consumption (kW)',
        data: historicalData.map(data => data.consumption),
        borderColor: 'rgba(85, 107, 47, 0.7)',
        backgroundColor: 'rgba(85, 107, 47, 0.3)',
      }
    ]
  }

  const batteryChartData = {
    labels: historicalData.map(data => data.time),
    datasets: [
      {
        label: 'Battery Level (%)',
        data: historicalData.map(data => data.battery),
        borderColor: 'darkolivegreen',
        backgroundColor: 'rgba(85, 107, 47, 0.5)',
        fill: true
      }
    ]
  }

  return (
    <div style={{ padding: '20px', maxWidth: '1200px', margin: '0 auto' }}>
      <h1 style={{ color: 'darkolivegreen', textAlign: 'center' }}>Solar Dashboard</h1>
      
      <div style={{ 
        display: 'grid', 
        gridTemplateColumns: 'repeat(3, 1fr)',
        gap: '20px',
        marginBottom: '30px'
      }}>
        <div style={{
          backgroundColor: 'white',
          padding: '20px',
          borderRadius: '8px',
          boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
          border: '1px solid darkolivegreen'
        }}>
          <h3 style={{ color: 'darkolivegreen' }}>Power Generation</h3>
          <p style={{ fontSize: '24px', fontWeight: 'bold' }}>
            {realTimeData.generation.toFixed(2)} kW
          </p>
        </div>
        
        <div style={{
          backgroundColor: 'white',
          padding: '20px',
          borderRadius: '8px',
          boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
          border: '1px solid darkolivegreen'
        }}>
          <h3 style={{ color: 'darkolivegreen' }}>Power Consumption</h3>
          <p style={{ fontSize: '24px', fontWeight: 'bold' }}>
            {realTimeData.consumption.toFixed(2)} kW
          </p>
        </div>
        
        <div style={{
          backgroundColor: 'white',
          padding: '20px',
          borderRadius: '8px',
          boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
          border: '1px solid darkolivegreen'
        }}>
          <h3 style={{ color: 'darkolivegreen' }}>Battery Level</h3>
          <p style={{ fontSize: '24px', fontWeight: 'bold' }}>
            {realTimeData.battery.toFixed(0)}%
          </p>
        </div>
      </div>
      
      <div style={{ 
        display: 'flex', 
        justifyContent: 'center',
        marginBottom: '20px',
        gap: '10px'
      }}>
        <button 
          onClick={() => setTimeRange('day')}
          style={{
            padding: '8px 16px',
            backgroundColor: timeRange === 'day' ? 'darkolivegreen' : 'white',
            color: timeRange === 'day' ? 'white' : 'darkolivegreen',
            border: '1px solid darkolivegreen',
            borderRadius: '4px',
            cursor: 'pointer'
          }}
        >
          Day
        </button>
        <button 
          onClick={() => setTimeRange('week')}
          style={{
            padding: '8px 16px',
            backgroundColor: timeRange === 'week' ? 'darkolivegreen' : 'white',
            color: timeRange === 'week' ? 'white' : 'darkolivegreen',
            border: '1px solid darkolivegreen',
            borderRadius: '4px',
            cursor: 'pointer'
          }}
        >
          Week
        </button>
        <button 
          onClick={() => setTimeRange('month')}
          style={{
            padding: '8px 16px',
            backgroundColor: timeRange === 'month' ? 'darkolivegreen' : 'white',
            color: timeRange === 'month' ? 'white' : 'darkolivegreen',
            border: '1px solid darkolivegreen',
            borderRadius: '4px',
            cursor: 'pointer'
          }}
        >
          Month
        </button>
      </div>
      
      <div style={{ 
        backgroundColor: 'white',
        padding: '20px',
        borderRadius: '8px',
        boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
        border: '1px solid darkolivegreen',
        marginBottom: '20px'
      }}>
        <h2 style={{ color: 'darkolivegreen' }}>Power Generation vs Consumption</h2>
        <Line data={chartData} />
      </div>
      
      <div style={{ 
        backgroundColor: 'white',
        padding: '20px',
        borderRadius: '8px',
        boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
        border: '1px solid darkolivegreen'
      }}>
        <h2 style={{ color: 'darkolivegreen' }}>Battery Level History</h2>
        <Line data={batteryChartData} />
      </div>
    </div>
  )
}

export default App
