require('dotenv').config()
const express = require('express')
const cors = require('cors')
const sqlite3 = require('sqlite3').verbose()
const bcrypt = require('bcryptjs')
const jwt = require('jsonwebtoken')
const { v4: uuidv4 } = require('uuid')

const app = express()
const PORT = process.env.PORT || 5000
const JWT_SECRET = process.env.JWT_SECRET || 'your-secret-key'

app.use(cors())
app.use(express.json())

// Database setup
const db = new sqlite3.Database('./sports-recruitment.db', (err) => {
  if (err) {
    console.error('Database connection error:', err)
  } else {
    console.log('Connected to SQLite database')
    initializeDatabase()
  }
})

function initializeDatabase() {
  db.serialize(() => {
    // Create tables if they don't exist
    db.run(`
      CREATE TABLE IF NOT EXISTS users (
        id TEXT PRIMARY KEY,
        email TEXT UNIQUE NOT NULL,
        password TEXT NOT NULL,
        name TEXT NOT NULL,
        user_type TEXT NOT NULL CHECK(user_type IN ('athlete', 'organization')),
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
      )
    `)

    db.run(`
      CREATE TABLE IF NOT EXISTS jobs (
        id TEXT PRIMARY KEY,
        organization_id TEXT NOT NULL,
        title TEXT NOT NULL,
        description TEXT NOT NULL,
        sport TEXT NOT NULL,
        location TEXT NOT NULL,
        salary_range TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (organization_id) REFERENCES users(id)
      )
    `)

    db.run(`
      CREATE TABLE IF NOT EXISTS applications (
        id TEXT PRIMARY KEY,
        job_id TEXT NOT NULL,
        athlete_id TEXT NOT NULL,
        status TEXT DEFAULT 'pending',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (job_id) REFERENCES jobs(id),
        FOREIGN KEY (athlete_id) REFERENCES users(id)
      )
    `)

    db.run(`
      CREATE TABLE IF NOT EXISTS notifications (
        id TEXT PRIMARY KEY,
        user_id TEXT NOT NULL,
        message TEXT NOT NULL,
        is_read BOOLEAN DEFAULT FALSE,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id)
      )
    `)
  })
}

// Authentication middleware
const authenticate = (req, res, next) => {
  const token = req.headers.authorization?.split(' ')[1]
  if (!token) return res.status(401).json({ error: 'Unauthorized' })

  try {
    const decoded = jwt.verify(token, JWT_SECRET)
    req.user = decoded
    next()
  } catch (err) {
    res.status(401).json({ error: 'Invalid token' })
  }
}

// Auth routes
app.post('/api/register', async (req, res) => {
  const { email, password, name, user_type } = req.body
  const id = uuidv4()
  const hashedPassword = await bcrypt.hash(password, 10)

  db.run(
    'INSERT INTO users (id, email, password, name, user_type) VALUES (?, ?, ?, ?, ?)',
    [id, email, hashedPassword, name, user_type],
    function(err) {
      if (err) {
        return res.status(400).json({ error: 'Email already exists' })
      }
      const token = jwt.sign({ id, email, user_type }, JWT_SECRET, { expiresIn: '1d' })
      res.json({ token, user: { id, email, name, user_type } })
    }
  )
})

app.post('/api/login', async (req, res) => {
  const { email, password } = req.body

  db.get('SELECT * FROM users WHERE email = ?', [email], async (err, user) => {
    if (err || !user) {
      return res.status(400).json({ error: 'Invalid credentials' })
    }

    const isValid = await bcrypt.compare(password, user.password)
    if (!isValid) {
      return res.status(400).json({ error: 'Invalid credentials' })
    }

    const token = jwt.sign(
      { id: user.id, email: user.email, user_type: user.user_type },
      JWT_SECRET,
      { expiresIn: '1d' }
    )
    res.json({ 
      token, 
      user: {
        id: user.id,
        email: user.email,
        name: user.name,
        user_type: user.user_type
      }
    })
  })
})

// Job routes
app.get('/api/jobs', (req, res) => {
  const { sport, location } = req.query
  let query = 'SELECT j.*, u.name as organization_name FROM jobs j JOIN users u ON j.organization_id = u.id'
  const params = []

  if (sport || location) {
    query += ' WHERE'
    if (sport) {
      query += ' j.sport = ?'
      params.push(sport)
    }
    if (location) {
      if (sport) query += ' AND'
      query += ' j.location = ?'
      params.push(location)
    }
  }

  db.all(query, params, (err, jobs) => {
    if (err) {
      return res.status(500).json({ error: 'Failed to fetch jobs' })
    }
    res.json(jobs)
  })
})

app.get('/api/jobs/:id', (req, res) => {
  const { id } = req.params
  db.get(
    'SELECT j.*, u.name as organization_name FROM jobs j JOIN users u ON j.organization_id = u.id WHERE j.id = ?',
    [id],
    (err, job) => {
      if (err || !job) {
        return res.status(404).json({ error: 'Job not found' })
      }
      res.json(job)
    }
  )
})

app.post('/api/jobs', authenticate, (req, res) => {
  if (req.user.user_type !== 'organization') {
    return res.status(403).json({ error: 'Only organizations can post jobs' })
  }

  const { title, description, sport, location, salary_range } = req.body
  const id = uuidv4()

  db.run(
    'INSERT INTO jobs (id, organization_id, title, description, sport, location, salary_range) VALUES (?, ?, ?, ?, ?, ?, ?)',
    [id, req.user.id, title, description, sport, location, salary_range],
    function(err) {
      if (err) {
        return res.status(400).json({ error: 'Failed to create job' })
      }
      res.json({ id, organization_id: req.user.id, title, description, sport, location, salary_range })
    }
  )
})

// Application routes
app.post('/api/applications', authenticate, (req, res) => {
  if (req.user.user_type !== 'athlete') {
    return res.status(403).json({ error: 'Only athletes can apply for jobs' })
  }

  const { job_id } = req.body
  const id = uuidv4()

  // First check if job exists
  db.get('SELECT * FROM jobs WHERE id = ?', [job_id], (err, job) => {
    if (err || !job) {
      return res.status(404).json({ error: 'Job not found' })
    }

    // Check if already applied
    db.get(
      'SELECT * FROM applications WHERE job_id = ? AND athlete_id = ?',
      [job_id, req.user.id],
      (err, existingApp) => {
        if (err) {
          return res.status(500).json({ error: 'Database error' })
        }
        if (existingApp) {
          return res.status(400).json({ error: 'Already applied to this job' })
        }

        // Create application
        db.run(
          'INSERT INTO applications (id, job_id, athlete_id) VALUES (?, ?, ?)',
          [id, job_id, req.user.id],
          function(err) {
            if (err) {
              return res.status(500).json({ error: 'Failed to create application' })
            }

            // Create notification for organization
            const notificationId = uuidv4()
            db.run(
              'INSERT INTO notifications (id, user_id, message) VALUES (?, ?, ?)',
              [notificationId, job.organization_id, `New application for your job: ${job.title}`],
              () => {
                res.json({ id, job_id, athlete_id: req.user.id, status: 'pending' })
              }
            )
          }
        )
      }
    )
  })
})

app.get('/api/applications', authenticate, (req, res) => {
  let query, params

  if (req.user.user_type === 'athlete') {
    query = `
      SELECT a.*, j.title as job_title, j.sport, j.location, u.name as organization_name 
      FROM applications a
      JOIN jobs j ON a.job_id = j.id
      JOIN users u ON j.organization_id = u.id
      WHERE a.athlete_id = ?
    `
    params = [req.user.id]
  } else {
    query = `
      SELECT a.*, j.title as job_title, j.sport, j.location, u.name as athlete_name 
      FROM applications a
      JOIN jobs j ON a.job_id = j.id
      JOIN users u ON a.athlete_id = u.id
      WHERE j.organization_id = ?
    `
    params = [req.user.id]
  }

  db.all(query, params, (err, applications) => {
    if (err) {
      return res.status(500).json({ error: 'Failed to fetch applications' })
    }
    res.json(applications)
  })
})

app.put('/api/applications/:id/status', authenticate, (req, res) => {
  if (req.user.user_type !== 'organization') {
    return res.status(403).json({ error: 'Only organizations can update application status' })
  }

  const { id } = req.params
  const { status } = req.body

  // First verify the application belongs to this organization
  db.get(
    'SELECT a.* FROM applications a JOIN jobs j ON a.job_id = j.id WHERE a.id = ? AND j.organization_id = ?',
    [id, req.user.id],
    (err, application) => {
      if (err || !application) {
        return res.status(404).json({ error: 'Application not found or unauthorized' })
      }

      db.run(
        'UPDATE applications SET status = ? WHERE id = ?',
        [status, id],
        function(err) {
          if (err) {
            return res.status(500).json({ error: 'Failed to update application' })
          }

          // Create notification for athlete
          const notificationId = uuidv4()
          db.run(
            'INSERT INTO notifications (id, user_id, message) VALUES (?, ?, ?)',
            [notificationId, application.athlete_id, `Your application status changed to ${status}`],
            () => {
              res.json({ id, status })
            }
          )
        }
      )
    }
  )
})

// Notification routes
app.get('/api/notifications', authenticate, (req, res) => {
  db.all(
    'SELECT * FROM notifications WHERE user_id = ? ORDER BY created_at DESC',
    [req.user.id],
    (err, notifications) => {
      if (err) {
        return res.status(500).json({ error: 'Failed to fetch notifications' })
      }
      res.json(notifications)
    }
  )
})

app.put('/api/notifications/:id/read', authenticate, (req, res) => {
  const { id } = req.params

  db.run(
    'UPDATE notifications SET is_read = TRUE WHERE id = ? AND user_id = ?',
    [id, req.user.id],
    function(err) {
      if (err) {
        return res.status(500).json({ error: 'Failed to update notification' })
      }
      if (this.changes === 0) {
        return res.status(404).json({ error: 'Notification not found or unauthorized' })
      }
      res.json({ success: true })
    }
  )
})

app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
