const express = require('express')
const cors = require('cors')
const sqlite3 = require('sqlite3').verbose()
const db = new sqlite3.Database('./database.db')

const app = express()
app.use(cors())
app.use(express.json())

// Create tables
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS courses (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      title TEXT NOT NULL,
      description TEXT,
      instructor TEXT,
      duration INTEGER,
      level TEXT
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS assignments (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      course_id INTEGER,
      title TEXT NOT NULL,
      description TEXT,
      deadline TEXT,
      FOREIGN KEY (course_id) REFERENCES courses(id)
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS submissions (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      assignment_id INTEGER,
      student_id INTEGER,
      submission_text TEXT,
      submission_date TEXT,
      grade INTEGER,
      FOREIGN KEY (assignment_id) REFERENCES assignments(id)
    )
  `)
})

// API Routes
app.get('/api/courses', (req, res) => {
  db.all('SELECT * FROM courses', [], (err, rows) => {
    if (err) {
      res.status(500).json({ error: err.message })
      return
    }
    res.json(rows)
  })
})

app.get('/api/courses/:id', (req, res) => {
  db.get('SELECT * FROM courses WHERE id = ?', [req.params.id], (err, row) => {
    if (err) {
      res.status(500).json({ error: err.message })
      return
    }
    res.json(row)
  })
})

app.get('/api/assignments/:courseId', (req, res) => {
  db.all(
    'SELECT * FROM assignments WHERE course_id = ?',
    [req.params.courseId],
    (err, rows) => {
      if (err) {
        res.status(500).json({ error: err.message })
        return
      }
      res.json(rows)
    }
  )
})

app.post('/api/submit', (req, res) => {
  const { assignment_id, student_id, submission_text } = req.body
  const submission_date = new Date().toISOString()

  db.run(
    'INSERT INTO submissions (assignment_id, student_id, submission_text, submission_date) VALUES (?, ?, ?, ?)',
    [assignment_id, student_id, submission_text, submission_date],
    function (err) {
      if (err) {
        res.status(500).json({ error: err.message })
        return
      }
      res.json({ id: this.lastID })
    }
  )
})

app.get('/api/grades/:studentId', (req, res) => {
  db.all(
    `SELECT s.id, a.title as assignment_title, c.title as course_title, 
     s.grade, s.submission_date 
     FROM submissions s
     JOIN assignments a ON s.assignment_id = a.id
     JOIN courses c ON a.course_id = c.id
     WHERE s.student_id = ?`,
    [req.params.studentId],
    (err, rows) => {
      if (err) {
        res.status(500).json({ error: err.message })
        return
      }
      res.json(rows)
    }
  )
})

const PORT = 5000
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
