const express = require('express')
const http = require('http')
const { Server } = require('socket.io')
const cors = require('cors')
const libsql = require('libsql')

const app = express()
app.use(cors())
const server = http.createServer(app)
const io = new Server(server, {
  cors: {
    origin: "http://localhost:3000",
    methods: ["GET", "POST"]
  }
})

// Initialize database
const db = new libsql.Database('poker.db')

// Game state management
const tables = new Map()
const players = new Map()

// Poker game constants
const GameStates = {
  WAITING: 'waiting',
  PREFLOP: 'preflop',
  FLOP: 'flop',
  TURN: 'turn',
  RIVER: 'river',
  SHOWDOWN: 'showdown'
}

const Actions = {
  FOLD: 'fold',
  CHECK: 'check',
  CALL: 'call',
  RAISE: 'raise',
  ALL_IN: 'all-in'
}

// Helper functions
const createDeck = () => {
  const suits = ['♥', '♦', '♣', '♠']
  const values = ['2', '3', '4', '5', '6', '7', '8', '9', '10', 'J', 'Q', 'K', 'A']
  let deck = []
  for (let suit of suits) {
    for (let value of values) {
      deck.push(`${value}${suit}`)
    }
  }
  return deck.sort(() => Math.random() - 0.5)
}

io.on('connection', (socket) => {
  console.log('User connected:', socket.id)

  // Player authentication
  socket.on('authenticate', async ({ token }) => {
    try {
      // In a real app, verify JWT token here
      const user = await db.get('SELECT * FROM users WHERE id = ?', [token])
      if (user) {
        players.set(socket.id, {
          id: user.id.toString(),
          username: user.username,
          chips: user.chips,
          socketId: socket.id
        })
        socket.emit('authenticated', user)
      }
    } catch (err) {
      console.error('Authentication error:', err)
    }
  })

  // Table management
  socket.on('createTable', ({ name, buyIn }) => {
    const tableId = `table_${Date.now()}`
    const newTable = {
      id: tableId,
      name,
      buyIn,
      maxPlayers: 6,
      players: [],
      spectators: [],
      deck: createDeck(),
      state: GameStates.WAITING,
      smallBlind: 10,
      bigBlind: 20,
      dealerPosition: 0,
      currentPlayer: null,
      communityCards: [],
      pot: 0,
      currentBet: 0
    }
    tables.set(tableId, newTable)
    io.emit('tablesUpdated', Array.from(tables.values()))
  })

  socket.on('joinTable', ({ tableId }) => {
    const table = tables.get(tableId)
    const player = players.get(socket.id)
    
    if (table && player) {
      if (table.players.length >= table.maxPlayers) {
        table.spectators.push(player)
      } else {
        table.players.push({
          id: player.id,
          username: player.username,
          chips: player.chips,
          cards: [],
          folded: false,
          currentBet: 0,
          allIn: false
        })
      }
      
      socket.join(tableId)
      updateTableState(tableId)
    }
  })

  socket.on('leaveTable', ({ tableId }) => {
    const table = tables.get(tableId)
    const player = players.get(socket.id)
    
    if (table && player) {
      table.players = table.players.filter(p => p.id !== player.id)
      table.spectators = table.spectators.filter(p => p.id !== player.id)
      socket.leave(tableId)
      updateTableState(tableId)
    }
  })

  // Game actions
  socket.on('playerAction', ({ tableId, action, amount }) => {
    const table = tables.get(tableId)
    const player = players.get(socket.id)
    
    if (!table || !player || table.state === GameStates.WAITING) return

    const playerIdx = table.players.findIndex(p => p.id === player.id)
    if (playerIdx === -1 || playerIdx !== table.currentPlayer) return

    switch (action) {
      case Actions.FOLD:
        table.players[playerIdx].folded = true
        break
      case Actions.CHECK:
        if (table.currentBet > 0) return // Can't check if there's a bet
        break
      case Actions.CALL:
        const callAmount = table.currentBet - table.players[playerIdx].currentBet
        table.players[playerIdx].chips -= callAmount
        table.players[playerIdx].currentBet += callAmount
        table.pot += callAmount
        break
      case Actions.RAISE:
        if (amount <= table.currentBet) return // Must raise more than current bet
        table.players[playerIdx].chips -= amount
        table.players[playerIdx].currentBet += amount
        table.currentBet = table.players[playerIdx].currentBet
        table.pot += amount
        break
      case Actions.ALL_IN:
        const allInAmount = table.players[playerIdx].chips
        table.players[playerIdx].chips = 0
        table.players[playerIdx].currentBet += allInAmount
        table.pot += allInAmount
        if (allInAmount > table.currentBet) {
          table.currentBet = allInAmount
        }
        break
    }

    advanceGameState(tableId)
  })

  socket.on('startGame', ({ tableId }) => {
    const table = tables.get(tableId)
    if (table && table.players.length >= 2) {
      startNewHand(tableId)
    }
  })

  socket.on('disconnect', () => {
    console.log('User disconnected:', socket.id)
    players.delete(socket.id)
  })

  // Helper functions
  function updateTableState(tableId) {
    const table = tables.get(tableId)
    if (table) {
      io.to(tableId).emit('tableUpdated', table)
      io.emit('tablesUpdated', Array.from(tables.values()))
    }
  }

  function startNewHand(tableId) {
    const table = tables.get(tableId)
    if (!table) return

    // Reset hand state
    table.deck = createDeck()
    table.communityCards = []
    table.state = GameStates.PREFLOP
    table.pot = 0
    table.currentBet = 0

    // Deal cards
    table.players.forEach(player => {
      player.cards = [table.deck.pop(), table.deck.pop()]
      player.folded = false
      player.currentBet = 0
      player.allIn = false
    })

    // Post blinds
    const sbPlayer = table.players[table.dealerPosition]
    const bbPlayer = table.players[(table.dealerPosition + 1) % table.players.length]
    
    sbPlayer.chips -= table.smallBlind
    sbPlayer.currentBet = table.smallBlind
    bbPlayer.chips -= table.bigBlind
    bbPlayer.currentBet = table.bigBlind
    
    table.pot = table.smallBlind + table.bigBlind
    table.currentBet = table.bigBlind
    table.currentPlayer = (table.dealerPosition + 2) % table.players.length

    updateTableState(tableId)
  }

  function advanceGameState(tableId) {
    const table = tables.get(tableId)
    if (!table) return

    // Check if round is complete
    const activePlayers = table.players.filter(p => !p.folded && !p.allIn)
    if (activePlayers.length <= 1) {
      // Go to showdown or end hand
      table.state = GameStates.SHOWDOWN
      determineWinner(tableId)
      return
    }

    // Move to next player
    table.currentPlayer = (table.currentPlayer + 1) % table.players.length

    // Check if betting round is complete
    if (table.players.every(p => 
      p.folded || p.allIn || 
      p.currentBet === table.currentBet || 
      (p.currentBet === 0 && table.currentBet === 0)
    )) {
      advanceGameStage(tableId)
    }

    updateTableState(tableId)
  }

  function advanceGameStage(tableId) {
    const table = tables.get(tableId)
    if (!table) return

    switch (table.state) {
      case GameStates.PREFLOP:
        table.state = GameStates.FLOP
        table.communityCards.push(table.deck.pop(), table.deck.pop(), table.deck.pop())
        break
      case GameStates.FLOP:
        table.state = GameStates.TURN
        table.communityCards.push(table.deck.pop())
        break
      case GameStates.TURN:
        table.state = GameStates.RIVER
        table.communityCards.push(table.deck.pop())
        break
      case GameStates.RIVER:
        table.state = GameStates.SHOWDOWN
        determineWinner(tableId)
        return
    }

    // Reset betting for new round
    table.currentBet = 0
    table.players.forEach(p => p.currentBet = 0)
    table.currentPlayer = (table.dealerPosition + 1) % table.players.length

    updateTableState(tableId)
  }

  function determineWinner(tableId) {
    const table = tables.get(tableId)
    if (!table) return

    // In a real implementation, this would evaluate poker hands
    const activePlayers = table.players.filter(p => !p.folded)
    if (activePlayers.length === 1) {
      // Only one player left - they win
      const winner = activePlayers[0]
      winner.chips += table.pot
      
      // Record game result
      db.exec(`
        INSERT INTO games (table_id, winner_id) VALUES (?, ?);
        UPDATE users SET chips = ? WHERE id = ?;
      `, [table.id, winner.id, winner.chips, winner.id])
    } else {
      // For now, just split pot among remaining players
      const chipsPerPlayer = Math.floor(table.pot / activePlayers.length)
      activePlayers.forEach(player => {
        player.chips += chipsPerPlayer
        db.exec('UPDATE users SET chips = ? WHERE id = ?', [player.chips, player.id])
      })
    }

    // Prepare for next hand
    table.dealerPosition = (table.dealerPosition + 1) % table.players.length
    setTimeout(() => startNewHand(tableId), 5000) // Start new hand after 5 seconds
  }
})

const PORT = 5000
server.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
