import React, { useState, useEffect } from 'react'
import { useParams } from 'react-router-dom'
import { Box, Typography, Button, Card, CardContent, Chip, Avatar } from '@mui/material'
import { io } from 'socket.io-client'

const GameTable = () => {
  const { id } = useParams()
  const [gameState, setGameState] = useState(null)
  const [message, setMessage] = useState('')
  const [messages, setMessages] = useState([])
  const socket = io('http://localhost:5000')

  useEffect(() => {
    socket.emit('joinTable', { tableId: id })

    socket.on('gameState', (state) => {
      setGameState(state)
    })

    socket.on('chatMessage', (msg) => {
      setMessages((prev) => [...prev, msg])
    })

    return () => {
      socket.disconnect()
    }
  }, [id])

  const sendMessage = () => {
    if (message.trim()) {
      socket.emit('chatMessage', { tableId: id, message })
      setMessage('')
    }
  }

  const handleAction = (action, amount = 0) => {
    socket.emit('playerAction', { tableId: id, action, amount })
  }

  if (!gameState) return <Typography>Loading game...</Typography>

  return (
    <Box sx={{ p: 4 }}>
      <Typography variant="h4" gutterBottom>Table: {gameState.tableName}</Typography>
      
      {/* Community Cards */}
      <Box sx={{ display: 'flex', justifyContent: 'center', gap: 2, mb: 4 }}>
        {gameState.communityCards.map((card, index) => (
          <Card key={index} sx={{ width: 80, height: 120 }}>
            <CardContent sx={{ display: 'flex', justifyContent: 'center', alignItems: 'center', height: '100%' }}>
              {card ? <Typography variant="h5">{card}</Typography> : null}
            </CardContent>
          </Card>
        ))}
      </Box>

      {/* Players */}
      <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 4, mb: 4 }}>
        {gameState.players.map((player, index) => (
          <Card key={index} sx={{ p: 2, minWidth: 200 }}>
            <Box sx={{ display: 'flex', alignItems: 'center', mb: 1 }}>
              <Avatar sx={{ mr: 1 }}>{player.name.charAt(0)}</Avatar>
              <Typography>{player.name}</Typography>
              {player.isActive && <Chip label="Active" color="primary" size="small" sx={{ ml: 1 }} />}
            </Box>
            <Typography>Chips: {player.chips}</Typography>
            <Typography>Bet: {player.currentBet}</Typography>
            {player.cards && (
              <Box sx={{ display: 'flex', gap: 1, mt: 1 }}>
                {player.cards.map((card, i) => (
                  <Card key={i} sx={{ width: 60, height: 90 }}>
                    <CardContent sx={{ display: 'flex', justifyContent: 'center', alignItems: 'center', height: '100%' }}>
                      <Typography>{card}</Typography>
                    </CardContent>
                  </Card>
                ))}
              </Box>
            )}
          </Card>
        ))}
      </Box>

      {/* Action Buttons */}
      {gameState.currentPlayer && (
        <Box sx={{ display: 'flex', gap: 2, mb: 4 }}>
          <Button variant="contained" onClick={() => handleAction('fold')}>Fold</Button>
          <Button variant="contained" onClick={() => handleAction('check')}>Check</Button>
          <Button variant="contained" onClick={() => handleAction('call')}>Call</Button>
          <Button variant="contained" onClick={() => handleAction('raise', 20)}>Raise 20</Button>
          <Button variant="contained" onClick={() => handleAction('all-in')}>All-In</Button>
        </Box>
      )}

      {/* Chat */}
      <Box sx={{ mt: 4 }}>
        <Typography variant="h6">Chat</Typography>
        <Box sx={{ height: 200, overflowY: 'auto', border: '1px solid gray', p: 2, mb: 2 }}>
          {messages.map((msg, i) => (
            <Typography key={i}>{msg.sender}: {msg.message}</Typography>
          ))}
        </Box>
        <Box sx={{ display: 'flex', gap: 2 }}>
          <TextField
            fullWidth
            value={message}
            onChange={(e) => setMessage(e.target.value)}
            onKeyPress={(e) => e.key === 'Enter' && sendMessage()}
          />
          <Button variant="contained" onClick={sendMessage}>Send</Button>
        </Box>
      </Box>
    </Box>
  )
}

export default GameTable
