import { useState, useEffect } from 'react'
import { WiDaySunny, WiRain, WiCloudy, WiHumidity } from 'react-icons/wi'
import { Line } from 'react-chartjs-2'
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  Title,
  Tooltip,
  Legend,
} from 'chart.js'

ChartJS.register(
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  Title,
  Tooltip,
  Legend
)

function App() {
  const [weatherData, setWeatherData] = useState(null)
  const [aqiData, setAqiData] = useState([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    // Simulate fetching real-time data
    const fetchData = () => {
      // Mock weather data
      const mockWeather = {
        temperature: Math.round(Math.random() * 30 + 10),
        humidity: Math.round(Math.random() * 100),
        pressure: Math.round(Math.random() * 50 + 950),
        windSpeed: (Math.random() * 20).toFixed(1),
        condition: ['Sunny', 'Cloudy', 'Rainy'][Math.floor(Math.random() * 3)]
      }

      // Generate 14 days of AQI data
      const newAqiData = Array.from({ length: 14 }, (_, i) => ({
        date: new Date(Date.now() - (13 - i) * 24 * 60 * 60 * 1000).toLocaleDateString(),
        aqi: Math.round(Math.random() * 300),
        pm25: Math.round(Math.random() * 200),
        pm10: Math.round(Math.random() * 300),
        co: (Math.random() * 10).toFixed(2),
        no2: (Math.random() * 50).toFixed(2)
      }))

      setWeatherData(mockWeather)
      setAqiData(newAqiData)
      setLoading(false)
    }

    fetchData()
    const interval = setInterval(fetchData, 60000) // Update every minute

    return () => clearInterval(interval)
  }, [])

  const aqiChartData = {
    labels: aqiData.map(item => item.date),
    datasets: [
      {
        label: 'AQI',
        data: aqiData.map(item => item.aqi),
        borderColor: 'rgb(75, 192, 192)',
        backgroundColor: 'rgba(75, 192, 192, 0.5)',
      },
      {
        label: 'PM2.5',
        data: aqiData.map(item => item.pm25),
        borderColor: 'rgb(255, 99, 132)',
        backgroundColor: 'rgba(255, 99, 132, 0.5)',
      }
    ]
  }

  const aqiChartOptions = {
    responsive: true,
    plugins: {
      legend: {
        position: 'top',
      },
      title: {
        display: true,
        text: 'Air Quality Index (Last 14 Days)',
      },
    },
  }

  return (
    <div className="app">
      <header>
        <h1>Environmental Warning System</h1>
      </header>

      <main>
        {loading ? (
          <div className="loading">Loading data...</div>
        ) : (
          <>
            <section className="current-data">
              <h2>Current Conditions</h2>
              <div className="weather-card">
                <div className="weather-icon">
                  {weatherData.condition === 'Sunny' && <WiDaySunny size={80} />}
                  {weatherData.condition === 'Rainy' && <WiRain size={80} />}
                  {weatherData.condition === 'Cloudy' && <WiCloudy size={80} />}
                </div>
                <div className="weather-info">
                  <p>Temperature: {weatherData.temperature}°C</p>
                  <p><WiHumidity /> Humidity: {weatherData.humidity}%</p>
                  <p>Pressure: {weatherData.pressure} hPa</p>
                  <p>Wind: {weatherData.windSpeed} km/h</p>
                </div>
              </div>

              <div className="aqi-card">
                <h3>Air Quality Index</h3>
                <div className="aqi-value" style={{
                  backgroundColor: getAqiColor(aqiData[aqiData.length - 1].aqi)
                }}>
                  {aqiData[aqiData.length - 1].aqi}
                </div>
                <p>{getAqiDescription(aqiData[aqiData.length - 1].aqi)}</p>
                <div className="pollutants">
                  <p>PM2.5: {aqiData[aqiData.length - 1].pm25} µg/m³</p>
                  <p>PM10: {aqiData[aqiData.length - 1].pm10} µg/m³</p>
                  <p>CO: {aqiData[aqiData.length - 1].co} ppm</p>
                  <p>NO₂: {aqiData[aqiData.length - 1].no2} ppb</p>
                </div>
              </div>
            </section>

            <section className="historical-data">
              <div className="chart-container">
                <Line options={aqiChartOptions} data={aqiChartData} />
              </div>
            </section>
          </>
        )}
      </main>
    </div>
  )
}

function getAqiColor(aqi) {
  if (aqi <= 50) return '#009966' // Good
  if (aqi <= 100) return '#ffde33' // Moderate
  if (aqi <= 150) return '#ff9933' // Unhealthy for sensitive
  if (aqi <= 200) return '#cc0033' // Unhealthy
  if (aqi <= 300) return '#660099' // Very unhealthy
  return '#7e0023' // Hazardous
}

function getAqiDescription(aqi) {
  if (aqi <= 50) return 'Good'
  if (aqi <= 100) return 'Moderate'
  if (aqi <= 150) return 'Unhealthy for Sensitive Groups'
  if (aqi <= 200) return 'Unhealthy'
  if (aqi <= 300) return 'Very Unhealthy'
  return 'Hazardous'
}

export default App
