const express = require('express')
const cors = require('cors')
const sqlite3 = require('sqlite3').verbose()
const db = new sqlite3.Database('./poetry.db')

const app = express()
app.use(cors())
app.use(express.json())

// Initialize database
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS poems (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      title TEXT NOT NULL,
      author TEXT NOT NULL,
      content TEXT NOT NULL,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS comments (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      poemId INTEGER NOT NULL,
      author TEXT NOT NULL,
      content TEXT NOT NULL,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(poemId) REFERENCES poems(id)
    )
  `)
})

// API routes
app.get('/api/poems', (req, res) => {
  db.all('SELECT * FROM poems ORDER BY created_at DESC', [], (err, rows) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    res.json(rows)
  })
})

app.get('/api/poems/:id', (req, res) => {
  db.get('SELECT * FROM poems WHERE id = ?', [req.params.id], (err, row) => {
    if (err) {
      return res.status(500).json({ error: err.message })
    }
    if (!row) {
      return res.status(404).json({ error: 'Poem not found' })
    }
    res.json(row)
  })
})

app.post('/api/poems', (req, res) => {
  const { title, author, content } = req.body
  db.run(
    'INSERT INTO poems (title, author, content) VALUES (?, ?, ?)',
    [title, author, content],
    function(err) {
      if (err) {
        return res.status(500).json({ error: err.message })
      }
      res.json({ id: this.lastID, title, author, content })
    }
  )
})

app.get('/api/comments', (req, res) => {
  const poemId = req.query.poemId
  db.all(
    'SELECT * FROM comments WHERE poemId = ? ORDER BY created_at ASC',
    [poemId],
    (err, rows) => {
      if (err) {
        return res.status(500).json({ error: err.message })
      }
      res.json(rows)
    }
  )
})

app.post('/api/comments', (req, res) => {
  const { poemId, author, content } = req.body
  db.run(
    'INSERT INTO comments (poemId, author, content) VALUES (?, ?, ?)',
    [poemId, author, content],
    function(err) {
      if (err) {
        return res.status(500).json({ error: err.message })
      }
      res.json({ id: this.lastID, poemId, author, content })
    }
  )
})

const PORT = process.env.PORT || 3000
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
