import { useState, useEffect } from 'react'
import { useParams } from 'react-router-dom'
import { Box, Typography, Button, Radio, RadioGroup, FormControlLabel, FormControl, Card, CardContent } from '@mui/material'
import axios from 'axios'

export default function Contest() {
  const { id } = useParams()
  const [contest, setContest] = useState(null)
  const [currentQuestion, setCurrentQuestion] = useState(0)
  const [selectedAnswer, setSelectedAnswer] = useState('')
  const [score, setScore] = useState(0)

  useEffect(() => {
    const fetchContest = async () => {
      try {
        const res = await axios.get(`/api/contest/${id}`)
        setContest(res.data)
      } catch (err) {
        console.error(err)
      }
    }
    fetchContest()
  }, [id])

  const handleSubmit = async () => {
    try {
      const res = await axios.post('/api/submit-answer', {
        userId: 1, // TODO: Replace with actual user ID
        questionId: contest.questions[currentQuestion].id,
        answer: selectedAnswer
      })
      
      if (res.data.isCorrect) {
        setScore(score + contest.questions[currentQuestion].points)
      }
      
      if (currentQuestion < contest.questions.length - 1) {
        setCurrentQuestion(currentQuestion + 1)
        setSelectedAnswer('')
      }
    } catch (err) {
      console.error(err)
    }
  }

  if (!contest) return <Typography>Loading...</Typography>

  return (
    <Box sx={{ p: 4 }}>
      <Typography variant="h4" gutterBottom color="secondary">
        {contest.title}
      </Typography>
      <Typography variant="subtitle1" gutterBottom>
        Question {currentQuestion + 1} of {contest.questions.length}
      </Typography>
      
      <Card sx={{ bgcolor: 'primary.main', mb: 2 }}>
        <CardContent>
          <Typography variant="h6" color="white">
            {contest.questions[currentQuestion].question_text}
          </Typography>
          
          <FormControl component="fieldset">
            <RadioGroup
              value={selectedAnswer}
              onChange={(e) => setSelectedAnswer(e.target.value)}
            >
              {['option_a', 'option_b', 'option_c', 'option_d'].map((opt) => (
                contest.questions[currentQuestion][opt] && (
                  <FormControlLabel
                    key={opt}
                    value={opt.split('_')[1]}
                    control={<Radio color="secondary" />}
                    label={contest.questions[currentQuestion][opt]}
                    sx={{ color: 'white' }}
                  />
                )
              ))}
            </RadioGroup>
          </FormControl>
        </CardContent>
      </Card>
      
      <Button
        variant="contained"
        color="secondary"
        onClick={handleSubmit}
        disabled={!selectedAnswer}
      >
        {currentQuestion < contest.questions.length - 1 ? 'Next Question' : 'Finish'}
      </Button>
      
      <Typography variant="h6" sx={{ mt: 2 }}>
        Current Score: {score}
      </Typography>
    </Box>
  )
}
