import { useState, useEffect } from 'react'
import { Chess } from 'chess.js'
import { Chessboard } from 'react-chessboard'
import { v4 as uuidv4 } from 'uuid'
import './App.css'

function App() {
  const [game, setGame] = useState(new Chess())
  const [gameId, setGameId] = useState('')
  const [games, setGames] = useState([])
  const [boardWidth, setBoardWidth] = useState(600)
  const [boardTheme, setBoardTheme] = useState('default')
  const [difficulty, setDifficulty] = useState('medium')
  const [gameHistory, setGameHistory] = useState([])
  const [stats, setStats] = useState({
    wins: 0,
    losses: 0,
    draws: 0
  })

  function safeGameMutate(modify) {
    setGame((g) => {
      const update = { ...g }
      modify(update)
      return update
    })
  }

  function onDrop(sourceSquare, targetSquare) {
    try {
      const move = game.move({
        from: sourceSquare,
        to: targetSquare,
        promotion: 'q'
      })

      if (move === null) return false
      
      setGameHistory(prev => [...prev, move])
      return true
    } catch (e) {
      return false
    }
  }

  function createNewGame() {
    const newGameId = uuidv4()
    setGameId(newGameId)
    setGame(new Chess())
    setGameHistory([])
    setGames(prev => [...prev, newGameId])
  }

  function joinGame(id) {
    setGameId(id)
    setGame(new Chess())
    setGameHistory([])
  }

  function undoMove() {
    safeGameMutate((game) => {
      game.undo()
    })
    setGameHistory(prev => prev.slice(0, -1))
  }

  function resign() {
    alert('You resigned the game')
    setStats(prev => ({...prev, losses: prev.losses + 1}))
    createNewGame()
  }

  function makeRandomMove() {
    const possibleMoves = game.moves()
    if (game.isGameOver() || game.isDraw() || possibleMoves.length === 0) return
    
    const randomIndex = Math.floor(Math.random() * possibleMoves.length)
    const move = possibleMoves[randomIndex]
    safeGameMutate((game) => {
      game.move(move)
    })
    setGameHistory(prev => [...prev, move])
  }

  useEffect(() => {
    if (difficulty === 'easy') {
      setTimeout(makeRandomMove, 300)
    } else if (difficulty === 'medium') {
      setTimeout(makeRandomMove, 200)
    } else {
      setTimeout(makeRandomMove, 100)
    }
  }, [gameHistory.length])

  return (
    <div className="app">
      <div className="sidebar">
        <h2>Chess Game</h2>
        
        <div className="game-controls">
          <button onClick={createNewGame}>New Game</button>
          <button onClick={undoMove}>Undo Move</button>
          <button onClick={resign}>Resign</button>
        </div>

        <div className="game-settings">
          <h3>Board Theme</h3>
          <select value={boardTheme} onChange={(e) => setBoardTheme(e.target.value)}>
            <option value="default">Default</option>
            <option value="brown">Brown</option>
            <option value="green">Green</option>
            <option value="blue">Blue</option>
          </select>

          <h3>Difficulty</h3>
          <select value={difficulty} onChange={(e) => setDifficulty(e.target.value)}>
            <option value="easy">Easy</option>
            <option value="medium">Medium</option>
            <option value="hard">Hard</option>
          </select>
        </div>

        <div className="game-info">
          <h3>Game ID: {gameId}</h3>
          <h3>Status: {game.isGameOver() ? 'Game Over' : 'In Progress'}</h3>
          <h3>Turn: {game.turn() === 'w' ? 'White' : 'Black'}</h3>
        </div>

        <div className="stats">
          <h3>Statistics</h3>
          <p>Wins: {stats.wins}</p>
          <p>Losses: {stats.losses}</p>
          <p>Draws: {stats.draws}</p>
        </div>
      </div>

      <div className="board-container">
        <Chessboard 
          position={game.fen()} 
          onPieceDrop={onDrop}
          boardWidth={boardWidth}
          customBoardStyle={{
            borderRadius: '4px',
            boxShadow: '0 5px 15px rgba(0, 0, 0, 0.5)'
          }}
          customDarkSquareStyle={{ backgroundColor: boardTheme === 'default' ? '#779556' : 
            boardTheme === 'brown' ? '#b58863' : 
            boardTheme === 'green' ? '#86a666' : '#4682b4' }}
          customLightSquareStyle={{ backgroundColor: boardTheme === 'default' ? '#ebecd0' : 
            boardTheme === 'brown' ? '#f0d9b5' : 
            boardTheme === 'green' ? '#ffffdd' : '#b0c4de' }}
        />
      </div>
    </div>
  )
}

export default App
