import express from 'express'
import cors from 'cors'
import sqlite3 from 'sqlite3'
import { open } from 'sqlite'
import bcrypt from 'bcrypt'
import jwt from 'jsonwebtoken'

const app = express()
app.use(cors())
app.use(express.json())

const db = await open({
  filename: './database.db',
  driver: sqlite3.Database
})

await db.exec(`
  CREATE TABLE IF NOT EXISTS users (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    email TEXT UNIQUE NOT NULL,
    password TEXT NOT NULL,
    role TEXT DEFAULT 'user'
  );
  
  CREATE TABLE IF NOT EXISTS donations (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    userId INTEGER,
    name TEXT NOT NULL,
    quantity INTEGER NOT NULL,
    expiry TEXT NOT NULL,
    location TEXT NOT NULL,
    status TEXT DEFAULT 'available',
    FOREIGN KEY(userId) REFERENCES users(id)
  );
  
  CREATE TABLE IF NOT EXISTS distributions (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    donationId INTEGER,
    userId INTEGER,
    date TEXT NOT NULL,
    status TEXT DEFAULT 'pending',
    FOREIGN KEY(donationId) REFERENCES donations(id),
    FOREIGN KEY(userId) REFERENCES users(id)
  );
  
  CREATE TABLE IF NOT EXISTS volunteers (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    userId INTEGER,
    skills TEXT,
    availability TEXT,
    status TEXT DEFAULT 'pending',
    FOREIGN KEY(userId) REFERENCES users(id)
  );
`)

// Authentication middleware
const authenticate = async (req, res, next) => {
  const token = req.headers.authorization?.split(' ')[1]
  if (!token) return res.status(401).json({ error: 'Unauthorized' })

  try {
    const decoded = jwt.verify(token, 'secret_key')
    req.user = await db.get('SELECT * FROM users WHERE id = ?', [decoded.id])
    next()
  } catch (err) {
    res.status(401).json({ error: 'Invalid token' })
  }
}

// User routes
app.post('/api/register', async (req, res) => {
  const { name, email, password } = req.body
  const hashedPassword = await bcrypt.hash(password, 10)
  try {
    const result = await db.run(
      'INSERT INTO users (name, email, password) VALUES (?, ?, ?)',
      [name, email, hashedPassword]
    )
    res.status(201).json({ id: result.lastID })
  } catch (err) {
    res.status(400).json({ error: err.message })
  }
})

app.post('/api/login', async (req, res) => {
  const { email, password } = req.body
  const user = await db.get('SELECT * FROM users WHERE email = ?', [email])
  
  if (!user) {
    return res.status(401).json({ error: 'Invalid credentials' })
  }

  const isValid = await bcrypt.compare(password, user.password)
  if (!isValid) {
    return res.status(401).json({ error: 'Invalid credentials' })
  }

  const token = jwt.sign({ id: user.id }, 'secret_key', { expiresIn: '1h' })
  res.json({ token, user: { id: user.id, name: user.name, email: user.email, role: user.role } })
})

// Donation routes
app.get('/api/donations', async (req, res) => {
  const donations = await db.all('SELECT * FROM donations WHERE status = "available"')
  res.json(donations)
})

app.post('/api/donations', authenticate, async (req, res) => {
  try {
    const { name, quantity, expiry, location } = req.body
    const result = await db.run(
      'INSERT INTO donations (userId, name, quantity, expiry, location) VALUES (?, ?, ?, ?, ?)',
      [req.user.id, name, quantity, expiry, location]
    )
    res.status(201).json({ id: result.lastID })
  } catch (err) {
    res.status(400).json({ error: err.message })
  }
})

// Distribution routes
app.post('/api/distributions', authenticate, async (req, res) => {
  try {
    const { donationId, date } = req.body
    const result = await db.run(
      'INSERT INTO distributions (donationId, userId, date) VALUES (?, ?, ?)',
      [donationId, req.user.id, date]
    )
    // Update donation status
    await db.run('UPDATE donations SET status = "reserved" WHERE id = ?', [donationId])
    res.status(201).json({ id: result.lastID })
  } catch (err) {
    res.status(400).json({ error: err.message })
  }
})

// Volunteer routes
app.post('/api/volunteers', authenticate, async (req, res) => {
  try {
    const { skills, availability } = req.body
    const result = await db.run(
      'INSERT INTO volunteers (userId, skills, availability) VALUES (?, ?, ?)',
      [req.user.id, skills, availability]
    )
    res.status(201).json({ id: result.lastID })
  } catch (err) {
    res.status(400).json({ error: err.message })
  }
})

app.get('/api/volunteers', authenticate, async (req, res) => {
  if (req.user.role !== 'admin') {
    return res.status(403).json({ error: 'Forbidden' })
  }
  const volunteers = await db.all('SELECT * FROM volunteers')
  res.json(volunteers)
})

// Account routes
app.get('/api/account', authenticate, async (req, res) => {
  const userDonations = await db.all('SELECT * FROM donations WHERE userId = ?', [req.user.id])
  const userDistributions = await db.all(
    `SELECT d.* FROM distributions di
    JOIN donations d ON di.donationId = d.id
    WHERE di.userId = ?`, 
    [req.user.id]
  )
  res.json({
    user: req.user,
    donations: userDonations,
    distributions: userDistributions
  })
})

app.listen(3001, () => {
  console.log('Server running on http://localhost:3001')
})
