const express = require('express');
const sqlite3 = require('sqlite3').verbose();
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const multer = require('multer');
const cors = require('cors');
const path = require('path');
const { Server } = require('socket.io');
const http = require('http');

const app = express();
const server = http.createServer(app);
const io = new Server(server, {
  cors: {
    origin: "http://localhost:5173",
    methods: ["GET", "POST"]
  }
});

app.use(cors());
app.use(express.json());
app.use('/uploads', express.static('uploads'));

const db = new sqlite3.Database('./matrimony.db');

// Initialize database
db.serialize(() => {
  db.run(`CREATE TABLE IF NOT EXISTS users (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    email TEXT UNIQUE,
    password TEXT,
    name TEXT,
    gender TEXT,
    dob TEXT,
    religion TEXT,
    caste TEXT,
    education TEXT,
    profession TEXT,
    income TEXT,
    country TEXT,
    state TEXT,
    city TEXT,
    about TEXT,
    profile_pic TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
  )`);

  db.run(`CREATE TABLE IF NOT EXISTS messages (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    sender_id INTEGER,
    receiver_id INTEGER,
    message TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY(sender_id) REFERENCES users(id),
    FOREIGN KEY(receiver_id) REFERENCES users(id)
  )`);
});

// Configure file upload
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'uploads/');
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + path.extname(file.originalname));
  }
});
const upload = multer({ storage });

// Authentication middleware
const authenticate = (req, res, next) => {
  const token = req.headers.authorization?.split(' ')[1];
  if (!token) return res.status(401).json({ error: 'Unauthorized' });

  try {
    const decoded = jwt.verify(token, 'secret_key');
    req.user = decoded;
    next();
  } catch (err) {
    res.status(401).json({ error: 'Invalid token' });
  }
};

// Routes
app.post('/api/register', async (req, res) => {
  const { email, password, name } = req.body;
  if (!email || !password || !name) {
    return res.status(400).json({ error: 'All fields are required' });
  }

  const hashedPassword = await bcrypt.hash(password, 10);
  db.run(
    'INSERT INTO users (email, password, name) VALUES (?, ?, ?)',
    [email, hashedPassword, name],
    function(err) {
      if (err) {
        return res.status(400).json({ error: 'Email already exists' });
      }
      res.json({ id: this.lastID });
    }
  );
});

app.post('/api/login', (req, res) => {
  const { email, password } = req.body;
  db.get('SELECT * FROM users WHERE email = ?', [email], async (err, user) => {
    if (err || !user) {
      return res.status(400).json({ error: 'Invalid credentials' });
    }

    const validPassword = await bcrypt.compare(password, user.password);
    if (!validPassword) {
      return res.status(400).json({ error: 'Invalid credentials' });
    }

    const token = jwt.sign({ id: user.id, email: user.email }, 'secret_key');
    res.json({ token, user });
  });
});

app.post('/api/profile', authenticate, upload.single('profile_pic'), (req, res) => {
  const userId = req.user.id;
  const profileData = req.body;
  const profilePic = req.file ? `/uploads/${req.file.filename}` : null;

  const fields = [];
  const values = [];
  Object.entries(profileData).forEach(([key, value]) => {
    if (value) {
      fields.push(`${key} = ?`);
      values.push(value);
    }
  });

  if (profilePic) {
    fields.push('profile_pic = ?');
    values.push(profilePic);
  }

  if (fields.length === 0) {
    return res.status(400).json({ error: 'No data to update' });
  }

  const query = `UPDATE users SET ${fields.join(', ')} WHERE id = ?`;
  db.run(query, [...values, userId], function(err) {
    if (err) {
      return res.status(500).json({ error: 'Failed to update profile' });
    }
    res.json({ success: true });
  });
});

app.get('/api/users', authenticate, (req, res) => {
  const { gender, religion, caste, education, profession, country, state, city } = req.query;
  let query = 'SELECT id, name, gender, dob, religion, caste, education, profession, country, state, city, profile_pic FROM users WHERE id != ?';
  const params = [req.user.id];

  if (gender) {
    query += ' AND gender = ?';
    params.push(gender);
  }
  if (religion) {
    query += ' AND religion = ?';
    params.push(religion);
  }
  if (caste) {
    query += ' AND caste = ?';
    params.push(caste);
  }
  if (education) {
    query += ' AND education = ?';
    params.push(education);
  }
  if (profession) {
    query += ' AND profession = ?';
    params.push(profession);
  }
  if (country) {
    query += ' AND country = ?';
    params.push(country);
  }
  if (state) {
    query += ' AND state = ?';
    params.push(state);
  }
  if (city) {
    query += ' AND city = ?';
    params.push(city);
  }

  db.all(query, params, (err, users) => {
    if (err) {
      return res.status(500).json({ error: 'Failed to fetch users' });
    }
    res.json(users);
  });
});

app.get('/api/messages/:userId', authenticate, (req, res) => {
  const { userId } = req.params;
  db.all(
    'SELECT * FROM messages WHERE (sender_id = ? AND receiver_id = ?) OR (sender_id = ? AND receiver_id = ?) ORDER BY created_at',
    [req.user.id, userId, userId, req.user.id],
    (err, messages) => {
      if (err) {
        return res.status(500).json({ error: 'Failed to fetch messages' });
      }
      res.json(messages);
    }
  );
});

// Socket.io for real-time chat
io.on('connection', (socket) => {
  socket.on('send_message', (data) => {
    db.run(
      'INSERT INTO messages (sender_id, receiver_id, message) VALUES (?, ?, ?)',
      [data.senderId, data.receiverId, data.message],
      function(err) {
        if (err) return;
        io.emit('receive_message', {
          id: this.lastID,
          sender_id: data.senderId,
          receiver_id: data.receiverId,
          message: data.message,
          created_at: new Date().toISOString()
        });
      }
    );
  });
});

const PORT = 3001;
server.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});
