import express from 'express'
import sqlite3 from 'sqlite3'
import { open } from 'sqlite'
import cors from 'cors'

const app = express()
const port = 3001

app.use(cors())
app.use(express.json())

// Database setup
let db
async function initializeDatabase() {
  db = await open({
    filename: './skillshare.db',
    driver: sqlite3.Database
  })

  await db.exec(`
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      bio TEXT,
      email TEXT UNIQUE NOT NULL,
      createdAt DATETIME DEFAULT CURRENT_TIMESTAMP
    );
    
    CREATE TABLE IF NOT EXISTS skills (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      skill TEXT NOT NULL,
      FOREIGN KEY(user_id) REFERENCES users(id) ON DELETE CASCADE
    );
    
    CREATE TABLE IF NOT EXISTS certifications (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      certification TEXT NOT NULL,
      FOREIGN KEY(user_id) REFERENCES users(id) ON DELETE CASCADE
    );
  `)
}

// Error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack)
  res.status(500).json({ error: 'Something went wrong!' })
})

// API endpoints
app.get('/api/profiles', async (req, res, next) => {
  try {
    const searchTerm = req.query.search || ''
    const profiles = await db.all(`
      SELECT u.*, 
        GROUP_CONCAT(DISTINCT s.skill) AS skills,
        GROUP_CONCAT(DISTINCT c.certification) AS certifications
      FROM users u
      LEFT JOIN skills s ON u.id = s.user_id
      LEFT JOIN certifications c ON u.id = c.user_id
      WHERE s.skill LIKE ? OR c.certification LIKE ? OR u.name LIKE ?
      GROUP BY u.id
      ORDER BY u.createdAt DESC
    `, [`%${searchTerm}%`, `%${searchTerm}%`, `%${searchTerm}%`])
    
    const formattedProfiles = profiles.map(profile => ({
      ...profile,
      skills: profile.skills ? profile.skills.split(',') : [],
      certifications: profile.certifications ? profile.certifications.split(',') : []
    }))
    
    res.json(formattedProfiles)
  } catch (err) {
    next(err)
  }
})

app.post('/api/profiles', async (req, res, next) => {
  try {
    const { name, bio, email, skills = [], certifications = [] } = req.body
    
    // Basic validation
    if (!name || !email) {
      return res.status(400).json({ error: 'Name and email are required' })
    }

    const result = await db.run(
      'INSERT INTO users (name, bio, email) VALUES (?, ?, ?)',
      [name, bio, email]
    )
    
    const userId = result.lastID
    
    // Insert skills
    for (const skill of skills) {
      await db.run(
        'INSERT INTO skills (user_id, skill) VALUES (?, ?)',
        [userId, skill]
      )
    }
    
    // Insert certifications
    for (const cert of certifications) {
      await db.run(
        'INSERT INTO certifications (user_id, certification) VALUES (?, ?)',
        [userId, cert]
      )
    }
    
    res.status(201).json({ id: userId, message: 'Profile created successfully' })
  } catch (err) {
    next(err)
  }
})

// Initialize and start server
initializeDatabase().then(() => {
  app.listen(port, () => {
    console.log(`Server running on http://localhost:${port}`)
  })
}).catch(err => {
  console.error('Failed to initialize database:', err)
  process.exit(1)
})
