const express = require('express')
const cors = require('cors')
const { createClient } = require('@libsql/client')
const twilio = require('twilio')

const app = express()
app.use(cors())
app.use(express.json())

const db = createClient({
  url: 'file:animal-spotter.db'
})

// Initialize database
async function initDb() {
  await db.execute(`
    CREATE TABLE IF NOT EXISTS detections (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      animal_type TEXT NOT NULL,
      timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
      confidence REAL,
      image_url TEXT
    )
  `)

  await db.execute(`
    CREATE TABLE IF NOT EXISTS alerts (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      phone_number TEXT NOT NULL,
      animal_type TEXT NOT NULL,
      min_confidence REAL DEFAULT 0.7
    )
  `)
}

// Animal color coding
const animalColors = {
  bear: '#8B4513',
  wolf: '#808080',
  deer: '#CD853F',
  fox: '#FF4500',
  rabbit: '#FFFFFF',
  default: '#000000'
}

// Twilio setup
const twilioClient = twilio(
  process.env.TWILIO_ACCOUNT_SID,
  process.env.TWILIO_AUTH_TOKEN
)

// API endpoints
app.post('/api/detections', async (req, res) => {
  const { animal_type, confidence, image_url } = req.body
  const result = await db.execute({
    sql: 'INSERT INTO detections (animal_type, confidence, image_url) VALUES (?, ?, ?)',
    args: [animal_type, confidence, image_url]
  })
  
  // Check for matching alerts
  const { rows: alerts } = await db.execute({
    sql: 'SELECT * FROM alerts WHERE animal_type = ? AND min_confidence <= ?',
    args: [animal_type, confidence]
  })
  
  // Send SMS alerts
  for (const alert of alerts) {
    await twilioClient.messages.create({
      body: `Alert: ${animal_type} detected with ${confidence*100}% confidence!`,
      from: process.env.TWILIO_PHONE_NUMBER,
      to: alert.phone_number
    })
  }
  
  res.status(201).json({ id: result.lastInsertRowid.toString() })
})

app.get('/api/detections', async (req, res) => {
  const { rows } = await db.execute('SELECT * FROM detections ORDER BY timestamp DESC')
  res.json(rows.map(row => ({
    ...row,
    color: animalColors[row.animal_type] || animalColors.default
  })))
})

app.post('/api/alerts', async (req, res) => {
  const { phone_number, animal_type, min_confidence } = req.body
  const result = await db.execute({
    sql: 'INSERT INTO alerts (phone_number, animal_type, min_confidence) VALUES (?, ?, ?)',
    args: [phone_number, animal_type, min_confidence || 0.7]
  })
  res.status(201).json({ id: result.lastInsertRowid.toString() })
})

app.get('/api/alerts', async (req, res) => {
  const { rows } = await db.execute('SELECT * FROM alerts')
  res.json(rows)
})

const PORT = process.env.PORT || 5000
initDb().then(() => {
  app.listen(PORT, () => console.log(`Server running on port ${PORT}`))
})
