class Game2048 {
    private board: number[][];
    private score: number;
    private gameOver: boolean;
    private boardSize: number;
    private boardElement: HTMLElement;
    private scoreElement: HTMLElement;
    private gameOverElement: HTMLElement;

    constructor() {
        this.boardSize = 4;
        this.score = 0;
        this.gameOver = false;
        this.board = Array(this.boardSize).fill(0).map(() => Array(this.boardSize).fill(0));
        this.boardElement = document.querySelector('.game-board')!;
        this.scoreElement = document.querySelector('.score-value')!;
        this.gameOverElement = document.querySelector('.game-over')!;

        this.initGame();
        this.setupEventListeners();
    }

    private initGame(): void {
        this.board = Array(this.boardSize).fill(0).map(() => Array(this.boardSize).fill(0));
        this.score = 0;
        this.gameOver = false;
        this.gameOverElement.style.display = 'none';
        this.addRandomTile();
        this.addRandomTile();
        this.updateView();
    }

    private setupEventListeners(): void {
        document.addEventListener('keydown', (e) => this.handleKeyPress(e));
        document.querySelector('.new-game-btn')!.addEventListener('click', () => this.initGame());
        document.querySelector('.revive-btn')!.addEventListener('click', () => this.reviveGame());
    }

    private handleKeyPress(e: KeyboardEvent): void {
        if (this.gameOver) return;

        let moved = false;
        switch (e.key) {
            case 'ArrowUp': moved = this.moveUp(); break;
            case 'ArrowDown': moved = this.moveDown(); break;
            case 'ArrowLeft': moved = this.moveLeft(); break;
            case 'ArrowRight': moved = this.moveRight(); break;
            default: return;
        }

        if (moved) {
            this.addRandomTile();
            this.updateView();
            if (this.isGameOver()) {
                this.gameOver = true;
                this.gameOverElement.style.display = 'flex';
            }
        }
    }

    private moveUp(): boolean {
        let moved = false;
        for (let col = 0; col < this.boardSize; col++) {
            for (let row = 1; row < this.boardSize; row++) {
                if (this.board[row][col] !== 0) {
                    let currentRow = row;
                    while (currentRow > 0 && this.board[currentRow - 1][col] === 0) {
                        this.board[currentRow - 1][col] = this.board[currentRow][col];
                        this.board[currentRow][col] = 0;
                        currentRow--;
                        moved = true;
                    }
                    if (currentRow > 0 && this.board[currentRow - 1][col] === this.board[currentRow][col]) {
                        this.board[currentRow - 1][col] *= 2;
                        this.score += this.board[currentRow - 1][col];
                        this.board[currentRow][col] = 0;
                        moved = true;
                    }
                }
            }
        }
        return moved;
    }

    private moveDown(): boolean {
        let moved = false;
        for (let col = 0; col < this.boardSize; col++) {
            for (let row = this.boardSize - 2; row >= 0; row--) {
                if (this.board[row][col] !== 0) {
                    let currentRow = row;
                    while (currentRow < this.boardSize - 1 && this.board[currentRow + 1][col] === 0) {
                        this.board[currentRow + 1][col] = this.board[currentRow][col];
                        this.board[currentRow][col] = 0;
                        currentRow++;
                        moved = true;
                    }
                    if (currentRow < this.boardSize - 1 && this.board[currentRow + 1][col] === this.board[currentRow][col]) {
                        this.board[currentRow + 1][col] *= 2;
                        this.score += this.board[currentRow + 1][col];
                        this.board[currentRow][col] = 0;
                        moved = true;
                    }
                }
            }
        }
        return moved;
    }

    private moveLeft(): boolean {
        let moved = false;
        for (let row = 0; row < this.boardSize; row++) {
            for (let col = 1; col < this.boardSize; col++) {
                if (this.board[row][col] !== 0) {
                    let currentCol = col;
                    while (currentCol > 0 && this.board[row][currentCol - 1] === 0) {
                        this.board[row][currentCol - 1] = this.board[row][currentCol];
                        this.board[row][currentCol] = 0;
                        currentCol--;
                        moved = true;
                    }
                    if (currentCol > 0 && this.board[row][currentCol - 1] === this.board[row][currentCol]) {
                        this.board[row][currentCol - 1] *= 2;
                        this.score += this.board[row][currentCol - 1];
                        this.board[row][currentCol] = 0;
                        moved = true;
                    }
                }
            }
        }
        return moved;
    }

    private moveRight(): boolean {
        let moved = false;
        for (let row = 0; row < this.boardSize; row++) {
            for (let col = this.boardSize - 2; col >= 0; col--) {
                if (this.board[row][col] !== 0) {
                    let currentCol = col;
                    while (currentCol < this.boardSize - 1 && this.board[row][currentCol + 1] === 0) {
                        this.board[row][currentCol + 1] = this.board[row][currentCol];
                        this.board[row][currentCol] = 0;
                        currentCol++;
                        moved = true;
                    }
                    if (currentCol < this.boardSize - 1 && this.board[row][currentCol + 1] === this.board[row][currentCol]) {
                        this.board[row][currentCol + 1] *= 2;
                        this.score += this.board[row][currentCol + 1];
                        this.board[row][currentCol] = 0;
                        moved = true;
                    }
                }
            }
        }
        return moved;
    }

    private addRandomTile(): void {
        const emptyCells: [number, number][] = [];
        for (let row = 0; row < this.boardSize; row++) {
            for (let col = 0; col < this.boardSize; col++) {
                if (this.board[row][col] === 0) {
                    emptyCells.push([row, col]);
                }
            }
        }

        if (emptyCells.length > 0) {
            const [row, col] = emptyCells[Math.floor(Math.random() * emptyCells.length)];
            this.board[row][col] = Math.random() < 0.9 ? 2 : 4;
        }
    }

    private updateView(): void {
        this.boardElement.innerHTML = '';
        this.scoreElement.textContent = this.score.toString();

        for (let row = 0; row < this.boardSize; row++) {
            for (let col = 0; col < this.boardSize; col++) {
                const tile = document.createElement('div');
                tile.className = 'tile';
                tile.textContent = this.board[row][col] !== 0 ? this.board[row][col].toString() : '';
                tile.style.backgroundColor = this.getTileColor(this.board[row][col]);
                tile.style.color = this.board[row][col] > 4 ? 'white' : 'maroon';
                this.boardElement.appendChild(tile);
            }
        }
    }

    private getTileColor(value: number): string {
        const colors: Record<number, string> = {
            0: 'rgba(128, 0, 0, 0.1)',
            2: 'rgba(255, 200, 200, 0.8)',
            4: 'rgba(255, 180, 180, 0.8)',
            8: 'rgba(255, 150, 150, 0.8)',
            16: 'rgba(255, 120, 120, 0.8)',
            32: 'rgba(255, 90, 90, 0.8)',
            64: 'rgba(255, 60, 60, 0.8)',
            128: 'rgba(255, 30, 30, 0.8)',
            256: 'rgba(255, 0, 0, 0.8)',
            512: 'rgba(220, 0, 0, 0.8)',
            1024: 'rgba(180, 0, 0, 0.8)',
            2048: 'rgba(150, 0, 0, 0.8)'
        };
        return colors[value] || 'maroon';
    }

    private isGameOver(): boolean {
        // Check for empty cells
        for (let row = 0; row < this.boardSize; row++) {
            for (let col = 0; col < this.boardSize; col++) {
                if (this.board[row][col] === 0) {
                    return false;
                }
            }
        }

        // Check for possible merges
        for (let row = 0; row < this.boardSize; row++) {
            for (let col = 0; col < this.boardSize; col++) {
                if (col < this.boardSize - 1 && this.board[row][col] === this.board[row][col + 1]) {
                    return false;
                }
                if (row < this.boardSize - 1 && this.board[row][col] === this.board[row + 1][col]) {
                    return false;
                }
            }
        }

        return true;
    }

    private reviveGame(): void {
        this.gameOver = false;
        this.gameOverElement.style.display = 'none';
    }
}

new Game2048();