import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';

interface User {
  username: string;
  email: string;
  joinDate: string;
  posts: Post[];
  replies: Reply[];
}

interface Post {
  id: number;
  title: string;
  category: string;
  createdAt: string;
}

interface Reply {
  id: number;
  postId: number;
  postTitle: string;
  content: string;
  createdAt: string;
}

const UserCenter: React.FC = () => {
  const [user, setUser] = useState<User | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    const fetchUserData = async () => {
      try {
        // In a real app, this would fetch the logged-in user's data
        const response = await axios.get('/api/user/current');
        setUser(response.data);
        setIsLoading(false);
      } catch (error) {
        console.error('Error fetching user data:', error);
        setIsLoading(false);
      }
    };

    fetchUserData();
  }, []);

  if (isLoading) return <div>Loading...</div>;
  if (!user) return <div>User data not available</div>;

  return (
    <div className="user-panel">
      <h2>User Center</h2>
      
      <div className="user-info">
        <h3>Profile Information</h3>
        <p><strong>Username:</strong> {user.username}</p>
        <p><strong>Email:</strong> {user.email}</p>
        <p><strong>Member since:</strong> {new Date(user.joinDate).toLocaleDateString()}</p>
      </div>

      <div className="user-posts">
        <h3>Your Posts ({user.posts.length})</h3>
        {user.posts.map(post => (
          <div key={post.id} className="post-card">
            <Link to={`/post/${post.id}`}>
              <h4>{post.title}</h4>
            </Link>
            <span className="category">{post.category}</span>
            <small>Posted on {new Date(post.createdAt).toLocaleString()}</small>
          </div>
        ))}
      </div>

      <div className="user-replies">
        <h3>Your Replies ({user.replies.length})</h3>
        {user.replies.map(reply => (
          <div key={reply.id} className="reply-box">
            <Link to={`/post/${reply.postId}`}>
              <h4>Re: {reply.postTitle}</h4>
            </Link>
            <p>{reply.content}</p>
            <small>Posted on {new Date(reply.createdAt).toLocaleString()}</small>
          </div>
        ))}
      </div>
    </div>
  );
};

export default UserCenter;