// Domain class
class Domain {
    constructor(
        public name: string,
        public price: number,
        public available: boolean,
        public expiryDate?: Date
    ) {}
}

// User class
class User {
    constructor(
        public username: string,
        public email: string,
        public domains: Domain[] = []
    ) {}
}

// Main application class
class DomainLeasingApp {
    private currentUser: User | null = null;
    private availableDomains: Domain[] = [
        new Domain('example.com', 10.99, true),
        new Domain('test.org', 8.99, true),
        new Domain('mysite.net', 12.99, false)
    ];

    constructor() {
        this.initEventListeners();
    }

    private initEventListeners(): void {
        const searchForm = document.querySelector('form');
        if (searchForm) {
            searchForm.addEventListener('submit', (e) => {
                e.preventDefault();
                const input = searchForm.querySelector('input');
                if (input) this.searchDomains(input.value);
            });
        }
    }

    private searchDomains(query: string): void {
        const results = this.availableDomains.filter(domain => 
            domain.name.includes(query) && domain.available
        );
        this.displayResults(results);
    }

    private displayResults(domains: Domain[]): void {
        const searchSection = document.querySelector('.search-section');
        if (!searchSection) return;

        // Clear previous results
        const oldResults = searchSection.querySelector('.results');
        if (oldResults) oldResults.remove();

        // Create results container
        const resultsDiv = document.createElement('div');
        resultsDiv.className = 'results';

        if (domains.length === 0) {
            resultsDiv.innerHTML = '<p>No domains found matching your search.</p>';
        } else {
            resultsDiv.innerHTML = '<h3>Available Domains:</h3><ul></ul>';
            const ul = resultsDiv.querySelector('ul')!;
            domains.forEach(domain => {
                const li = document.createElement('li');
                li.innerHTML = `
                    <span>${domain.name} - $${domain.price}/year</span>
                    <button class="lease-btn" data-domain="${domain.name}">Lease</button>
                `;
                ul.appendChild(li);
            });
        }

        searchSection.appendChild(resultsDiv);
    }
}

// Initialize the app when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new DomainLeasingApp();
});