// Environmental Warning System - Client-side JavaScript
document.addEventListener('DOMContentLoaded', () => {
    // DOM elements
    const alertForm = document.getElementById('alert-form');
    const alertAudio = document.getElementById('alert-audio');
    
    // Real-time data update function
    async function updateData() {
        try {
            const response = await fetch('/api/weather');
            const data = await response.json();
            
            // Update weather display
            document.getElementById('temperature').textContent = `${data.temperature}°C`;
            document.getElementById('humidity').textContent = `${data.humidity}%`;
            document.getElementById('aqi-value').textContent = data.aqi;
            document.getElementById('aqi-category').textContent = getAQICategory(data.aqi);
            document.getElementById('aqi-category').style.color = getAQIColor(data.aqi);
            
            // Update AQI chart
            updateAQIChart(data.aqiHistory);
            
        } catch (error) {
            console.error('Error fetching data:', error);
        }
    }

    // AQI category helper
    function getAQICategory(aqi) {
        if (aqi <= 50) return 'Good';
        if (aqi <= 100) return 'Moderate';
        if (aqi <= 150) return 'Unhealthy for Sensitive Groups';
        if (aqi <= 200) return 'Unhealthy';
        if (aqi <= 300) return 'Very Unhealthy';
        return 'Hazardous';
    }

    // AQI color helper
    function getAQIColor(aqi) {
        if (aqi <= 50) return '#4CAF50';
        if (aqi <= 100) return '#FFEB3B';
        if (aqi <= 150) return '#FF9800';
        if (aqi <= 200) return '#F44336';
        if (aqi <= 300) return '#9C27B0';
        return '#795548';
    }

    // AQI chart rendering
    function updateAQIChart(aqiHistory) {
        const chart = document.getElementById('aqi-chart');
        chart.innerHTML = aqiHistory.map((aqi, i) => `
            <div class="chart-bar" style="height: ${Math.min(aqi/3, 100)}%; background: ${getAQIColor(aqi)}">
                <span>${14 - i}d</span>
                <span>${aqi}</span>
            </div>
        `).join('');
    }

    // Alert system
    alertForm?.addEventListener('submit', async (e) => {
        e.preventDefault();
        const alertCode = document.getElementById('alert-code').value;
        const response = await fetch('/api/alerts', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ code: alertCode })
        });
        
        if (response.ok) {
            alertAudio.src = `/alerts/${alertCode}.mp3`;
            alertAudio.play();
        }
    });

    // Initial load and periodic updates
    updateData();
    setInterval(updateData, 30000);
});