import { Router, Request, Response } from 'express';
import {
    createArticle,
    getArticles,
    getArticleById,
    searchArticles,
    getArticlesByCategory,
    Article
} from '../models/article';

const router = Router();

// Get all articles
router.get('/', (req: Request, res: Response) => {
    const articles = getArticles();
    res.render('articles/index', { articles });
});

// Get article by ID
router.get('/:id', (req: Request<{id: string}>, res: Response) => {
    const article = getArticleById(req.params.id);
    if (!article) {
        return res.status(404).render('404');
    }
    res.render('articles/show', { article });
});

// Search articles
router.get('/search', (req: Request<{},{},{},{q: string}>, res: Response) => {
    const query = req.query.q;
    if (!query) {
        return res.redirect('/articles');
    }
    const results = searchArticles(query);
    res.render('articles/search', { results, query });
});

// Create new article
router.post('/', (req: Request<{},{},Article>, res: Response) => {
    try {
        const newArticle = createArticle(req.body);
        res.redirect(`/articles/${newArticle.id}`);
    } catch (error) {
        res.status(400).render('500');
    }
});

// Get articles by category
router.get('/category/:category', (req: Request<{category: string}>, res: Response) => {
    const articles = getArticlesByCategory(req.params.category);
    res.render('articles/category', { 
        articles,
        category: req.params.category 
    });
});

export default router;