import { useState } from 'react'
import './App.css'

interface Link {
  id: string
  title: string
  url: string
}

export default function App() {
  const [links, setLinks] = useState<Link[]>([
    { id: '1', title: 'Example Link', url: 'https://example.com' }
  ])
  const [newLink, setNewLink] = useState<Omit<Link, 'id'>>({ title: '', url: '' })
  const [editingId, setEditingId] = useState<string | null>(null)

  const addLink = () => {
    if (newLink.title && newLink.url) {
      setLinks([...links, { ...newLink, id: Date.now().toString() }])
      setNewLink({ title: '', url: '' })
    }
  }

  const updateLink = (id: string) => {
    setLinks(links.map(link => 
      link.id === id ? { ...link, ...newLink } : link
    ))
    setEditingId(null)
    setNewLink({ title: '', url: '' })
  }

  const deleteLink = (id: string) => {
    setLinks(links.filter(link => link.id !== id))
  }

  return (
    <div className="app">
      <h1>My LinkTree</h1>
      
      <div className="link-form">
        <input
          type="text"
          placeholder="Link Title"
          value={newLink.title}
          onChange={(e) => setNewLink({...newLink, title: e.target.value})}
        />
        <input
          type="url"
          placeholder="https://example.com"
          value={newLink.url}
          onChange={(e) => setNewLink({...newLink, url: e.target.value})}
        />
        {editingId ? (
          <button onClick={() => updateLink(editingId)}>Update Link</button>
        ) : (
          <button onClick={addLink}>Add Link</button>
        )}
      </div>

      <div className="links-container">
        {links.map(link => (
          <div key={link.id} className="link-item">
            <a href={link.url} className="link">
              {link.title}
            </a>
            <div className="link-actions">
              <button onClick={() => {
                setNewLink({ title: link.title, url: link.url })
                setEditingId(link.id)
              }}>Edit</button>
              <button onClick={() => deleteLink(link.id)}>Delete</button>
            </div>
          </div>
        ))}
      </div>
    </div>
  )
}