interface Task {
  id: number;
  text: string;
  completed: boolean;
  createdAt: Date;
}

class TodoApp {
  private tasks: Task[] = [];
  private taskIdCounter = 1;

  constructor() {
    this.initElements();
    this.loadTasks();
    this.renderTasks();
  }

  private initElements() {
    this.newTaskInput = document.getElementById('new-task-input') as HTMLInputElement;
    this.addTaskBtn = document.getElementById('add-task-btn') as HTMLButtonElement;
    this.searchInput = document.getElementById('search-input') as HTMLInputElement;
    this.filterSelect = document.getElementById('filter-select') as HTMLSelectElement;
    this.taskList = document.getElementById('task-list') as HTMLUListElement;

    this.addTaskBtn.addEventListener('click', () => this.addTask());
    this.newTaskInput.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') this.addTask();
    });
    this.searchInput.addEventListener('input', () => this.renderTasks());
    this.filterSelect.addEventListener('change', () => this.renderTasks());
  }

  private addTask() {
    const text = this.newTaskInput.value.trim();
    if (text) {
      const newTask: Task = {
        id: this.taskIdCounter++,
        text,
        completed: false,
        createdAt: new Date()
      };
      this.tasks.push(newTask);
      this.saveTasks();
      this.renderTasks();
      this.newTaskInput.value = '';
    }
  }

  private renderTasks() {
    this.taskList.innerHTML = '';
    
    const searchTerm = this.searchInput.value.toLowerCase();
    const filterValue = this.filterSelect.value;
    
    const filteredTasks = this.tasks.filter(task => {
      const matchesSearch = task.text.toLowerCase().includes(searchTerm);
      const matchesFilter = filterValue === 'all' || 
                          (filterValue === 'active' && !task.completed) || 
                          (filterValue === 'completed' && task.completed);
      return matchesSearch && matchesFilter;
    });

    filteredTasks.forEach(task => {
      const taskItem = document.createElement('li');
      taskItem.className = `task-item ${task.completed ? 'completed' : ''}`;
      
      const taskText = document.createElement('span');
      taskText.textContent = task.text;
      
      const taskActions = document.createElement('div');
      taskActions.className = 'task-actions';
      
      const completeBtn = document.createElement('button');
      completeBtn.textContent = task.completed ? 'Undo' : 'Complete';
      completeBtn.addEventListener('click', () => this.toggleTaskComplete(task.id));
      
      const deleteBtn = document.createElement('button');
      deleteBtn.textContent = 'Delete';
      deleteBtn.addEventListener('click', () => this.deleteTask(task.id));
      
      taskActions.append(completeBtn, deleteBtn);
      taskItem.append(taskText, taskActions);
      this.taskList.appendChild(taskItem);
    });
  }

  private toggleTaskComplete(id: number) {
    const task = this.tasks.find(t => t.id === id);
    if (task) {
      task.completed = !task.completed;
      this.saveTasks();
      this.renderTasks();
    }
  }

  private deleteTask(id: number) {
    this.tasks = this.tasks.filter(t => t.id !== id);
    this.saveTasks();
    this.renderTasks();
  }

  private saveTasks() {
    localStorage.setItem('tasks', JSON.stringify(this.tasks));
    localStorage.setItem('taskIdCounter', this.taskIdCounter.toString());
  }

  private loadTasks() {
    const savedTasks = localStorage.getItem('tasks');
    const savedIdCounter = localStorage.getItem('taskIdCounter');
    
    if (savedTasks) this.tasks = JSON.parse(savedTasks);
    if (savedIdCounter) this.taskIdCounter = parseInt(savedIdCounter);
  }
}

// Initialize the app when DOM is loaded
document.addEventListener('DOMContentLoaded', () => new TodoApp());