// Wheel configuration
const prizes = [
    "Free Spin",
    "$10 Gift Card", 
    "50% Off Coupon",
    "Grand Prize",
    "Better Luck Next Time",
    "$5 Gift Card",
    "25% Off Coupon"
];

// Wheel colors
const colors = [
    "#FF5733", "#33FF57", "#3357FF", 
    "#F3FF33", "#FF33F3", "#33FFF3", "#8B0000"
];

// Initialize wheel
const canvas = document.getElementById('wheel') as HTMLCanvasElement;
const ctx = canvas.getContext('2d')!;
const spinBtn = document.getElementById('spin-btn') as HTMLButtonElement;
const prizeList = document.getElementById('prize-list') as HTMLUListElement;
const winRecords = document.getElementById('win-records') as HTMLUListElement;

// Display prize list
prizes.forEach((prize, index) => {
    const li = document.createElement('li');
    li.textContent = `${index + 1}. ${prize}`;
    prizeList.appendChild(li);
});

// Draw wheel function
function drawWheel() {
    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2;
    const radius = Math.min(canvas.width, canvas.height) / 2 - 10;
    
    const arc = Math.PI * 2 / prizes.length;
    
    prizes.forEach((_, i) => {
        ctx.beginPath();
        ctx.fillStyle = colors[i % colors.length];
        ctx.moveTo(centerX, centerY);
        ctx.arc(centerX, centerY, radius, i * arc, (i + 1) * arc);
        ctx.lineTo(centerX, centerY);
        ctx.fill();
        
        // Draw text
        ctx.save();
        ctx.translate(centerX, centerY);
        ctx.rotate(i * arc + arc / 2);
        ctx.textAlign = 'right';
        ctx.fillStyle = '#fff';
        ctx.font = 'bold 14px Arial';
        ctx.fillText(prizes[i], radius - 10, 5);
        ctx.restore();
    });
}

// Initialize canvas
function init() {
    canvas.width = 500;
    canvas.height = 500;
    drawWheel();
}

// Spin wheel function
function spinWheel() {
    spinBtn.disabled = true;
    const spinDuration = 3000; // 3 seconds
    const startTime = Date.now();
    const spins = 5 + Math.random() * 5; // 5-10 spins
    
    function animate() {
        const elapsed = Date.now() - startTime;
        const progress = Math.min(elapsed / spinDuration, 1);
        const easeProgress = 1 - Math.pow(1 - progress, 3); // Easing function
        
        if (progress < 1) {
            const rotation = easeProgress * spins * Math.PI * 2;
            canvas.style.transform = `rotate(${rotation}rad)`;
            requestAnimationFrame(animate);
        } else {
            const winningIndex = Math.floor(Math.random() * prizes.length);
            announceWinner(winningIndex);
            spinBtn.disabled = false;
            canvas.style.transform = '';
        }
    }
    
    requestAnimationFrame(animate);
}

// Announce winner
function announceWinner(index: number) {
    const prize = prizes[index];
    const li = document.createElement('li');
    li.textContent = `${new Date().toLocaleString()}: ${prize}`;
    winRecords.prepend(li);
    
    alert(`Congratulations! You won: ${prize}`);
}

// Event listeners
spinBtn.addEventListener('click', spinWheel);
window.addEventListener('load', init);