import { useState } from 'react'
import axios from 'axios'
import StockSearch from '../components/StockSearch'
import ReportOptions from '../components/ReportOptions'
import StockChart from '../components/StockChart'
import './ReportGenerator.css'

export default function ReportGenerator() {
  const [stockData, setStockData] = useState(null)
  const [loading, setLoading] = useState(false)
  const [reportOptions, setReportOptions] = useState({
    includeTrends: true,
    includeFinancials: true,
    includeRecommendations: true,
    timeRange: '1y'
  })

  const handleSearch = async (symbol) => {
    setLoading(true)
    try {
      // In a real app, you would call your backend API here
      // This is a mock implementation
      const mockData = {
        symbol,
        name: 'Example Company',
        price: 150.42,
        change: 2.34,
        changePercent: 1.58,
        chartData: generateMockChartData(),
        financials: {
          revenue: 12500000000,
          profit: 2500000000,
          eps: 5.42
        }
      }
      setStockData(mockData)
    } catch (error) {
      console.error('Error fetching stock data:', error)
    } finally {
      setLoading(false)
    }
  }

  const generateMockChartData = () => {
    // Generate mock chart data
    const labels = []
    const data = []
    for (let i = 0; i < 30; i++) {
      labels.push(`Day ${i + 1}`)
      data.push(Math.random() * 50 + 100)
    }
    return { labels, data }
  }

  const handleGenerateReport = () => {
    // In a real app, this would generate a PDF or detailed report
    alert('Report generation would be implemented here')
  }

  return (
    <div className="report-generator">
      <h2>Generate Stock Report</h2>
      
      <StockSearch onSearch={handleSearch} />
      
      {loading && <div className="loading">Loading...</div>}
      
      {stockData && (
        <>
          <div className="stock-info">
            <h3>{stockData.name} ({stockData.symbol})</h3>
            <p>Price: ${stockData.price.toFixed(2)}</p>
            <p>Change: {stockData.change > 0 ? '+' : ''}{stockData.change.toFixed(2)} ({stockData.changePercent.toFixed(2)}%)</p>
          </div>
          
          <StockChart data={stockData.chartData} />
          
          <ReportOptions 
            options={reportOptions}
            onChange={setReportOptions}
          />
          
          <button 
            className="generate-button"
            onClick={handleGenerateReport}
          >
            Generate Full Report
          </button>
        </>
      )}
    </div>
  )
}
