import React, { createContext, useContext, useState, useEffect } from 'react'

type Race = 'Human' | 'Dragonborn' | 'Elf' | 'Dwarf'

interface Resources {
  gold: number
  iron: number
  wood: number
  wheat: number
}

interface Workers {
  miners: number
  lumberjacks: number
  farmers: number
}

interface Soldiers {
  infantry: number
  archers: number
  cavalry: number
}

interface Upgrades {
  city: number
  barracks: number
  farms: number
  mines: number
  lumbermills: number
}

interface GameState {
  race: Race
  resources: Resources
  workers: Workers
  soldiers: Soldiers
  upgrades: Upgrades
  lastUpdated: Date
}

const initialState: GameState = {
  race: 'Human',
  resources: {
    gold: 1000,
    iron: 500,
    wood: 500,
    wheat: 500
  },
  workers: {
    miners: 10,
    lumberjacks: 10,
    farmers: 10
  },
  soldiers: {
    infantry: 0,
    archers: 0,
    cavalry: 0
  },
  upgrades: {
    city: 1,
    barracks: 1,
    farms: 1,
    mines: 1,
    lumbermills: 1
  },
  lastUpdated: new Date()
}

const GameStateContext = createContext<{
  state: GameState
  setState: React.Dispatch<React.SetStateAction<GameState>>
}>({
  state: initialState,
  setState: () => null
})

export const GameStateProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
  const [state, setState] = useState<GameState>(() => {
    const saved = localStorage.getItem('gameState')
    return saved ? JSON.parse(saved) : initialState
  })

  useEffect(() => {
    localStorage.setItem('gameState', JSON.stringify(state))
  }, [state])

  return (
    <GameStateContext.Provider value={{ state, setState }}>
      {children}
    </GameStateContext.Provider>
  )
}

export const useGameState = () => useContext(GameStateContext)
