const express = require('express')
const sqlite3 = require('sqlite3').verbose()
const cors = require('cors')
const { v4: uuidv4 } = require('uuid')
const bcrypt = require('bcryptjs')
const jwt = require('jsonwebtoken')
const { body, validationResult } = require('express-validator')
const multer = require('multer')
const path = require('path')

const app = express()
app.use(cors())
app.use(express.json())
app.use('/uploads', express.static('uploads'))

const db = new sqlite3.Database('./promoshare.db')
const JWT_SECRET = process.env.JWT_SECRET || 'your_jwt_secret_here'

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'uploads/')
  },
  filename: (req, file, cb) => {
    cb(null, `${Date.now()}-${file.originalname}`)
  }
})
const upload = multer({ storage })

// Authentication middleware
const authenticate = (req, res, next) => {
  const token = req.header('x-auth-token')
  if (!token) return res.status(401).json({ message: 'No token, authorization denied' })

  try {
    const decoded = jwt.verify(token, JWT_SECRET)
    req.user = decoded
    next()
  } catch (err) {
    res.status(401).json({ message: 'Token is not valid' })
  }
}

// Admin middleware
const isAdmin = (req, res, next) => {
  if (req.user.role !== 'admin') {
    return res.status(403).json({ message: 'Admin access required' })
  }
  next()
}

// User Routes
app.post('/api/auth/register', [
  body('username').notEmpty(),
  body('email').isEmail(),
  body('password').isLength({ min: 6 })
], async (req, res) => {
  const errors = validationResult(req)
  if (!errors.isEmpty()) {
    return res.status(400).json({ errors: errors.array() })
  }

  const { username, email, password } = req.body
  const hashedPassword = await bcrypt.hash(password, 10)
  const userId = uuidv4()

  db.run(
    'INSERT INTO users (id, username, email, password) VALUES (?, ?, ?, ?)',
    [userId, username, email, hashedPassword],
    function(err) {
      if (err) {
        return res.status(400).json({ message: 'User already exists' })
      }
      res.status(201).json({ message: 'User registered successfully' })
    }
  )
})

app.post('/api/auth/login', async (req, res) => {
  const { email, password } = req.body

  db.get('SELECT * FROM users WHERE email = ?', [email], async (err, user) => {
    if (err || !user) {
      return res.status(400).json({ message: 'Invalid credentials' })
    }

    const isMatch = await bcrypt.compare(password, user.password)
    if (!isMatch) {
      return res.status(400).json({ message: 'Invalid credentials' })
    }

    const payload = {
      user: {
        id: user.id,
        role: user.role
      }
    }

    jwt.sign(payload, JWT_SECRET, { expiresIn: '7d' }, (err, token) => {
      if (err) throw err
      res.json({ token, user: { id: user.id, username: user.username, role: user.role } })
    })
  })
})

// Promotion Routes
app.get('/api/promotions', (req, res) => {
  const { category, search } = req.query
  let query = 'SELECT * FROM promotions WHERE is_approved = 1'
  const params = []

  if (category) {
    query += ' AND category = ?'
    params.push(category)
  }

  if (search) {
    query += ' AND (title LIKE ? OR description LIKE ?)'
    params.push(`%${search}%`, `%${search}%`)
  }

  query += ' ORDER BY created_at DESC'

  db.all(query, params, (err, promotions) => {
    if (err) {
      return res.status(500).json({ message: 'Server error' })
    }
    res.json(promotions)
  })
})

app.post('/api/promotions', authenticate, upload.single('image'), [
  body('title').notEmpty(),
  body('description').notEmpty(),
  body('url').isURL(),
  body('category').notEmpty()
], (req, res) => {
  const errors = validationResult(req)
  if (!errors.isEmpty()) {
    return res.status(400).json({ errors: errors.array() })
  }

  const { title, description, discount_code, url, category } = req.body
  const promotionId = uuidv4()
  const imageUrl = req.file ? `/uploads/${req.file.filename}` : null

  db.run(
    `INSERT INTO promotions 
    (id, title, description, discount_code, url, image_url, category, user_id) 
    VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
    [promotionId, title, description, discount_code, url, imageUrl, category, req.user.id],
    function(err) {
      if (err) {
        return res.status(500).json({ message: 'Failed to create promotion' })
      }
      res.status(201).json({ message: 'Promotion submitted for approval' })
    }
  )
})

// Admin Routes
app.get('/api/admin/promotions', authenticate, isAdmin, (req, res) => {
  db.all(
    `SELECT p.*, u.username 
    FROM promotions p 
    JOIN users u ON p.user_id = u.id 
    WHERE p.is_approved = 0 
    ORDER BY p.created_at DESC`,
    (err, promotions) => {
      if (err) {
        return res.status(500).json({ message: 'Server error' })
      }
      res.json(promotions)
    }
  )
})

app.put('/api/admin/promotions/:id', authenticate, isAdmin, (req, res) => {
  const { action } = req.body
  const { id } = req.params

  if (!['approve', 'reject'].includes(action)) {
    return res.status(400).json({ message: 'Invalid action' })
  }

  if (action === 'approve') {
    db.run(
      'UPDATE promotions SET is_approved = 1 WHERE id = ?',
      [id],
      function(err) {
        if (err) {
          return res.status(500).json({ message: 'Failed to approve promotion' })
        }
        res.json({ message: 'Promotion approved' })
      }
    )
  } else {
    db.run(
      'DELETE FROM promotions WHERE id = ?',
      [id],
      function(err) {
        if (err) {
          return res.status(500).json({ message: 'Failed to reject promotion' })
        }
        res.json({ message: 'Promotion rejected' })
      }
    )
  }
})

// Error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack)
  res.status(500).json({ message: 'Something broke!' })
})

const PORT = process.env.PORT || 3001
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
