const express = require('express')
const cors = require('cors')
const sqlite3 = require('sqlite3').verbose()
const db = new sqlite3.Database('./fishing.db')

const app = express()
app.use(cors())
app.use(express.json())

// Create tables with better schema constraints
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS contests (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL CHECK(length(name) > 0),
      date TEXT NOT NULL CHECK(date IS date(date)),
      location TEXT NOT NULL CHECK(length(location) > 2),
      description TEXT,
      registration_open BOOLEAN DEFAULT true,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS participants (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      contest_id INTEGER NOT NULL,
      name TEXT NOT NULL CHECK(length(name) > 0),
      email TEXT NOT NULL CHECK(email LIKE '%@%.%'),
      phone TEXT CHECK(length(phone) >= 10),
      fish_type TEXT,
      registered_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(contest_id) REFERENCES contests(id) ON DELETE CASCADE
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS results (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      contest_id INTEGER NOT NULL,
      participant_id INTEGER NOT NULL,
      fish_length REAL CHECK(fish_length > 0),
      fish_weight REAL CHECK(fish_weight > 0),
      position INTEGER CHECK(position > 0),
      recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(contest_id) REFERENCES contests(id) ON DELETE CASCADE,
      FOREIGN KEY(participant_id) REFERENCES participants(id) ON DELETE CASCADE,
      UNIQUE(contest_id, participant_id)
    )
  `)

  db.run(`
    CREATE TABLE IF NOT EXISTS gallery (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      contest_id INTEGER NOT NULL,
      image_url TEXT NOT NULL CHECK(image_url LIKE 'http%'),
      caption TEXT,
      uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(contest_id) REFERENCES contests(id) ON DELETE CASCADE
    )
  `)
})

// Error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack)
  res.status(500).json({ error: 'Something went wrong!' })
})

// API Routes with improved validation
app.get('/api/contests', (req, res, next) => {
  db.all('SELECT * FROM contests ORDER BY date DESC', [], (err, rows) => {
    if (err) return next(err)
    res.json(rows)
  })
})

app.get('/api/contests/:id', (req, res, next) => {
  const { id } = req.params
  db.get('SELECT * FROM contests WHERE id = ?', [id], (err, row) => {
    if (err) return next(err)
    if (!row) return res.status(404).json({ error: 'Contest not found' })
    res.json(row)
  })
})

app.post('/api/contests', (req, res, next) => {
  const { name, date, location, description } = req.body
  if (!name || !date || !location) {
    return res.status(400).json({ error: 'Missing required fields' })
  }

  db.run(
    'INSERT INTO contests (name, date, location, description) VALUES (?, ?, ?, ?)',
    [name, date, location, description],
    function (err) {
      if (err) return next(err)
      res.status(201).json({ id: this.lastID })
    }
  )
})

app.post('/api/register', (req, res, next) => {
  const { contest_id, name, email, phone, fish_type } = req.body
  if (!contest_id || !name || !email) {
    return res.status(400).json({ error: 'Missing required fields' })
  }

  db.run(
    `INSERT INTO participants 
    (contest_id, name, email, phone, fish_type) 
    VALUES (?, ?, ?, ?, ?)`,
    [contest_id, name, email, phone, fish_type],
    function (err) {
      if (err) return next(err)
      res.status(201).json({ id: this.lastID })
    }
  )
})

app.post('/api/results', (req, res, next) => {
  const { contest_id, participant_id, fish_length, fish_weight, position } = req.body
  if (!contest_id || !participant_id || !fish_length || !fish_weight || !position) {
    return res.status(400).json({ error: 'Missing required fields' })
  }

  db.run(
    `INSERT INTO results 
    (contest_id, participant_id, fish_length, fish_weight, position) 
    VALUES (?, ?, ?, ?, ?)`,
    [contest_id, participant_id, fish_length, fish_weight, position],
    function (err) {
      if (err) return next(err)
      res.status(201).json({ id: this.lastID })
    }
  )
})

app.get('/api/results/:contest_id', (req, res, next) => {
  const contest_id = req.params.contest_id
  db.all(
    `SELECT p.name, r.fish_length, r.fish_weight, r.position 
     FROM results r
     JOIN participants p ON r.participant_id = p.id
     WHERE r.contest_id = ? 
     ORDER BY r.position`,
    [contest_id],
    (err, rows) => {
      if (err) return next(err)
      res.json(rows)
    }
  )
})

app.post('/api/gallery', (req, res, next) => {
  const { contest_id, image_url, caption } = req.body
  if (!contest_id || !image_url) {
    return res.status(400).json({ error: 'Missing required fields' })
  }

  db.run(
    'INSERT INTO gallery (contest_id, image_url, caption) VALUES (?, ?, ?)',
    [contest_id, image_url, caption],
    function (err) {
      if (err) return next(err)
      res.status(201).json({ id: this.lastID })
    }
  )
})

app.get('/api/gallery/:contest_id', (req, res, next) => {
  const contest_id = req.params.contest_id
  db.all(
    'SELECT * FROM gallery WHERE contest_id = ? ORDER BY uploaded_at DESC',
    [contest_id],
    (err, rows) => {
      if (err) return next(err)
      res.json(rows)
    }
  )
})

const PORT = process.env.PORT || 3001
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})

process.on('SIGINT', () => {
  db.close()
  process.exit()
})
